(function() {
  var BufferOffsetIndex, CompositeChangeEvent, CompositeDisposable, DisplayLayer, Disposable, Emitter, File, Grim, History, MarkerLayer, MatchIterator, NativeTextBuffer, Patch, Point, Range, TextBuffer, TransactionAbortedError, _, crypto, debounce, diff, extentForText, fs, mkdirp, newlineRegex, normalizePatchChanges, path, ref, ref1, ref2, regexIsSingleLine, spliceArray, traversal,
    extend = function(child, parent) { for (var key in parent) { if (hasProp.call(parent, key)) child[key] = parent[key]; } function ctor() { this.constructor = child; } ctor.prototype = parent.prototype; child.prototype = new ctor(); child.__super__ = parent.prototype; return child; },
    hasProp = {}.hasOwnProperty;

  ref = require('event-kit'), Emitter = ref.Emitter, CompositeDisposable = ref.CompositeDisposable, Disposable = ref.Disposable;

  File = require('pathwatcher').File;

  diff = require('diff');

  _ = require('underscore-plus');

  fs = require('fs-plus');

  path = require('path');

  crypto = require('crypto');

  mkdirp = require('mkdirp');

  ref1 = require('superstring'), BufferOffsetIndex = ref1.BufferOffsetIndex, Patch = ref1.Patch, NativeTextBuffer = ref1.TextBuffer;

  Point = require('./point');

  Range = require('./range');

  History = require('./history');

  MarkerLayer = require('./marker-layer');

  MatchIterator = require('./match-iterator');

  DisplayLayer = require('./display-layer');

  ref2 = require('./helpers'), spliceArray = ref2.spliceArray, newlineRegex = ref2.newlineRegex, normalizePatchChanges = ref2.normalizePatchChanges, regexIsSingleLine = ref2.regexIsSingleLine, extentForText = ref2.extentForText, debounce = ref2.debounce;

  traversal = require('./point-helpers').traversal;

  Grim = require('grim');

  TransactionAbortedError = (function(superClass) {
    extend(TransactionAbortedError, superClass);

    function TransactionAbortedError() {
      TransactionAbortedError.__super__.constructor.apply(this, arguments);
    }

    return TransactionAbortedError;

  })(Error);

  CompositeChangeEvent = (function() {
    function CompositeChangeEvent(buffer, patch) {
      var compositeStart, newEnd, newText, oldEnd, oldText, ref3;
      ref3 = patch.getBounds(), compositeStart = ref3.oldStart, oldEnd = ref3.oldEnd, newEnd = ref3.newEnd;
      this.oldRange = new Range(compositeStart, oldEnd);
      this.newRange = new Range(compositeStart, newEnd);
      oldText = null;
      newText = null;
      Object.defineProperty(this, 'didChange', {
        enumerable: false,
        writable: true,
        value: false
      });
      Object.defineProperty(this, 'oldText', {
        enumerable: true,
        get: function() {
          var change, i, oldBuffer, ref4;
          if (oldText == null) {
            if (this.didChange) {
              oldBuffer = new NativeTextBuffer(this.newText);
              ref4 = patch.getChanges();
              for (i = ref4.length - 1; i >= 0; i += -1) {
                change = ref4[i];
                oldBuffer.setTextInRange(new Range(traversal(change.newStart, compositeStart), traversal(change.newEnd, compositeStart)), change.oldText);
              }
              oldText = oldBuffer.getText();
            } else {
              oldText = buffer.getTextInRange(this.oldRange);
            }
          }
          return oldText;
        }
      });
      Object.defineProperty(this, 'newText', {
        enumerable: true,
        get: function() {
          var change, i, newBuffer, ref4;
          if (newText == null) {
            if (this.didChange) {
              newText = buffer.getTextInRange(this.newRange);
            } else {
              newBuffer = new NativeTextBuffer(this.oldText);
              ref4 = patch.getChanges();
              for (i = ref4.length - 1; i >= 0; i += -1) {
                change = ref4[i];
                newBuffer.setTextInRange(new Range(traversal(change.oldStart, compositeStart), traversal(change.oldEnd, compositeStart)), change.newText);
              }
              newText = newBuffer.getText();
            }
          }
          return newText;
        }
      });
    }

    return CompositeChangeEvent;

  })();

  module.exports = TextBuffer = (function() {
    TextBuffer.version = 5;

    TextBuffer.Point = Point;

    TextBuffer.Range = Range;

    TextBuffer.newlineRegex = newlineRegex;

    TextBuffer.prototype.encoding = null;

    TextBuffer.prototype.stoppedChangingDelay = 300;

    TextBuffer.prototype.fileChangeDelay = 200;

    TextBuffer.prototype.stoppedChangingTimeout = null;

    TextBuffer.prototype.conflict = false;

    TextBuffer.prototype.file = null;

    TextBuffer.prototype.refcount = 0;

    TextBuffer.prototype.fileSubscriptions = null;

    TextBuffer.prototype.backwardsScanChunkSize = 8000;

    TextBuffer.prototype.defaultMaxUndoEntries = 10000;

    TextBuffer.prototype.nextMarkerLayerId = 0;


    /*
    Section: Construction
     */

    function TextBuffer(params) {
      var maxUndoEntries, ref3, ref4, text;
      text = typeof params === 'string' ? params : params != null ? params.text : void 0;
      this.emitter = new Emitter;
      this.patchesSinceLastStoppedChangingEvent = [];
      this.id = crypto.randomBytes(16).toString('hex');
      this.buffer = new NativeTextBuffer(text);
      this.debouncedEmitDidStopChangingEvent = debounce(this.emitDidStopChangingEvent.bind(this), this.stoppedChangingDelay);
      this.textDecorationLayers = new Set();
      maxUndoEntries = (ref3 = params != null ? params.maxUndoEntries : void 0) != null ? ref3 : this.defaultMaxUndoEntries;
      this.history = new History(this, maxUndoEntries);
      this.nextMarkerLayerId = 0;
      this.nextDisplayLayerId = 0;
      this.defaultMarkerLayer = new MarkerLayer(this, String(this.nextMarkerLayerId++));
      this.displayLayers = {};
      this.markerLayers = {};
      this.markerLayers[this.defaultMarkerLayer.id] = this.defaultMarkerLayer;
      this.markerLayersWithPendingUpdateEvents = new Set();
      this.nextMarkerId = 1;
      this.outstandingSaveCount = 0;
      this.loadCount = 0;
      this.setEncoding(params != null ? params.encoding : void 0);
      this.setPreferredLineEnding(params != null ? params.preferredLineEnding : void 0);
      this.loaded = false;
      this.destroyed = false;
      this.transactCallDepth = 0;
      this.digestWhenLastPersisted = false;
      this.shouldDestroyOnFileDelete = (ref4 = params != null ? params.shouldDestroyOnFileDelete : void 0) != null ? ref4 : function() {
        return false;
      };
      if (params != null ? params.filePath : void 0) {
        this.setPath(params.filePath);
        if (params != null ? params.load : void 0) {
          Grim.deprecate('The `load` option to the TextBuffer constructor is deprecated. ' + 'Get a loaded buffer using TextBuffer.load(filePath) instead.');
          this.load({
            internal: true
          });
        }
      }
    }

    TextBuffer.prototype.toString = function() {
      return "<TextBuffer " + this.id + ">";
    };

    TextBuffer.load = function(source, params) {
      var buffer;
      buffer = new TextBuffer(params);
      if (typeof source === 'string') {
        buffer.setPath(source);
      } else {
        buffer.setFile(source);
      }
      return buffer.load({
        clearHistory: true,
        internal: true
      }).then(function() {
        return buffer;
      });
    };

    TextBuffer.loadSync = function(filePath, params) {
      var buffer;
      buffer = new TextBuffer(params);
      buffer.setPath(filePath);
      buffer.loadSync({
        internal: true
      });
      return buffer;
    };

    TextBuffer.deserialize = function(params) {
      var promise;
      if (params.version !== TextBuffer.prototype.version) {
        return;
      }
      delete params.load;
      if (params.filePath != null) {
        promise = this.load(params.filePath, params).then(function(buffer) {
          if (typeof params.text === 'string') {
            buffer.setText(params.text);
          } else if (buffer.digestWhenLastPersisted === params.digestWhenLastPersisted) {
            buffer.buffer.deserializeChanges(params.outstandingChanges);
          } else {
            params.history = {};
          }
          return buffer;
        });
      } else {
        promise = Promise.resolve(new TextBuffer(params));
      }
      return promise.then(function(buffer) {
        var layer, layerId, layerState, ref3, ref4;
        buffer.id = params.id;
        buffer.preferredLineEnding = params.preferredLineEnding;
        buffer.nextMarkerId = params.nextMarkerId;
        buffer.nextMarkerLayerId = params.nextMarkerLayerId;
        buffer.nextDisplayLayerId = params.nextDisplayLayerId;
        buffer.history.deserialize(params.history, buffer);
        ref3 = params.markerLayers;
        for (layerId in ref3) {
          layerState = ref3[layerId];
          if (layerId === params.defaultMarkerLayerId) {
            buffer.defaultMarkerLayer.id = layerId;
            buffer.defaultMarkerLayer.deserialize(layerState);
            layer = buffer.defaultMarkerLayer;
          } else {
            layer = MarkerLayer.deserialize(buffer, layerState);
          }
          buffer.markerLayers[layerId] = layer;
        }
        ref4 = params.displayLayers;
        for (layerId in ref4) {
          layerState = ref4[layerId];
          buffer.displayLayers[layerId] = DisplayLayer.deserialize(buffer, layerState);
        }
        return buffer;
      });
    };

    TextBuffer.prototype.getId = function() {
      return this.id;
    };

    TextBuffer.prototype.serialize = function(options) {
      var displayLayers, filePath, history, id, layer, markerLayers, ref3, ref4, result;
      if (options == null) {
        options = {};
      }
      if (options.markerLayers == null) {
        options.markerLayers = true;
      }
      if (options.history == null) {
        options.history = true;
      }
      markerLayers = {};
      if (options.markerLayers) {
        ref3 = this.markerLayers;
        for (id in ref3) {
          layer = ref3[id];
          if (layer.persistent) {
            markerLayers[id] = layer.serialize();
          }
        }
      }
      displayLayers = {};
      ref4 = this.displayLayers;
      for (id in ref4) {
        layer = ref4[id];
        displayLayers[id] = layer.serialize();
      }
      history = {};
      if (options.history) {
        history = this.history.serialize(options);
      }
      result = {
        id: this.getId(),
        defaultMarkerLayerId: this.defaultMarkerLayer.id,
        markerLayers: markerLayers,
        displayLayers: displayLayers,
        nextMarkerLayerId: this.nextMarkerLayerId,
        nextDisplayLayerId: this.nextDisplayLayerId,
        history: history,
        encoding: this.getEncoding(),
        preferredLineEnding: this.preferredLineEnding,
        nextMarkerId: this.nextMarkerId
      };
      if (filePath = this.getPath()) {
        if (this.baseTextDigestCache == null) {
          this.baseTextDigestCache = this.buffer.baseTextDigest();
        }
        result.filePath = filePath;
        result.digestWhenLastPersisted = this.digestWhenLastPersisted;
        result.outstandingChanges = this.buffer.serializeChanges();
      } else {
        result.text = this.getText();
      }
      return result;
    };


    /*
    Section: Event Subscription
     */

    TextBuffer.prototype.onWillChange = function(callback) {
      return this.emitter.on('will-change', callback);
    };

    TextBuffer.prototype.onDidChange = function(callback) {
      return this.emitter.on('did-change', callback);
    };

    TextBuffer.prototype.onDidChangeText = function(callback) {
      return this.emitter.on('did-change-text', callback);
    };

    TextBuffer.prototype.preemptDidChange = function(callback) {
      return this.emitter.preempt('did-change', callback);
    };

    TextBuffer.prototype.onDidStopChanging = function(callback) {
      return this.emitter.on('did-stop-changing', callback);
    };

    TextBuffer.prototype.onDidConflict = function(callback) {
      return this.emitter.on('did-conflict', callback);
    };

    TextBuffer.prototype.onDidChangeModified = function(callback) {
      return this.emitter.on('did-change-modified', callback);
    };

    TextBuffer.prototype.onDidUpdateMarkers = function(callback) {
      return this.emitter.on('did-update-markers', callback);
    };

    TextBuffer.prototype.onDidCreateMarker = function(callback) {
      return this.emitter.on('did-create-marker', callback);
    };

    TextBuffer.prototype.onDidChangePath = function(callback) {
      return this.emitter.on('did-change-path', callback);
    };

    TextBuffer.prototype.onDidChangeEncoding = function(callback) {
      return this.emitter.on('did-change-encoding', callback);
    };

    TextBuffer.prototype.onWillSave = function(callback) {
      return this.emitter.on('will-save', callback);
    };

    TextBuffer.prototype.onDidSave = function(callback) {
      return this.emitter.on('did-save', callback);
    };

    TextBuffer.prototype.onDidDelete = function(callback) {
      return this.emitter.on('did-delete', callback);
    };

    TextBuffer.prototype.onWillReload = function(callback) {
      return this.emitter.on('will-reload', callback);
    };

    TextBuffer.prototype.onDidReload = function(callback) {
      return this.emitter.on('did-reload', callback);
    };

    TextBuffer.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    TextBuffer.prototype.onWillThrowWatchError = function(callback) {
      return this.emitter.on('will-throw-watch-error', callback);
    };

    TextBuffer.prototype.getStoppedChangingDelay = function() {
      return this.stoppedChangingDelay;
    };


    /*
    Section: File Details
     */

    TextBuffer.prototype.isModified = function() {
      var ref3;
      if ((ref3 = this.file) != null ? ref3.existsSync() : void 0) {
        return this.buffer.isModified();
      } else {
        return this.buffer.getLength() > 0;
      }
    };

    TextBuffer.prototype.isInConflict = function() {
      return this.isModified() && this.fileHasChangedSinceLastLoad;
    };

    TextBuffer.prototype.getPath = function() {
      var ref3;
      return (ref3 = this.file) != null ? ref3.getPath() : void 0;
    };

    TextBuffer.prototype.setPath = function(filePath) {
      if (filePath === this.getPath()) {
        return;
      }
      return this.setFile(filePath ? new File(filePath) : void 0);
    };

    TextBuffer.prototype.setFile = function(file) {
      var base;
      if ((file != null ? file.getPath() : void 0) === this.getPath()) {
        return;
      }
      this.file = file;
      if (this.file != null) {
        if (typeof (base = this.file).setEncoding === "function") {
          base.setEncoding(this.getEncoding());
        }
        this.subscribeToFile();
      }
      return this.emitter.emit('did-change-path', this.getPath());
    };

    TextBuffer.prototype.setEncoding = function(encoding) {
      var base;
      if (encoding == null) {
        encoding = 'utf8';
      }
      if (encoding === this.getEncoding()) {
        return;
      }
      this.encoding = encoding;
      if (this.file != null) {
        if (typeof (base = this.file).setEncoding === "function") {
          base.setEncoding(encoding);
        }
        this.emitter.emit('did-change-encoding', encoding);
        if (!this.isModified()) {
          this.load({
            clearHistory: true,
            internal: true
          });
        }
      } else {
        this.emitter.emit('did-change-encoding', encoding);
      }
    };

    TextBuffer.prototype.getEncoding = function() {
      var ref3, ref4;
      return (ref3 = this.encoding) != null ? ref3 : (ref4 = this.file) != null ? ref4.getEncoding() : void 0;
    };

    TextBuffer.prototype.setPreferredLineEnding = function(preferredLineEnding) {
      if (preferredLineEnding == null) {
        preferredLineEnding = null;
      }
      return this.preferredLineEnding = preferredLineEnding;
    };

    TextBuffer.prototype.getPreferredLineEnding = function() {
      return this.preferredLineEnding;
    };

    TextBuffer.prototype.getUri = function() {
      return this.getPath();
    };

    TextBuffer.prototype.getBaseName = function() {
      var ref3;
      return (ref3 = this.file) != null ? ref3.getBaseName() : void 0;
    };


    /*
    Section: Reading Text
     */

    TextBuffer.prototype.isEmpty = function() {
      return this.buffer.getLength() === 0;
    };

    TextBuffer.prototype.getText = function() {
      return this.cachedText != null ? this.cachedText : this.cachedText = this.buffer.getText();
    };

    TextBuffer.prototype.getTextInRange = function(range) {
      return this.buffer.getTextInRange(Range.fromObject(range));
    };

    TextBuffer.prototype.getLines = function() {
      return this.buffer.getLines();
    };

    TextBuffer.prototype.getLastLine = function() {
      return this.lineForRow(this.getLastRow());
    };

    TextBuffer.prototype.lineForRow = function(row) {
      return this.buffer.lineForRow(row);
    };

    TextBuffer.prototype.lineEndingForRow = function(row) {
      return this.buffer.lineEndingForRow(row);
    };

    TextBuffer.prototype.lineLengthForRow = function(row) {
      return this.buffer.lineLengthForRow(row);
    };

    TextBuffer.prototype.isRowBlank = function(row) {
      return !/\S/.test(this.lineForRow(row));
    };

    TextBuffer.prototype.previousNonBlankRow = function(startRow) {
      var i, ref3, row;
      if (startRow === 0) {
        return null;
      }
      startRow = Math.min(startRow, this.getLastRow());
      for (row = i = ref3 = startRow - 1; ref3 <= 0 ? i <= 0 : i >= 0; row = ref3 <= 0 ? ++i : --i) {
        if (!this.isRowBlank(row)) {
          return row;
        }
      }
      return null;
    };

    TextBuffer.prototype.nextNonBlankRow = function(startRow) {
      var i, lastRow, ref3, ref4, row;
      lastRow = this.getLastRow();
      if (startRow < lastRow) {
        for (row = i = ref3 = startRow + 1, ref4 = lastRow; ref3 <= ref4 ? i <= ref4 : i >= ref4; row = ref3 <= ref4 ? ++i : --i) {
          if (!this.isRowBlank(row)) {
            return row;
          }
        }
      }
      return null;
    };


    /*
    Section: Mutating Text
     */

    TextBuffer.prototype.setText = function(text) {
      return this.setTextInRange(this.getRange(), text, {
        normalizeLineEndings: false
      });
    };

    TextBuffer.prototype.setTextViaDiff = function(text) {
      var computeBufferColumn, currentText, endsWithNewline;
      currentText = this.getText();
      if (currentText === text) {
        return;
      }
      endsWithNewline = function(str) {
        return /[\r\n]+$/g.test(str);
      };
      computeBufferColumn = function(str) {
        var newlineIndex;
        newlineIndex = Math.max(str.lastIndexOf('\n'), str.lastIndexOf('\r'));
        if (endsWithNewline(str)) {
          return 0;
        } else if (newlineIndex === -1) {
          return str.length;
        } else {
          return str.length - newlineIndex - 1;
        }
      };
      return this.transact((function(_this) {
        return function() {
          var change, changeOptions, column, currentPosition, endColumn, endRow, i, len, lineCount, lineDiff, ref3, ref4, row;
          row = 0;
          column = 0;
          currentPosition = [0, 0];
          lineDiff = diff.diffLines(currentText, text);
          changeOptions = {
            normalizeLineEndings: false
          };
          for (i = 0, len = lineDiff.length; i < len; i++) {
            change = lineDiff[i];
            lineCount = (ref3 = (ref4 = change.value.match(newlineRegex)) != null ? ref4.length : void 0) != null ? ref3 : 0;
            currentPosition[0] = row;
            currentPosition[1] = column;
            if (change.added) {
              row += lineCount;
              column = computeBufferColumn(change.value);
              _this.setTextInRange([currentPosition, currentPosition], change.value, changeOptions);
            } else if (change.removed) {
              endRow = row + lineCount;
              endColumn = column + computeBufferColumn(change.value);
              _this.setTextInRange([currentPosition, [endRow, endColumn]], '', changeOptions);
            } else {
              row += lineCount;
              column = computeBufferColumn(change.value);
            }
          }
        };
      })(this));
    };

    TextBuffer.prototype.setTextInRange = function(range, newText, options) {
      var change, normalizeLineEndings, oldRange, oldText, undo;
      if (this.transactCallDepth === 0) {
        return this.transact((function(_this) {
          return function() {
            return _this.setTextInRange(range, newText, options);
          };
        })(this));
      }
      if (options != null) {
        normalizeLineEndings = options.normalizeLineEndings, undo = options.undo;
      }
      if (normalizeLineEndings == null) {
        normalizeLineEndings = true;
      }
      oldRange = this.clipRange(range);
      oldText = this.getTextInRange(oldRange);
      change = {
        oldStart: oldRange.start,
        newStart: oldRange.start,
        oldEnd: oldRange.end,
        oldText: oldText,
        newText: newText,
        normalizeLineEndings: normalizeLineEndings
      };
      return this.applyChange(change, undo !== 'skip');
    };

    TextBuffer.prototype.insert = function(position, text, options) {
      return this.setTextInRange(new Range(position, position), text, options);
    };

    TextBuffer.prototype.append = function(text, options) {
      return this.insert(this.getEndPosition(), text, options);
    };

    TextBuffer.prototype.applyChange = function(change, pushToHistory) {
      var changeEvent, displayLayer, id, markerLayer, newExtent, newRange, newStart, newText, normalizeLineEndings, normalizedEnding, oldEnd, oldExtent, oldRange, oldStart, oldText, ref3, ref4, ref5, start, startRow;
      if (pushToHistory == null) {
        pushToHistory = false;
      }
      newStart = change.newStart, oldStart = change.oldStart, oldEnd = change.oldEnd, oldText = change.oldText, newText = change.newText, normalizeLineEndings = change.normalizeLineEndings;
      oldExtent = traversal(oldEnd, oldStart);
      start = Point.fromObject(newStart);
      oldRange = Range(start, start.traverse(oldExtent));
      oldRange.freeze();
      if (normalizeLineEndings) {
        startRow = oldRange.start.row;
        normalizedEnding = this.preferredLineEnding || this.lineEndingForRow(startRow) || this.lineEndingForRow(startRow - 1);
        if (normalizedEnding) {
          newText = newText.replace(newlineRegex, normalizedEnding);
        }
      }
      newExtent = extentForText(newText);
      newRange = Range(start, start.traverse(newExtent));
      newRange.freeze();
      if (pushToHistory) {
        if (change.oldExtent == null) {
          change.oldExtent = oldExtent;
        }
        if (change.newExtent == null) {
          change.newExtent = newExtent;
        }
        if ((ref3 = this.history) != null) {
          ref3.pushChange(change);
        }
      }
      changeEvent = {
        oldRange: oldRange,
        newRange: newRange,
        oldText: oldText,
        newText: newText
      };
      ref4 = this.displayLayers;
      for (id in ref4) {
        displayLayer = ref4[id];
        displayLayer.bufferWillChange(changeEvent);
      }
      this.emitter.emit('will-change', {
        oldRange: oldRange
      });
      this.buffer.setTextInRange(oldRange, newText);
      if (this.markerLayers != null) {
        ref5 = this.markerLayers;
        for (id in ref5) {
          markerLayer = ref5[id];
          markerLayer.splice(oldRange.start, oldExtent, newExtent);
          this.markerLayersWithPendingUpdateEvents.add(markerLayer);
        }
      }
      this.cachedText = null;
      this.emitDidChangeEvent(changeEvent);
      return newRange;
    };

    TextBuffer.prototype.emitDidChangeEvent = function(changeEvent) {
      var changeEventsByDisplayLayer, displayLayer, event, id, ref3;
      this.textDecorationLayers.forEach(function(textDecorationLayer) {
        return textDecorationLayer.bufferDidChange(changeEvent);
      });
      changeEventsByDisplayLayer = new Map();
      ref3 = this.displayLayers;
      for (id in ref3) {
        displayLayer = ref3[id];
        event = displayLayer.bufferDidChange(changeEvent);
        changeEventsByDisplayLayer.set(displayLayer, event);
      }
      this.emitter.emit('did-change', changeEvent);
      return changeEventsByDisplayLayer.forEach(function(event, displayLayer) {
        return displayLayer.emitDidChangeSyncEvent(event);
      });
    };

    TextBuffer.prototype["delete"] = function(range) {
      return this.setTextInRange(range, '');
    };

    TextBuffer.prototype.deleteRow = function(row) {
      return this.deleteRows(row, row);
    };

    TextBuffer.prototype.deleteRows = function(startRow, endRow) {
      var endPoint, lastRow, ref3, startPoint;
      lastRow = this.getLastRow();
      if (startRow > endRow) {
        ref3 = [endRow, startRow], startRow = ref3[0], endRow = ref3[1];
      }
      if (endRow < 0) {
        return new Range(this.getFirstPosition(), this.getFirstPosition());
      }
      if (startRow > lastRow) {
        return new Range(this.getEndPosition(), this.getEndPosition());
      }
      startRow = Math.max(0, startRow);
      endRow = Math.min(lastRow, endRow);
      if (endRow < lastRow) {
        startPoint = new Point(startRow, 0);
        endPoint = new Point(endRow + 1, 0);
      } else {
        if (startRow === 0) {
          startPoint = new Point(startRow, 0);
        } else {
          startPoint = new Point(startRow - 1, this.lineLengthForRow(startRow - 1));
        }
        endPoint = new Point(endRow, this.lineLengthForRow(endRow));
      }
      return this["delete"](new Range(startPoint, endPoint));
    };


    /*
    Section: Markers
     */

    TextBuffer.prototype.addMarkerLayer = function(options) {
      var layer;
      layer = new MarkerLayer(this, String(this.nextMarkerLayerId++), options);
      this.markerLayers[layer.id] = layer;
      return layer;
    };

    TextBuffer.prototype.getMarkerLayer = function(id) {
      return this.markerLayers[id];
    };

    TextBuffer.prototype.getDefaultMarkerLayer = function() {
      return this.defaultMarkerLayer;
    };

    TextBuffer.prototype.markRange = function(range, properties) {
      return this.defaultMarkerLayer.markRange(range, properties);
    };

    TextBuffer.prototype.markPosition = function(position, options) {
      return this.defaultMarkerLayer.markPosition(position, options);
    };

    TextBuffer.prototype.getMarkers = function() {
      return this.defaultMarkerLayer.getMarkers();
    };

    TextBuffer.prototype.getMarker = function(id) {
      return this.defaultMarkerLayer.getMarker(id);
    };

    TextBuffer.prototype.findMarkers = function(params) {
      return this.defaultMarkerLayer.findMarkers(params);
    };

    TextBuffer.prototype.getMarkerCount = function() {
      return this.defaultMarkerLayer.getMarkerCount();
    };

    TextBuffer.prototype.destroyMarker = function(id) {
      var ref3;
      return (ref3 = this.getMarker(id)) != null ? ref3.destroy() : void 0;
    };


    /*
    Section: History
     */

    TextBuffer.prototype.undo = function() {
      var change, i, len, pop, ref3;
      if (pop = this.history.popUndoStack()) {
        ref3 = pop.patch.getChanges();
        for (i = 0, len = ref3.length; i < len; i++) {
          change = ref3[i];
          this.applyChange(change);
        }
        this.restoreFromMarkerSnapshot(pop.snapshot);
        this.emitDidChangeTextEvent(pop.patch);
        this.emitMarkerChangeEvents(pop.snapshot);
        return true;
      } else {
        return false;
      }
    };

    TextBuffer.prototype.redo = function() {
      var change, i, len, pop, ref3;
      if (pop = this.history.popRedoStack()) {
        ref3 = pop.patch.getChanges();
        for (i = 0, len = ref3.length; i < len; i++) {
          change = ref3[i];
          this.applyChange(change);
        }
        this.restoreFromMarkerSnapshot(pop.snapshot);
        this.emitDidChangeTextEvent(pop.patch);
        this.emitMarkerChangeEvents(pop.snapshot);
        return true;
      } else {
        return false;
      }
    };

    TextBuffer.prototype.transact = function(groupingInterval, fn) {
      var checkpointBefore, compactedChanges, endMarkerSnapshot, exception, ref3, result;
      if (typeof groupingInterval === 'function') {
        fn = groupingInterval;
        groupingInterval = 0;
      }
      checkpointBefore = this.history.createCheckpoint(this.createMarkerSnapshot(), true);
      try {
        this.transactCallDepth++;
        result = fn();
      } catch (error1) {
        exception = error1;
        this.revertToCheckpoint(checkpointBefore, true);
        if (!(exception instanceof TransactionAbortedError)) {
          throw exception;
        }
        return;
      } finally {
        this.transactCallDepth--;
      }
      endMarkerSnapshot = this.createMarkerSnapshot();
      compactedChanges = this.history.groupChangesSinceCheckpoint(checkpointBefore, endMarkerSnapshot, true);
      if ((ref3 = global.atom) != null) {
        ref3.assert(compactedChanges, "groupChangesSinceCheckpoint() returned false.", (function(_this) {
          return function(error) {
            return error.metadata = {
              history: _this.history.toString()
            };
          };
        })(this));
      }
      this.history.applyGroupingInterval(groupingInterval);
      this.history.enforceUndoStackSizeLimit();
      if (compactedChanges) {
        this.emitDidChangeTextEvent(compactedChanges);
      }
      this.emitMarkerChangeEvents(endMarkerSnapshot);
      return result;
    };

    TextBuffer.prototype.abortTransaction = function() {
      throw new TransactionAbortedError("Transaction aborted.");
    };

    TextBuffer.prototype.clearUndoStack = function() {
      return this.history.clearUndoStack();
    };

    TextBuffer.prototype.createCheckpoint = function() {
      return this.history.createCheckpoint(this.createMarkerSnapshot(), false);
    };

    TextBuffer.prototype.revertToCheckpoint = function(checkpoint) {
      var change, i, len, ref3, truncated;
      if (truncated = this.history.truncateUndoStack(checkpoint)) {
        ref3 = truncated.patch.getChanges();
        for (i = 0, len = ref3.length; i < len; i++) {
          change = ref3[i];
          this.applyChange(change);
        }
        this.restoreFromMarkerSnapshot(truncated.snapshot);
        this.emitDidChangeTextEvent(truncated.patch);
        this.emitter.emit('did-update-markers');
        this.emitMarkerChangeEvents(truncated.snapshot);
        return true;
      } else {
        return false;
      }
    };

    TextBuffer.prototype.groupChangesSinceCheckpoint = function(checkpoint) {
      return this.history.groupChangesSinceCheckpoint(checkpoint, this.createMarkerSnapshot(), false);
    };

    TextBuffer.prototype.getChangesSinceCheckpoint = function(checkpoint) {
      var patch;
      if (patch = this.history.getChangesSinceCheckpoint(checkpoint)) {
        return normalizePatchChanges(patch.getChanges());
      } else {
        return [];
      }
    };


    /*
    Section: Search And Replace
     */

    TextBuffer.prototype.scan = function(regex, options, iterator) {
      if (options == null) {
        options = {};
      }
      if (_.isFunction(options)) {
        iterator = options;
        options = {};
      }
      return this.scanInRange(regex, this.getRange(), options, iterator);
    };

    TextBuffer.prototype.backwardsScan = function(regex, options, iterator) {
      if (options == null) {
        options = {};
      }
      if (_.isFunction(options)) {
        iterator = options;
        options = {};
      }
      return this.backwardsScanInRange(regex, this.getRange(), options, iterator);
    };

    TextBuffer.prototype.scanInRange = function(regex, range, options, callback, reverse) {
      var flags, global, iterator;
      if (options == null) {
        options = {};
      }
      if (reverse == null) {
        reverse = false;
      }
      if (_.isFunction(options)) {
        reverse = callback;
        callback = options;
        options = {};
      }
      range = this.clipRange(range);
      global = regex.global;
      flags = "gm";
      if (regex.ignoreCase) {
        flags += "i";
      }
      regex = new RegExp(regex.source, flags);
      if (regexIsSingleLine(regex)) {
        if (reverse) {
          iterator = new MatchIterator.BackwardsSingleLine(this, regex, range, options);
        } else {
          iterator = new MatchIterator.ForwardsSingleLine(this, regex, range, options);
        }
      } else {
        if (reverse) {
          iterator = new MatchIterator.BackwardsMultiLine(this, regex, range, this.backwardsScanChunkSize, options);
        } else {
          iterator = new MatchIterator.ForwardsMultiLine(this, regex, range, options);
        }
      }
      return iterator.iterate(callback, global);
    };

    TextBuffer.prototype.backwardsScanInRange = function(regex, range, options, iterator) {
      if (options == null) {
        options = {};
      }
      if (_.isFunction(options)) {
        iterator = options;
        options = {};
      }
      return this.scanInRange(regex, range, options, iterator, true);
    };

    TextBuffer.prototype.replace = function(regex, replacementText) {
      var doSave, replacements;
      doSave = !this.isModified();
      replacements = 0;
      this.transact((function(_this) {
        return function() {
          return _this.scan(regex, function(arg) {
            var matchText, replace;
            matchText = arg.matchText, replace = arg.replace;
            replace(matchText.replace(regex, replacementText));
            return replacements++;
          });
        };
      })(this));
      if (doSave) {
        this.save();
      }
      return replacements;
    };

    TextBuffer.prototype.find = function(regex) {
      return this.buffer.find(regex);
    };

    TextBuffer.prototype.findSync = function(regex) {
      return this.buffer.findSync(regex);
    };

    TextBuffer.prototype.findAllSync = function(regex) {
      return this.buffer.findAllSync(regex);
    };


    /*
    Section: Buffer Range Details
     */

    TextBuffer.prototype.getRange = function() {
      return new Range(this.getFirstPosition(), this.getEndPosition());
    };

    TextBuffer.prototype.getLineCount = function() {
      return this.buffer.getLineCount();
    };

    TextBuffer.prototype.getLastRow = function() {
      return this.getLineCount() - 1;
    };

    TextBuffer.prototype.getFirstPosition = function() {
      return new Point(0, 0);
    };

    TextBuffer.prototype.getEndPosition = function() {
      return Point.fromObject(this.buffer.getExtent());
    };

    TextBuffer.prototype.getMaxCharacterIndex = function() {
      return this.characterIndexForPosition(Point.INFINITY);
    };

    TextBuffer.prototype.rangeForRow = function(row, includeNewline) {
      row = Math.max(row, 0);
      row = Math.min(row, this.getLastRow());
      if (includeNewline && row < this.getLastRow()) {
        return new Range(new Point(row, 0), new Point(row + 1, 0));
      } else {
        return new Range(new Point(row, 0), new Point(row, this.lineLengthForRow(row)));
      }
    };

    TextBuffer.prototype.characterIndexForPosition = function(position) {
      return this.buffer.characterIndexForPosition(Point.fromObject(position));
    };

    TextBuffer.prototype.positionForCharacterIndex = function(offset) {
      return Point.fromObject(this.buffer.positionForCharacterIndex(offset));
    };

    TextBuffer.prototype.clipRange = function(range) {
      var end, start;
      range = Range.fromObject(range);
      start = this.clipPosition(range.start);
      end = this.clipPosition(range.end);
      if (range.start.isEqual(start) && range.end.isEqual(end)) {
        return range;
      } else {
        return new Range(start, end);
      }
    };

    TextBuffer.prototype.clipPosition = function(position, options) {
      var column, lineLength, row;
      position = Point.fromObject(position);
      Point.assertValid(position);
      row = position.row, column = position.column;
      if (row < 0) {
        return this.getFirstPosition();
      } else if (row > this.getLastRow()) {
        return this.getEndPosition();
      } else if (column < 0) {
        return Point(row, 0);
      } else {
        lineLength = this.lineLengthForRow(row);
        if (column >= lineLength && (options != null ? options.clipDirection : void 0) === 'forward' && row < this.getLastRow()) {
          return Point(row + 1, 0);
        } else if (column > lineLength) {
          return Point(row, lineLength);
        } else {
          return position;
        }
      }
    };


    /*
    Section: Buffer Operations
     */

    TextBuffer.prototype.save = function() {
      return this.saveTo(this.file);
    };

    TextBuffer.prototype.saveAs = function(filePath) {
      if (!filePath) {
        throw new Error("Can't save buffer with no file path");
      }
      return this.saveTo(new File(filePath));
    };

    TextBuffer.prototype.saveTo = function(file) {
      var destination, directoryPromise, filePath;
      if (this.destroyed) {
        throw new Error("Can't save destroyed buffer");
      }
      if (!file) {
        throw new Error("Can't save a buffer with no file");
      }
      filePath = file.getPath();
      if (file instanceof File) {
        directoryPromise = new Promise(function(resolve, reject) {
          return mkdirp(path.dirname(filePath), function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        });
        destination = filePath;
      } else {
        destination = file.createWriteStream();
        directoryPromise = Promise.resolve();
      }
      this.emitter.emit('will-save', {
        path: filePath
      });
      this.outstandingSaveCount++;
      return directoryPromise.then((function(_this) {
        return function() {
          return _this.buffer.save(destination, _this.getEncoding());
        };
      })(this))["catch"]((function(_this) {
        return function(e) {
          _this.outstandingSaveCount--;
          throw e;
        };
      })(this)).then((function(_this) {
        return function() {
          _this.outstandingSaveCount--;
          _this.setFile(file);
          _this.fileHasChangedSinceLastLoad = false;
          _this.digestWhenLastPersisted = _this.buffer.baseTextDigest();
          _this.loaded = true;
          _this.emitModifiedStatusChanged(false);
          _this.emitter.emit('did-save', {
            path: filePath
          });
          return _this;
        };
      })(this));
    };

    TextBuffer.prototype.reload = function() {
      return this.load({
        discardChanges: true,
        internal: true
      });
    };


    /*
    Section: Display Layers
     */

    TextBuffer.prototype.addDisplayLayer = function(params) {
      var id;
      id = this.nextDisplayLayerId++;
      return this.displayLayers[id] = new DisplayLayer(id, this, params);
    };

    TextBuffer.prototype.getDisplayLayer = function(id) {
      return this.displayLayers[id];
    };

    TextBuffer.prototype.setDisplayLayers = function(displayLayers1) {
      this.displayLayers = displayLayers1;
    };

    TextBuffer.prototype.registerTextDecorationLayer = function(textDecorationLayer) {
      this.textDecorationLayers.add(textDecorationLayer);
      return new Disposable((function(_this) {
        return function() {
          return _this.textDecorationLayers["delete"](textDecorationLayer);
        };
      })(this));
    };


    /*
    Section: Private Utility Methods
     */

    TextBuffer.prototype.loadSync = function(options) {
      var changeEvent, checkpoint, error, patch;
      if (!(options != null ? options.internal : void 0)) {
        Grim.deprecate('The .loadSync instance method is deprecated. Create a loaded buffer using TextBuffer.loadSync(filePath) instead.');
      }
      checkpoint = null;
      changeEvent = null;
      try {
        patch = this.buffer.loadSync(this.getPath(), this.getEncoding(), (function(_this) {
          return function(percentDone, patch) {
            if (patch && patch.getChangeCount() > 0) {
              changeEvent = new CompositeChangeEvent(_this.buffer, patch);
              checkpoint = _this.history.createCheckpoint(_this.createMarkerSnapshot(), true);
              _this.emitter.emit('will-reload');
              return _this.emitter.emit('will-change', changeEvent);
            }
          };
        })(this));
      } catch (error1) {
        error = error1;
        if (error.code === 'ENOENT') {
          this.emitter.emit('did-reload');
          if (options != null ? options.discardChanges : void 0) {
            this.setText('');
          }
        } else {
          throw error;
        }
      }
      return this.finishLoading(changeEvent, checkpoint, patch);
    };

    TextBuffer.prototype.load = function(options) {
      var changeEvent, checkpoint, loadCount, source;
      if (!(options != null ? options.internal : void 0)) {
        Grim.deprecate('The .load instance method is deprecated. Create a loaded buffer using TextBuffer.load(filePath) instead.');
      }
      source = this.file instanceof File ? this.file.getPath() : this.file.createReadStream();
      checkpoint = null;
      changeEvent = null;
      loadCount = ++this.loadCount;
      return this.buffer.load(source, {
        encoding: this.getEncoding(),
        force: options != null ? options.discardChanges : void 0,
        patch: this.loaded
      }, (function(_this) {
        return function(percentDone, patch) {
          if (_this.loadCount > loadCount) {
            return false;
          }
          if (patch) {
            if (patch.getChangeCount() > 0) {
              changeEvent = new CompositeChangeEvent(_this.buffer, patch);
              checkpoint = _this.history.createCheckpoint(_this.createMarkerSnapshot(), true);
              _this.emitter.emit('will-reload');
              return _this.emitter.emit('will-change', changeEvent);
            } else if (options != null ? options.discardChanges : void 0) {
              return _this.emitter.emit('will-reload');
            }
          }
        };
      })(this)).then((function(_this) {
        return function(patch) {
          return _this.finishLoading(changeEvent, checkpoint, patch, options);
        };
      })(this))["catch"]((function(_this) {
        return function(error) {
          if (error.code === 'ENOENT') {
            _this.emitter.emit('will-reload');
            if (options != null ? options.discardChanges : void 0) {
              _this.setText('');
            }
            return _this.emitter.emit('did-reload');
          } else {
            throw error;
          }
        };
      })(this));
    };

    TextBuffer.prototype.finishLoading = function(changeEvent, checkpoint, patch, options) {
      var change, i, id, len, markerLayer, markerSnapshot, ref3, ref4;
      if (this.isDestroyed() || (this.loaded && (changeEvent == null) && (patch != null))) {
        if (options != null ? options.discardChanges : void 0) {
          this.emitter.emit('did-reload');
        }
        return null;
      }
      this.fileHasChangedSinceLastLoad = false;
      this.digestWhenLastPersisted = this.buffer.baseTextDigest();
      this.cachedText = null;
      if (this.loaded && patch && patch.getChangeCount() > 0) {
        if (options != null ? options.clearHistory : void 0) {
          this.history.clearUndoStack();
        } else {
          this.history.pushPatch(patch);
        }
        if (this.markerLayers != null) {
          ref3 = patch.getChanges();
          for (i = 0, len = ref3.length; i < len; i++) {
            change = ref3[i];
            ref4 = this.markerLayers;
            for (id in ref4) {
              markerLayer = ref4[id];
              markerLayer.splice(change.newStart, traversal(change.oldEnd, change.oldStart), traversal(change.newEnd, change.newStart));
            }
          }
        }
        changeEvent.didChange = true;
        this.emitDidChangeEvent(changeEvent);
        markerSnapshot = this.createMarkerSnapshot();
        this.history.groupChangesSinceCheckpoint(checkpoint, markerSnapshot, true);
        this.emitDidChangeTextEvent(patch);
        this.emitMarkerChangeEvents(markerSnapshot);
        this.emitModifiedStatusChanged(this.isModified());
      }
      this.loaded = true;
      this.emitter.emit('did-reload');
      return this;
    };

    TextBuffer.prototype.destroy = function() {
      var id, markerLayer, ref3, ref4;
      if (!this.destroyed) {
        this.destroyed = true;
        this.emitter.emit('did-destroy');
        this.emitter.clear();
        if ((ref3 = this.fileSubscriptions) != null) {
          ref3.dispose();
        }
        ref4 = this.markerLayers;
        for (id in ref4) {
          markerLayer = ref4[id];
          markerLayer.destroy();
        }
        this.buffer.reset('');
        this.cachedText = null;
        return this.history.clear();
      }
    };

    TextBuffer.prototype.isAlive = function() {
      return !this.destroyed;
    };

    TextBuffer.prototype.isDestroyed = function() {
      return this.destroyed;
    };

    TextBuffer.prototype.isRetained = function() {
      return this.refcount > 0;
    };

    TextBuffer.prototype.retain = function() {
      this.refcount++;
      return this;
    };

    TextBuffer.prototype.release = function() {
      this.refcount--;
      if (!this.isRetained()) {
        this.destroy();
      }
      return this;
    };

    TextBuffer.prototype.subscribeToFile = function() {
      var ref3;
      if ((ref3 = this.fileSubscriptions) != null) {
        ref3.dispose();
      }
      this.fileSubscriptions = new CompositeDisposable;
      if (this.file.onDidChange != null) {
        this.fileSubscriptions.add(this.file.onDidChange(debounce((function(_this) {
          return function() {
            var source;
            if (!_this.file.existsSync()) {
              return;
            }
            if (_this.outstandingSaveCount > 0) {
              return;
            }
            _this.fileHasChangedSinceLastLoad = true;
            if (_this.isModified()) {
              source = _this.file instanceof File ? _this.file.getPath() : _this.file.createReadStream();
              return _this.buffer.baseTextMatchesFile(source, _this.getEncoding()).then(function(matchesFile) {
                if (!matchesFile) {
                  return _this.emitter.emit('did-conflict');
                }
              });
            } else {
              return _this.load({
                internal: true
              });
            }
          };
        })(this), this.fileChangeDelay)));
      }
      if (this.file.onDidDelete != null) {
        this.fileSubscriptions.add(this.file.onDidDelete((function(_this) {
          return function() {
            var modified;
            modified = _this.isModified();
            _this.emitter.emit('did-delete');
            if (!modified && _this.shouldDestroyOnFileDelete()) {
              return _this.destroy();
            } else {
              return _this.emitModifiedStatusChanged(true);
            }
          };
        })(this)));
      }
      if (this.file.onDidRename != null) {
        this.fileSubscriptions.add(this.file.onDidRename((function(_this) {
          return function() {
            return _this.emitter.emit('did-change-path', _this.getPath());
          };
        })(this)));
      }
      if (this.file.onWillThrowWatchError != null) {
        return this.fileSubscriptions.add(this.file.onWillThrowWatchError((function(_this) {
          return function(error) {
            return _this.emitter.emit('will-throw-watch-error', error);
          };
        })(this)));
      }
    };

    TextBuffer.prototype.createMarkerSnapshot = function() {
      var markerLayer, markerLayerId, ref3, snapshot;
      snapshot = {};
      ref3 = this.markerLayers;
      for (markerLayerId in ref3) {
        markerLayer = ref3[markerLayerId];
        if (markerLayer.maintainHistory) {
          snapshot[markerLayerId] = markerLayer.createSnapshot();
        }
      }
      return snapshot;
    };

    TextBuffer.prototype.restoreFromMarkerSnapshot = function(snapshot) {
      var layerSnapshot, markerLayerId, ref3, results;
      results = [];
      for (markerLayerId in snapshot) {
        layerSnapshot = snapshot[markerLayerId];
        results.push((ref3 = this.markerLayers[markerLayerId]) != null ? ref3.restoreFromSnapshot(layerSnapshot) : void 0);
      }
      return results;
    };

    TextBuffer.prototype.emitMarkerChangeEvents = function(snapshot) {
      var i, len, markerLayer, markerLayerId, ref3, results, updatedMarkerLayers;
      if (this.transactCallDepth === 0) {
        while (this.markerLayersWithPendingUpdateEvents.size > 0) {
          updatedMarkerLayers = Array.from(this.markerLayersWithPendingUpdateEvents);
          this.markerLayersWithPendingUpdateEvents.clear();
          for (i = 0, len = updatedMarkerLayers.length; i < len; i++) {
            markerLayer = updatedMarkerLayers[i];
            markerLayer.emitUpdateEvent();
            if (markerLayer === this.defaultMarkerLayer) {
              this.emitter.emit('did-update-markers');
            }
          }
        }
      }
      ref3 = this.markerLayers;
      results = [];
      for (markerLayerId in ref3) {
        markerLayer = ref3[markerLayerId];
        results.push(markerLayer.emitChangeEvents(snapshot != null ? snapshot[markerLayerId] : void 0));
      }
      return results;
    };

    TextBuffer.prototype.emitDidChangeTextEvent = function(patch) {
      var hunks;
      if (this.transactCallDepth !== 0) {
        return;
      }
      hunks = patch.getChanges();
      if (hunks.length > 0) {
        this.emitter.emit('did-change-text', {
          changes: Object.freeze(normalizePatchChanges(hunks))
        });
        this.patchesSinceLastStoppedChangingEvent.push(patch);
        return this.debouncedEmitDidStopChangingEvent();
      }
    };

    TextBuffer.prototype.hasMultipleEditors = function() {
      return this.refcount > 1;
    };

    TextBuffer.prototype.emitDidStopChangingEvent = function() {
      var composedChanges, modifiedStatus;
      if (this.destroyed) {
        return;
      }
      modifiedStatus = this.isModified();
      composedChanges = Patch.compose(this.patchesSinceLastStoppedChangingEvent).getChanges();
      this.emitter.emit('did-stop-changing', {
        changes: Object.freeze(normalizePatchChanges(composedChanges))
      });
      this.patchesSinceLastStoppedChangingEvent = [];
      return this.emitModifiedStatusChanged(modifiedStatus);
    };

    TextBuffer.prototype.emitModifiedStatusChanged = function(modifiedStatus) {
      if (modifiedStatus === this.previousModifiedStatus) {
        return;
      }
      this.previousModifiedStatus = modifiedStatus;
      return this.emitter.emit('did-change-modified', modifiedStatus);
    };

    TextBuffer.prototype.logLines = function(start, end) {
      var i, line, ref3, ref4, row;
      if (start == null) {
        start = 0;
      }
      if (end == null) {
        end = this.getLastRow();
      }
      for (row = i = ref3 = start, ref4 = end; ref3 <= ref4 ? i <= ref4 : i >= ref4; row = ref3 <= ref4 ? ++i : --i) {
        line = this.lineForRow(row);
        console.log(row, line, line.length);
      }
    };


    /*
    Section: Private History Delegate Methods
     */

    TextBuffer.prototype.invertChange = function(change) {
      return Object.freeze({
        oldRange: change.newRange,
        newRange: change.oldRange,
        oldText: change.newText,
        newText: change.oldText
      });
    };

    TextBuffer.prototype.serializeChange = function(change) {
      return {
        oldRange: change.oldRange.serialize(),
        newRange: change.newRange.serialize(),
        oldText: change.oldText,
        newText: change.newText
      };
    };

    TextBuffer.prototype.deserializeChange = function(change) {
      return {
        oldRange: Range.deserialize(change.oldRange),
        newRange: Range.deserialize(change.newRange),
        oldText: change.oldText,
        newText: change.newText
      };
    };

    TextBuffer.prototype.serializeSnapshot = function(snapshot, options) {
      if (!options.markerLayers) {
        return;
      }
      return MarkerLayer.serializeSnapshot(snapshot);
    };

    TextBuffer.prototype.deserializeSnapshot = function(snapshot) {
      return MarkerLayer.deserializeSnapshot(snapshot);
    };


    /*
    Section: Private MarkerLayer Delegate Methods
     */

    TextBuffer.prototype.markerLayerDestroyed = function(markerLayer) {
      return delete this.markerLayers[markerLayer.id];
    };

    TextBuffer.prototype.markerCreated = function(layer, marker) {
      if (layer === this.defaultMarkerLayer) {
        return this.emitter.emit('did-create-marker', marker);
      }
    };

    TextBuffer.prototype.markersUpdated = function(layer) {
      if (this.transactCallDepth === 0) {
        layer.emitUpdateEvent();
        if (layer === this.defaultMarkerLayer) {
          return this.emitter.emit('did-update-markers');
        }
      } else {
        return this.markerLayersWithPendingUpdateEvents.add(layer);
      }
    };

    TextBuffer.prototype.getNextMarkerId = function() {
      return this.nextMarkerId++;
    };

    return TextBuffer;

  })();

}).call(this);
