(function() {
  var BufferedProcess, Client, CompositeDisposable, Emitter, PackageManager, _, createJsonParseError, createProcessError, handleProcessErrors, ref, semver,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  semver = require('semver');

  Client = require('./atom-io-client');

  module.exports = PackageManager = (function() {
    PackageManager.prototype.CACHE_EXPIRY = 1000 * 60 * 10;

    function PackageManager() {
      this.setProxyServersAsync = bind(this.setProxyServersAsync, this);
      this.setProxyServers = bind(this.setProxyServers, this);
      this.packagePromises = [];
      this.apmCache = {
        loadOutdated: {
          value: null,
          expiry: 0
        }
      };
      this.emitter = new Emitter;
    }

    PackageManager.prototype.getClient = function() {
      return this.client != null ? this.client : this.client = new Client(this);
    };

    PackageManager.prototype.isPackageInstalled = function(packageName) {
      if (atom.packages.isPackageLoaded(packageName)) {
        return true;
      } else {
        return atom.packages.getAvailablePackageNames().indexOf(packageName) > -1;
      }
    };

    PackageManager.prototype.packageHasSettings = function(packageName) {
      var grammar, grammars, i, len, pack, ref1, schema;
      grammars = (ref1 = atom.grammars.getGrammars()) != null ? ref1 : [];
      for (i = 0, len = grammars.length; i < len; i++) {
        grammar = grammars[i];
        if (grammar.path) {
          if (grammar.packageName === packageName) {
            return true;
          }
        }
      }
      pack = atom.packages.getLoadedPackage(packageName);
      if ((pack != null) && !atom.packages.isPackageActive(packageName)) {
        pack.activateConfig();
      }
      schema = atom.config.getSchema(packageName);
      return (schema != null) && (schema.type !== 'any');
    };

    PackageManager.prototype.setProxyServers = function(callback) {
      var session;
      session = atom.getCurrentWindow().webContents.session;
      return session.resolveProxy('http://atom.io', (function(_this) {
        return function(httpProxy) {
          _this.applyProxyToEnv('http_proxy', httpProxy);
          return session.resolveProxy('https://atom.io', function(httpsProxy) {
            _this.applyProxyToEnv('https_proxy', httpsProxy);
            return callback();
          });
        };
      })(this));
    };

    PackageManager.prototype.setProxyServersAsync = function(callback) {
      var httpProxyPromise, httpsProxyPromise;
      httpProxyPromise = atom.resolveProxy('http://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('http_proxy', proxy);
        };
      })(this));
      httpsProxyPromise = atom.resolveProxy('https://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('https_proxy', proxy);
        };
      })(this));
      return Promise.all([httpProxyPromise, httpsProxyPromise]).then(callback);
    };

    PackageManager.prototype.applyProxyToEnv = function(envName, proxy) {
      if (proxy != null) {
        proxy = proxy.split(' ');
        switch (proxy[0].trim().toUpperCase()) {
          case 'DIRECT':
            delete process.env[envName];
            break;
          case 'PROXY':
            process.env[envName] = 'http://' + proxy[1];
        }
      }
    };

    PackageManager.prototype.runCommand = function(args, callback) {
      var bufferedProcess, command, errorLines, exit, outputLines, stderr, stdout;
      command = atom.packages.getApmPath();
      outputLines = [];
      stdout = function(lines) {
        return outputLines.push(lines);
      };
      errorLines = [];
      stderr = function(lines) {
        return errorLines.push(lines);
      };
      exit = function(code) {
        return callback(code, outputLines.join('\n'), errorLines.join('\n'));
      };
      args.push('--no-color');
      if (atom.config.get('core.useProxySettingsWhenCallingApm')) {
        bufferedProcess = new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit,
          autoStart: false
        });
        if (atom.resolveProxy != null) {
          this.setProxyServersAsync(function() {
            return bufferedProcess.start();
          });
        } else {
          this.setProxyServers(function() {
            return bufferedProcess.start();
          });
        }
        return bufferedProcess;
      } else {
        return new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit
        });
      }
    };

    PackageManager.prototype.loadInstalled = function(callback) {
      var apmProcess, args, errorMessage;
      args = ['ls', '--json'];
      errorMessage = 'Fetching local packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadFeatured = function(loadThemes, callback) {
      var apmProcess, args, errorMessage, version;
      if (!callback) {
        callback = loadThemes;
        loadThemes = false;
      }
      args = ['featured', '--json'];
      version = atom.getVersion();
      if (loadThemes) {
        args.push('--themes');
      }
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching featured packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadOutdated = function(clearCache, callback) {
      var apmProcess, args, errorMessage, version;
      if (clearCache) {
        this.clearOutdatedCache();
      } else if (this.apmCache.loadOutdated.value && this.apmCache.loadOutdated.expiry > Date.now()) {
        return callback(null, this.apmCache.loadOutdated.value);
      }
      args = ['outdated', '--json'];
      version = atom.getVersion();
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching outdated packages and themes failed.';
      apmProcess = this.runCommand(args, (function(_this) {
        return function(code, stdout, stderr) {
          var error, i, len, pack, packages, parseError, ref1, updatablePackages;
          if (code === 0) {
            try {
              packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
            } catch (error1) {
              parseError = error1;
              error = createJsonParseError(errorMessage, parseError, stdout);
              return callback(error);
            }
            updatablePackages = (function() {
              var i, len, results;
              results = [];
              for (i = 0, len = packages.length; i < len; i++) {
                pack = packages[i];
                if (!this.getVersionPinnedPackages().includes(pack != null ? pack.name : void 0)) {
                  results.push(pack);
                }
              }
              return results;
            }).call(_this);
            _this.apmCache.loadOutdated = {
              value: updatablePackages,
              expiry: Date.now() + _this.CACHE_EXPIRY
            };
            for (i = 0, len = updatablePackages.length; i < len; i++) {
              pack = updatablePackages[i];
              _this.emitPackageEvent('update-available', pack);
            }
            return callback(null, updatablePackages);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return callback(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getVersionPinnedPackages = function() {
      var ref1;
      return (ref1 = atom.config.get('core.versionPinnedPackages')) != null ? ref1 : [];
    };

    PackageManager.prototype.clearOutdatedCache = function() {
      return this.apmCache.loadOutdated = {
        value: null,
        expiry: 0
      };
    };

    PackageManager.prototype.loadPackage = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json'];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadCompatiblePackageVersion = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json', '--compatible', this.normalizeVersion(atom.getVersion())];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getInstalled = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadInstalled(function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getFeatured = function(loadThemes) {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadFeatured(!!loadThemes, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getOutdated = function(clearCache) {
      if (clearCache == null) {
        clearCache = false;
      }
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadOutdated(clearCache, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getPackage = function(packageName) {
      var base;
      return (base = this.packagePromises)[packageName] != null ? base[packageName] : base[packageName] = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadPackage(packageName, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.satisfiesVersion = function(version, metadata) {
      var engine, ref1, ref2;
      engine = (ref1 = (ref2 = metadata.engines) != null ? ref2.atom : void 0) != null ? ref1 : '*';
      if (!semver.validRange(engine)) {
        return false;
      }
      return semver.satisfies(version, engine);
    };

    PackageManager.prototype.normalizeVersion = function(version) {
      if (typeof version === 'string') {
        version = version.split('-')[0];
      }
      return version;
    };

    PackageManager.prototype.search = function(query, options) {
      if (options == null) {
        options = {};
      }
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var apmProcess, args, errorMessage;
          args = ['search', query, '--json'];
          if (options.themes) {
            args.push('--themes');
          } else if (options.packages) {
            args.push('--packages');
          }
          errorMessage = "Searching for \u201C" + query + "\u201D failed.";
          apmProcess = _this.runCommand(args, function(code, stdout, stderr) {
            var error, packages, parseError, ref1;
            if (code === 0) {
              try {
                packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
                if (options.sortBy) {
                  packages = _.sortBy(packages, function(pkg) {
                    return pkg[options.sortBy] * -1;
                  });
                }
                return resolve(packages);
              } catch (error1) {
                parseError = error1;
                error = createJsonParseError(errorMessage, parseError, stdout);
                return reject(error);
              }
            } else {
              error = new Error(errorMessage);
              error.stdout = stdout;
              error.stderr = stderr;
              return reject(error);
            }
          });
          return handleProcessErrors(apmProcess, errorMessage, function(error) {
            return reject(error);
          });
        };
      })(this));
    };

    PackageManager.prototype.update = function(pack, newVersion, callback) {
      var apmInstallSource, apmProcess, args, errorMessage, exit, name, onError, theme;
      name = pack.name, theme = pack.theme, apmInstallSource = pack.apmInstallSource;
      errorMessage = newVersion ? "Updating to \u201C" + name + "@" + newVersion + "\u201D failed." : "Updating to latest sha failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('update-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      if ((apmInstallSource != null ? apmInstallSource.type : void 0) === 'git') {
        args = ['install', apmInstallSource.source];
      } else {
        args = ['install', name + "@" + newVersion];
      }
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('updated', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('updating', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.unload = function(name) {
      if (atom.packages.isPackageLoaded(name)) {
        if (atom.packages.isPackageActive(name)) {
          atom.packages.deactivatePackage(name);
        }
        return atom.packages.unloadPackage(name);
      }
    };

    PackageManager.prototype.install = function(pack, callback) {
      var activateOnFailure, activateOnSuccess, apmProcess, args, errorMessage, exit, name, nameWithVersion, onError, theme, version;
      name = pack.name, version = pack.version, theme = pack.theme;
      activateOnSuccess = !theme && !atom.packages.isPackageDisabled(name);
      activateOnFailure = atom.packages.isPackageActive(name);
      nameWithVersion = version != null ? name + "@" + version : name;
      this.unload(name);
      args = ['install', nameWithVersion, '--json'];
      errorMessage = "Installing \u201C" + nameWithVersion + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('install-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var err, error, packageInfo;
          if (code === 0) {
            try {
              packageInfo = JSON.parse(stdout)[0];
              pack = _.extend({}, pack, packageInfo.metadata);
              name = pack.name;
            } catch (error1) {
              err = error1;
            }
            _this.clearOutdatedCache();
            if (activateOnSuccess) {
              atom.packages.activatePackage(name);
            } else {
              atom.packages.loadPackage(name);
            }
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('installed', pack);
          } else {
            if (activateOnFailure) {
              atom.packages.activatePackage(name);
            }
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('installing', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.uninstall = function(pack, callback) {
      var apmProcess, errorMessage, name, onError;
      name = pack.name;
      if (atom.packages.isPackageActive(name)) {
        atom.packages.deactivatePackage(name);
      }
      errorMessage = "Uninstalling \u201C" + name + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          _this.emitPackageEvent('uninstall-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      this.emitPackageEvent('uninstalling', pack);
      apmProcess = this.runCommand(['uninstall', '--hard', name], (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            _this.unload(name);
            _this.removePackageNameFromDisabledPackages(name);
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('uninstalled', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.installAlternative = function(pack, alternativePackageName, callback) {
      var eventArg, installPromise, uninstallPromise;
      eventArg = {
        pack: pack,
        alternative: alternativePackageName
      };
      this.emitter.emit('package-installing-alternative', eventArg);
      uninstallPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.uninstall(pack, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      installPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.install({
            name: alternativePackageName
          }, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      return Promise.all([uninstallPromise, installPromise]).then((function(_this) {
        return function() {
          callback(null, eventArg);
          return _this.emitter.emit('package-installed-alternative', eventArg);
        };
      })(this))["catch"]((function(_this) {
        return function(error) {
          console.error(error.message, error.stack);
          callback(error, eventArg);
          eventArg.error = error;
          return _this.emitter.emit('package-install-alternative-failed', eventArg);
        };
      })(this));
    };

    PackageManager.prototype.canUpgrade = function(installedPackage, availableVersion) {
      var installedVersion;
      if (installedPackage == null) {
        return false;
      }
      installedVersion = installedPackage.metadata.version;
      if (!semver.valid(installedVersion)) {
        return false;
      }
      if (!semver.valid(availableVersion)) {
        return false;
      }
      return semver.gt(availableVersion, installedVersion);
    };

    PackageManager.prototype.getPackageTitle = function(arg) {
      var name;
      name = arg.name;
      return _.undasherize(_.uncamelcase(name));
    };

    PackageManager.prototype.getRepositoryUrl = function(arg) {
      var metadata, ref1, ref2, repoName, repoUrl, repository;
      metadata = arg.metadata;
      repository = metadata.repository;
      repoUrl = (ref1 = (ref2 = repository != null ? repository.url : void 0) != null ? ref2 : repository) != null ? ref1 : '';
      if (repoUrl.match('git@github')) {
        repoName = repoUrl.split(':')[1];
        repoUrl = "https://github.com/" + repoName;
      }
      return repoUrl.replace(/\.git$/, '').replace(/\/+$/, '').replace(/^git\+/, '');
    };

    PackageManager.prototype.checkNativeBuildTools = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var apmProcess;
          apmProcess = _this.runCommand(['install', '--check'], function(code, stdout, stderr) {
            if (code === 0) {
              return resolve();
            } else {
              return reject(new Error());
            }
          });
          return apmProcess.onWillThrowError(function(arg) {
            var error, handle;
            error = arg.error, handle = arg.handle;
            handle();
            return reject(error);
          });
        };
      })(this));
    };

    PackageManager.prototype.removePackageNameFromDisabledPackages = function(packageName) {
      return atom.config.removeAtKeyPath('core.disabledPackages', packageName);
    };

    PackageManager.prototype.emitPackageEvent = function(eventName, pack, error) {
      var ref1, ref2, theme;
      theme = (ref1 = pack.theme) != null ? ref1 : (ref2 = pack.metadata) != null ? ref2.theme : void 0;
      eventName = theme ? "theme-" + eventName : "package-" + eventName;
      return this.emitter.emit(eventName, {
        pack: pack,
        error: error
      });
    };

    PackageManager.prototype.on = function(selectors, callback) {
      var i, len, ref1, selector, subscriptions;
      subscriptions = new CompositeDisposable;
      ref1 = selectors.split(" ");
      for (i = 0, len = ref1.length; i < len; i++) {
        selector = ref1[i];
        subscriptions.add(this.emitter.on(selector, callback));
      }
      return subscriptions;
    };

    return PackageManager;

  })();

  createJsonParseError = function(message, parseError, stdout) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = parseError.message + ": " + stdout;
    return error;
  };

  createProcessError = function(message, processError) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = processError.message;
    return error;
  };

  handleProcessErrors = function(apmProcess, message, callback) {
    return apmProcess.onWillThrowError(function(arg) {
      var error, handle;
      error = arg.error, handle = arg.handle;
      handle();
      return callback(createProcessError(message, error));
    });
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
