(function() {
  var BufferedProcess, CompositeDisposable, Disposable, PackageGeneratorView, TextEditor, _, fs, path, ref,
    slice = [].slice;

  path = require('path');

  _ = require('underscore-plus');

  ref = require('atom'), TextEditor = ref.TextEditor, BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Disposable = ref.Disposable;

  fs = require('fs-plus');

  module.exports = PackageGeneratorView = (function() {
    PackageGeneratorView.prototype.previouslyFocusedElement = null;

    PackageGeneratorView.prototype.mode = null;

    function PackageGeneratorView() {
      var blurHandler;
      this.disposables = new CompositeDisposable;
      this.element = document.createElement('div');
      this.element.classList.add('package-generator');
      this.miniEditor = new TextEditor({
        mini: true
      });
      this.element.appendChild(this.miniEditor.element);
      this.error = document.createElement('div');
      this.error.classList.add('error');
      this.element.appendChild(this.error);
      this.message = document.createElement('div');
      this.message.classList.add('message');
      this.element.appendChild(this.message);
      this.disposables.add(atom.commands.add('atom-workspace', {
        'package-generator:generate-package': (function(_this) {
          return function() {
            return _this.attach('package');
          };
        })(this),
        'package-generator:generate-syntax-theme': (function(_this) {
          return function() {
            return _this.attach('theme');
          };
        })(this)
      }));
      blurHandler = (function(_this) {
        return function() {
          return _this.close();
        };
      })(this);
      this.miniEditor.element.addEventListener('blur', blurHandler);
      this.disposables.add(new Disposable((function(_this) {
        return function() {
          return _this.miniEditor.element.removeEventListener('blur', blurHandler);
        };
      })(this)));
      this.disposables.add(atom.commands.add(this.element, {
        'core:confirm': (function(_this) {
          return function() {
            return _this.confirm();
          };
        })(this),
        'core:cancel': (function(_this) {
          return function() {
            return _this.close();
          };
        })(this)
      }));
    }

    PackageGeneratorView.prototype.destroy = function() {
      var ref1;
      if ((ref1 = this.panel) != null) {
        ref1.destroy();
      }
      return this.disposables.dispose();
    };

    PackageGeneratorView.prototype.attach = function(mode) {
      this.mode = mode;
      if (this.panel == null) {
        this.panel = atom.workspace.addModalPanel({
          item: this,
          visible: false
        });
      }
      this.previouslyFocusedElement = document.activeElement;
      this.panel.show();
      this.message.textContent = "Enter " + this.mode + " path";
      if (this.isInPackageMode()) {
        this.setPathText("my-package");
      } else {
        this.setPathText("my-theme-syntax", [0, 8]);
      }
      return this.miniEditor.element.focus();
    };

    PackageGeneratorView.prototype.setPathText = function(placeholderName, rangeToSelect) {
      var endOfDirectoryIndex, packagesDirectory, pathLength;
      if (rangeToSelect == null) {
        rangeToSelect = [0, placeholderName.length];
      }
      packagesDirectory = this.getPackagesDirectory();
      this.miniEditor.setText(path.join(packagesDirectory, placeholderName));
      pathLength = this.miniEditor.getText().length;
      endOfDirectoryIndex = pathLength - placeholderName.length;
      return this.miniEditor.setSelectedBufferRange([[0, endOfDirectoryIndex + rangeToSelect[0]], [0, endOfDirectoryIndex + rangeToSelect[1]]]);
    };

    PackageGeneratorView.prototype.close = function() {
      var ref1;
      if (!this.panel.isVisible()) {
        return;
      }
      this.panel.hide();
      return (ref1 = this.previouslyFocusedElement) != null ? ref1.focus() : void 0;
    };

    PackageGeneratorView.prototype.confirm = function() {
      if (this.validPackagePath()) {
        return this.createPackageFiles((function(_this) {
          return function() {
            var packagePath;
            packagePath = _this.getPackagePath();
            atom.open({
              pathsToOpen: [packagePath]
            });
            return _this.close();
          };
        })(this));
      }
    };

    PackageGeneratorView.prototype.getPackagePath = function() {
      var packageName, packagePath;
      packagePath = fs.normalize(this.miniEditor.getText().trim());
      packageName = _.dasherize(path.basename(packagePath));
      return path.join(path.dirname(packagePath), packageName);
    };

    PackageGeneratorView.prototype.getPackagesDirectory = function() {
      return atom.config.get('core.projectHome') || process.env.ATOM_REPOS_HOME || path.join(fs.getHomeDirectory(), 'github');
    };

    PackageGeneratorView.prototype.validPackagePath = function() {
      if (fs.existsSync(this.getPackagePath())) {
        this.error.textContent = "Path already exists at '" + (this.getPackagePath()) + "'";
        this.error.style.display = 'block';
        return false;
      } else {
        return true;
      }
    };

    PackageGeneratorView.prototype.getInitOptions = function(packagePath) {
      var options;
      options = ["--" + this.mode, packagePath];
      if (this.isInPackageMode()) {
        return slice.call(options).concat(['--syntax'], [atom.config.get('package-generator.packageSyntax')]);
      } else {
        return options;
      }
    };

    PackageGeneratorView.prototype.initPackage = function(packagePath, callback) {
      var command;
      command = ['init'].concat(slice.call(this.getInitOptions(packagePath)));
      return this.runCommand(atom.packages.getApmPath(), command, callback);
    };

    PackageGeneratorView.prototype.linkPackage = function(packagePath, callback) {
      var args;
      args = ['link'];
      if (atom.config.get('package-generator.createInDevMode')) {
        args.push('--dev');
      }
      args.push(packagePath.toString());
      return this.runCommand(atom.packages.getApmPath(), args, callback);
    };

    PackageGeneratorView.prototype.isInPackageMode = function() {
      return this.mode === 'package';
    };

    PackageGeneratorView.prototype.isStoredInDotAtom = function(packagePath) {
      var devPackagesPath, packagesPath;
      packagesPath = path.join(atom.getConfigDirPath(), 'packages', path.sep);
      if (packagePath.indexOf(packagesPath) === 0) {
        return true;
      }
      devPackagesPath = path.join(atom.getConfigDirPath(), 'dev', 'packages', path.sep);
      return packagePath.indexOf(devPackagesPath) === 0;
    };

    PackageGeneratorView.prototype.createPackageFiles = function(callback) {
      var packagePath;
      packagePath = this.getPackagePath();
      if (this.isStoredInDotAtom(packagePath)) {
        return this.initPackage(packagePath, callback);
      } else {
        return this.initPackage(packagePath, (function(_this) {
          return function() {
            return _this.linkPackage(packagePath, callback);
          };
        })(this));
      }
    };

    PackageGeneratorView.prototype.runCommand = function(command, args, exit) {
      return new BufferedProcess({
        command: command,
        args: args,
        exit: exit
      });
    };

    return PackageGeneratorView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
