(function() {
  var GitHubFile, Range, parseUrl, path, shell;

  shell = require('electron').shell;

  Range = require('atom').Range;

  parseUrl = require('url').parse;

  path = require('path');

  module.exports = GitHubFile = (function() {
    GitHubFile.fromPath = function(filePath) {
      return new GitHubFile(filePath);
    };

    function GitHubFile(filePath1) {
      var rootDir, rootDirIndex;
      this.filePath = filePath1;
      rootDir = atom.project.relativizePath(this.filePath)[0];
      if (rootDir != null) {
        rootDirIndex = atom.project.getPaths().indexOf(rootDir);
        this.repo = atom.project.getRepositories()[rootDirIndex];
      }
    }

    GitHubFile.prototype.open = function(lineRange) {
      if (this.isOpenable()) {
        return this.openUrlInBrowser(this.blobUrl() + this.getLineRangeSuffix(lineRange));
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.openOnMaster = function(lineRange) {
      if (this.isOpenable()) {
        return this.openUrlInBrowser(this.blobUrlForMaster() + this.getLineRangeSuffix(lineRange));
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.blame = function(lineRange) {
      if (this.isOpenable()) {
        return this.openUrlInBrowser(this.blameUrl() + this.getLineRangeSuffix(lineRange));
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.history = function() {
      if (this.isOpenable()) {
        return this.openUrlInBrowser(this.historyUrl());
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.copyUrl = function(lineRange) {
      if (this.isOpenable()) {
        return atom.clipboard.write(this.shaUrl() + this.getLineRangeSuffix(lineRange));
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.openBranchCompare = function() {
      if (this.isOpenable()) {
        return this.openUrlInBrowser(this.branchCompareUrl());
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.openIssues = function() {
      if (this.isOpenable()) {
        return this.openUrlInBrowser(this.issuesUrl());
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.openRepository = function() {
      if (this.isOpenable()) {
        return this.openUrlInBrowser(this.githubRepoUrl());
      } else {
        return this.reportValidationErrors();
      }
    };

    GitHubFile.prototype.getLineRangeSuffix = function(lineRange) {
      var endRow, startRow;
      if (lineRange && atom.config.get('open-on-github.includeLineNumbersInUrls')) {
        lineRange = Range.fromObject(lineRange);
        startRow = lineRange.start.row + 1;
        endRow = lineRange.end.row + 1;
        if (startRow === endRow) {
          return "#L" + startRow;
        } else {
          return "#L" + startRow + "-L" + endRow;
        }
      } else {
        return '';
      }
    };

    GitHubFile.prototype.isOpenable = function() {
      return this.validationErrors().length === 0;
    };

    GitHubFile.prototype.validationErrors = function() {
      if (!this.repo) {
        return ["No repository found for path: " + this.filePath + "."];
      }
      if (!this.gitUrl()) {
        return ["No URL defined for remote: " + (this.remoteName())];
      }
      if (!this.githubRepoUrl()) {
        return ["Remote URL is not hosted on GitHub: " + (this.gitUrl())];
      }
      return [];
    };

    GitHubFile.prototype.reportValidationErrors = function() {
      var message;
      message = this.validationErrors().join('\n');
      return atom.notifications.addWarning(message);
    };

    GitHubFile.prototype.openUrlInBrowser = function(url) {
      return shell.openExternal(url);
    };

    GitHubFile.prototype.blobUrl = function() {
      var gitHubRepoUrl, remoteBranchName, repoRelativePath;
      gitHubRepoUrl = this.githubRepoUrl();
      remoteBranchName = this.remoteBranchName();
      repoRelativePath = this.repoRelativePath();
      if (this.isGitHubWikiUrl(gitHubRepoUrl)) {
        return (gitHubRepoUrl.slice(0, -5)) + "/wiki/" + (this.extractFileName(repoRelativePath));
      } else {
        return gitHubRepoUrl + "/blob/" + remoteBranchName + "/" + (this.encodeSegments(repoRelativePath));
      }
    };

    GitHubFile.prototype.blobUrlForMaster = function() {
      return (this.githubRepoUrl()) + "/blob/master/" + (this.encodeSegments(this.repoRelativePath()));
    };

    GitHubFile.prototype.shaUrl = function() {
      return (this.githubRepoUrl()) + "/blob/" + (this.encodeSegments(this.sha())) + "/" + (this.encodeSegments(this.repoRelativePath()));
    };

    GitHubFile.prototype.blameUrl = function() {
      return (this.githubRepoUrl()) + "/blame/" + (this.remoteBranchName()) + "/" + (this.encodeSegments(this.repoRelativePath()));
    };

    GitHubFile.prototype.historyUrl = function() {
      return (this.githubRepoUrl()) + "/commits/" + (this.remoteBranchName()) + "/" + (this.encodeSegments(this.repoRelativePath()));
    };

    GitHubFile.prototype.issuesUrl = function() {
      return (this.githubRepoUrl()) + "/issues";
    };

    GitHubFile.prototype.branchCompareUrl = function() {
      return (this.githubRepoUrl()) + "/compare/" + (this.encodeSegments(this.branchName()));
    };

    GitHubFile.prototype.encodeSegments = function(segments) {
      if (segments == null) {
        segments = '';
      }
      segments = segments.split('/');
      segments = segments.map(function(segment) {
        return encodeURIComponent(segment);
      });
      return segments.join('/');
    };

    GitHubFile.prototype.extractFileName = function(relativePath) {
      if (relativePath == null) {
        relativePath = '';
      }
      return path.parse(relativePath).name;
    };

    GitHubFile.prototype.gitUrl = function() {
      var ref, remoteOrBestGuess;
      remoteOrBestGuess = (ref = this.remoteName()) != null ? ref : 'origin';
      return this.repo.getConfigValue("remote." + remoteOrBestGuess + ".url", this.filePath);
    };

    GitHubFile.prototype.githubRepoUrl = function() {
      var url;
      url = this.gitUrl();
      if (url.match(/git@[^:]+:/)) {
        url = url.replace(/^git@([^:]+):(.+)$/, function(match, host, repoPath) {
          repoPath = repoPath.replace(/^\/+/, '');
          return "http://" + host + "/" + repoPath;
        });
      } else if (url.match(/ssh:\/\/git@([^\/]+)\//)) {
        url = "http://" + (url.substring(10));
      } else if (url.match(/^git:\/\/[^\/]+\//)) {
        url = "http" + (url.substring(3));
      }
      url = url.replace(/\.git$/, '');
      url = url.replace(/\/+$/, '');
      if (!this.isBitbucketUrl(url)) {
        return url;
      }
    };

    GitHubFile.prototype.isGitHubWikiUrl = function(url) {
      return /\.wiki$/.test(url);
    };

    GitHubFile.prototype.isBitbucketUrl = function(url) {
      var host;
      if (url.indexOf('git@bitbucket.org') === 0) {
        return true;
      }
      try {
        host = parseUrl(url).host;
        return host === 'bitbucket.org';
      } catch (error) {}
    };

    GitHubFile.prototype.repoRelativePath = function() {
      return this.repo.getRepo(this.filePath).relativize(this.filePath);
    };

    GitHubFile.prototype.remoteName = function() {
      var branchRemote, gitConfigRemote, shortBranch;
      gitConfigRemote = this.repo.getConfigValue("atom.open-on-github.remote", this.filePath);
      if (gitConfigRemote) {
        return gitConfigRemote;
      }
      shortBranch = this.repo.getShortHead(this.filePath);
      if (!shortBranch) {
        return null;
      }
      branchRemote = this.repo.getConfigValue("branch." + shortBranch + ".remote", this.filePath);
      if (!((branchRemote != null ? branchRemote.length : void 0) > 0)) {
        return null;
      }
      return branchRemote;
    };

    GitHubFile.prototype.sha = function() {
      return this.repo.getReferenceTarget('HEAD', this.filePath);
    };

    GitHubFile.prototype.branchName = function() {
      var branchMerge, shortBranch;
      shortBranch = this.repo.getShortHead(this.filePath);
      if (!shortBranch) {
        return null;
      }
      branchMerge = this.repo.getConfigValue("branch." + shortBranch + ".merge", this.filePath);
      if (!((branchMerge != null ? branchMerge.length : void 0) > 11)) {
        return shortBranch;
      }
      if (branchMerge.indexOf('refs/heads/') !== 0) {
        return shortBranch;
      }
      return branchMerge.substring(11);
    };

    GitHubFile.prototype.remoteBranchName = function() {
      var gitConfigBranch;
      gitConfigBranch = this.repo.getConfigValue("atom.open-on-github.branch", this.filePath);
      if (gitConfigBranch) {
        return gitConfigBranch;
      } else if (this.remoteName() != null) {
        return this.encodeSegments(this.branchName());
      } else {
        return 'master';
      }
    };

    return GitHubFile;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
