(function() {
  var CompositeDisposable, Disposable, Emitter, File, MarkdownPreviewView, _, fs, path, ref, renderer;

  path = require('path');

  ref = require('atom'), Emitter = ref.Emitter, Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable, File = ref.File;

  _ = require('underscore-plus');

  fs = require('fs-plus');

  renderer = require('./renderer');

  module.exports = MarkdownPreviewView = (function() {
    MarkdownPreviewView.deserialize = function(params) {
      return new MarkdownPreviewView(params);
    };

    function MarkdownPreviewView(arg) {
      this.editorId = arg.editorId, this.filePath = arg.filePath;
      this.element = document.createElement('div');
      this.element.classList.add('markdown-preview', 'native-key-bindings');
      this.element.tabIndex = -1;
      this.emitter = new Emitter;
      this.loaded = false;
      this.disposables = new CompositeDisposable;
      this.registerScrollCommands();
      if (this.editorId != null) {
        this.resolveEditor(this.editorId);
      } else if (atom.workspace != null) {
        this.subscribeToFilePath(this.filePath);
      } else {
        this.disposables.add(atom.packages.onDidActivateInitialPackages((function(_this) {
          return function() {
            return _this.subscribeToFilePath(_this.filePath);
          };
        })(this)));
      }
    }

    MarkdownPreviewView.prototype.serialize = function() {
      var ref1;
      return {
        deserializer: 'MarkdownPreviewView',
        filePath: (ref1 = this.getPath()) != null ? ref1 : this.filePath,
        editorId: this.editorId
      };
    };

    MarkdownPreviewView.prototype.copy = function() {
      var ref1;
      return new MarkdownPreviewView({
        editorId: this.editorId,
        filePath: (ref1 = this.getPath()) != null ? ref1 : this.filePath
      });
    };

    MarkdownPreviewView.prototype.destroy = function() {
      this.disposables.dispose();
      return this.element.remove();
    };

    MarkdownPreviewView.prototype.registerScrollCommands = function() {
      this.disposables.add(atom.commands.add(this.element, {
        'core:move-up': (function(_this) {
          return function() {
            _this.element.scrollTop -= document.body.offsetHeight / 20;
          };
        })(this),
        'core:move-down': (function(_this) {
          return function() {
            _this.element.scrollTop += document.body.offsetHeight / 20;
          };
        })(this),
        'core:page-up': (function(_this) {
          return function() {
            _this.element.scrollTop -= _this.element.offsetHeight;
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            _this.element.scrollTop += _this.element.offsetHeight;
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            _this.element.scrollTop = 0;
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            _this.element.scrollTop = _this.element.scrollHeight;
          };
        })(this)
      }));
    };

    MarkdownPreviewView.prototype.onDidChangeTitle = function(callback) {
      return this.emitter.on('did-change-title', callback);
    };

    MarkdownPreviewView.prototype.onDidChangeModified = function(callback) {
      return new Disposable;
    };

    MarkdownPreviewView.prototype.onDidChangeMarkdown = function(callback) {
      return this.emitter.on('did-change-markdown', callback);
    };

    MarkdownPreviewView.prototype.subscribeToFilePath = function(filePath) {
      this.file = new File(filePath);
      this.emitter.emit('did-change-title');
      this.disposables.add(this.file.onDidRename((function(_this) {
        return function() {
          return _this.emitter.emit('did-change-title');
        };
      })(this)));
      this.handleEvents();
      return this.renderMarkdown();
    };

    MarkdownPreviewView.prototype.resolveEditor = function(editorId) {
      var resolve;
      resolve = (function(_this) {
        return function() {
          _this.editor = _this.editorForId(editorId);
          if (_this.editor != null) {
            _this.emitter.emit('did-change-title');
            _this.disposables.add(_this.editor.onDidDestroy(function() {
              return _this.subscribeToFilePath(_this.getPath());
            }));
            _this.handleEvents();
            return _this.renderMarkdown();
          } else {
            return _this.subscribeToFilePath(_this.filePath);
          }
        };
      })(this);
      if (atom.workspace != null) {
        return resolve();
      } else {
        return this.disposables.add(atom.packages.onDidActivateInitialPackages(resolve));
      }
    };

    MarkdownPreviewView.prototype.editorForId = function(editorId) {
      var editor, i, len, ref1, ref2;
      ref1 = atom.workspace.getTextEditors();
      for (i = 0, len = ref1.length; i < len; i++) {
        editor = ref1[i];
        if (((ref2 = editor.id) != null ? ref2.toString() : void 0) === editorId.toString()) {
          return editor;
        }
      }
      return null;
    };

    MarkdownPreviewView.prototype.handleEvents = function() {
      var changeHandler;
      this.disposables.add(atom.grammars.onDidAddGrammar((function(_this) {
        return function() {
          return _.debounce((function() {
            return _this.renderMarkdown();
          }), 250);
        };
      })(this)));
      this.disposables.add(atom.grammars.onDidUpdateGrammar(_.debounce(((function(_this) {
        return function() {
          return _this.renderMarkdown();
        };
      })(this)), 250)));
      atom.commands.add(this.element, {
        'core:save-as': (function(_this) {
          return function(event) {
            event.stopPropagation();
            return _this.saveAs();
          };
        })(this),
        'core:copy': (function(_this) {
          return function(event) {
            if (_this.copyToClipboard()) {
              return event.stopPropagation();
            }
          };
        })(this),
        'markdown-preview:zoom-in': (function(_this) {
          return function() {
            var zoomLevel;
            zoomLevel = parseFloat(getComputedStyle(_this.element).zoom);
            return _this.element.style.zoom = zoomLevel + 0.1;
          };
        })(this),
        'markdown-preview:zoom-out': (function(_this) {
          return function() {
            var zoomLevel;
            zoomLevel = parseFloat(getComputedStyle(_this.element).zoom);
            return _this.element.style.zoom = zoomLevel - 0.1;
          };
        })(this),
        'markdown-preview:reset-zoom': (function(_this) {
          return function() {
            return _this.element.style.zoom = 1;
          };
        })(this)
      });
      changeHandler = (function(_this) {
        return function() {
          var pane;
          _this.renderMarkdown();
          pane = atom.workspace.paneForItem(_this);
          if ((pane != null) && pane !== atom.workspace.getActivePane()) {
            return pane.activateItem(_this);
          }
        };
      })(this);
      if (this.file != null) {
        this.disposables.add(this.file.onDidChange(changeHandler));
      } else if (this.editor != null) {
        this.disposables.add(this.editor.getBuffer().onDidStopChanging(function() {
          if (atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
        this.disposables.add(this.editor.onDidChangePath((function(_this) {
          return function() {
            return _this.emitter.emit('did-change-title');
          };
        })(this)));
        this.disposables.add(this.editor.getBuffer().onDidSave(function() {
          if (!atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
        this.disposables.add(this.editor.getBuffer().onDidReload(function() {
          if (!atom.config.get('markdown-preview.liveUpdate')) {
            return changeHandler();
          }
        }));
      }
      this.disposables.add(atom.config.onDidChange('markdown-preview.breakOnSingleNewline', changeHandler));
      return this.disposables.add(atom.config.observe('markdown-preview.useGitHubStyle', (function(_this) {
        return function(useGitHubStyle) {
          if (useGitHubStyle) {
            return _this.element.setAttribute('data-use-github-style', '');
          } else {
            return _this.element.removeAttribute('data-use-github-style');
          }
        };
      })(this)));
    };

    MarkdownPreviewView.prototype.renderMarkdown = function() {
      if (!this.loaded) {
        this.showLoading();
      }
      return this.getMarkdownSource().then((function(_this) {
        return function(source) {
          if (source != null) {
            return _this.renderMarkdownText(source);
          }
        };
      })(this))["catch"]((function(_this) {
        return function(reason) {
          return _this.showError({
            message: reason
          });
        };
      })(this));
    };

    MarkdownPreviewView.prototype.getMarkdownSource = function() {
      var ref1;
      if ((ref1 = this.file) != null ? ref1.getPath() : void 0) {
        return this.file.read().then((function(_this) {
          return function(source) {
            if (source === null) {
              return Promise.reject((_this.file.getBaseName()) + " could not be found");
            } else {
              return Promise.resolve(source);
            }
          };
        })(this))["catch"](function(reason) {
          return Promise.reject(reason);
        });
      } else if (this.editor != null) {
        return Promise.resolve(this.editor.getText());
      } else {
        return Promise.reject();
      }
    };

    MarkdownPreviewView.prototype.getHTML = function(callback) {
      return this.getMarkdownSource().then((function(_this) {
        return function(source) {
          if (source == null) {
            return;
          }
          return renderer.toHTML(source, _this.getPath(), _this.getGrammar(), callback);
        };
      })(this));
    };

    MarkdownPreviewView.prototype.renderMarkdownText = function(text) {
      return renderer.toDOMFragment(text, this.getPath(), this.getGrammar(), (function(_this) {
        return function(error, domFragment) {
          if (error) {
            return _this.showError(error);
          } else {
            _this.loading = false;
            _this.loaded = true;
            _this.element.textContent = '';
            _this.element.appendChild(domFragment);
            return _this.emitter.emit('did-change-markdown');
          }
        };
      })(this));
    };

    MarkdownPreviewView.prototype.getTitle = function() {
      if ((this.file != null) && (this.getPath() != null)) {
        return (path.basename(this.getPath())) + " Preview";
      } else if (this.editor != null) {
        return (this.editor.getTitle()) + " Preview";
      } else {
        return "Markdown Preview";
      }
    };

    MarkdownPreviewView.prototype.getIconName = function() {
      return "markdown";
    };

    MarkdownPreviewView.prototype.getURI = function() {
      if (this.file != null) {
        return "markdown-preview://" + (this.getPath());
      } else {
        return "markdown-preview://editor/" + this.editorId;
      }
    };

    MarkdownPreviewView.prototype.getPath = function() {
      if (this.file != null) {
        return this.file.getPath();
      } else if (this.editor != null) {
        return this.editor.getPath();
      }
    };

    MarkdownPreviewView.prototype.getGrammar = function() {
      var ref1;
      return (ref1 = this.editor) != null ? ref1.getGrammar() : void 0;
    };

    MarkdownPreviewView.prototype.getDocumentStyleSheets = function() {
      return document.styleSheets;
    };

    MarkdownPreviewView.prototype.getTextEditorStyles = function() {
      var textEditorStyles;
      textEditorStyles = document.createElement("atom-styles");
      textEditorStyles.initialize(atom.styles);
      textEditorStyles.setAttribute("context", "atom-text-editor");
      document.body.appendChild(textEditorStyles);
      return Array.prototype.slice.apply(textEditorStyles.childNodes).map(function(styleElement) {
        return styleElement.innerText;
      });
    };

    MarkdownPreviewView.prototype.getMarkdownPreviewCSS = function() {
      var cssUrlRegExp, i, j, len, len1, markdownPreviewRules, ref1, ref2, ref3, rule, ruleRegExp, stylesheet;
      markdownPreviewRules = [];
      ruleRegExp = /\.markdown-preview/;
      cssUrlRegExp = /url\(atom:\/\/markdown-preview\/assets\/(.*)\)/;
      ref1 = this.getDocumentStyleSheets();
      for (i = 0, len = ref1.length; i < len; i++) {
        stylesheet = ref1[i];
        if (stylesheet.rules != null) {
          ref2 = stylesheet.rules;
          for (j = 0, len1 = ref2.length; j < len1; j++) {
            rule = ref2[j];
            if (((ref3 = rule.selectorText) != null ? ref3.match(ruleRegExp) : void 0) != null) {
              markdownPreviewRules.push(rule.cssText);
            }
          }
        }
      }
      return markdownPreviewRules.concat(this.getTextEditorStyles()).join('\n').replace(/atom-text-editor/g, 'pre.editor-colors').replace(/:host/g, '.host').replace(cssUrlRegExp, function(match, assetsName, offset, string) {
        var assetPath, base64Data, originalData;
        assetPath = path.join(__dirname, '../assets', assetsName);
        originalData = fs.readFileSync(assetPath, 'binary');
        base64Data = new Buffer(originalData, 'binary').toString('base64');
        return "url('data:image/jpeg;base64," + base64Data + "')";
      });
    };

    MarkdownPreviewView.prototype.showError = function(result) {
      var failureMessage, h2, h3;
      this.element.textContent = '';
      h2 = document.createElement('h2');
      h2.textContent = 'Previewing Markdown Failed';
      this.element.appendChild(h2);
      if (failureMessage = result != null ? result.message : void 0) {
        h3 = document.createElement('h3');
        h3.textContent = failureMessage;
        return this.element.appendChild(h3);
      }
    };

    MarkdownPreviewView.prototype.showLoading = function() {
      var div;
      this.loading = true;
      this.element.textContent = '';
      div = document.createElement('div');
      div.classList.add('markdown-spinner');
      div.textContent = 'Loading Markdown\u2026';
      return this.element.appendChild(div);
    };

    MarkdownPreviewView.prototype.copyToClipboard = function() {
      var selectedNode, selectedText, selection;
      if (this.loading) {
        return false;
      }
      selection = window.getSelection();
      selectedText = selection.toString();
      selectedNode = selection.baseNode;
      if (selectedText && (selectedNode != null) && (this.element === selectedNode || this.element.contains(selectedNode))) {
        return false;
      }
      this.getHTML(function(error, html) {
        if (error != null) {
          return console.warn('Copying Markdown as HTML failed', error);
        } else {
          return atom.clipboard.write(html);
        }
      });
      return true;
    };

    MarkdownPreviewView.prototype.saveAs = function() {
      var filePath, htmlFilePath, projectPath, title;
      if (this.loading) {
        return;
      }
      filePath = this.getPath();
      title = 'Markdown to HTML';
      if (filePath) {
        title = path.parse(filePath).name;
        filePath += '.html';
      } else {
        filePath = 'untitled.md.html';
        if (projectPath = atom.project.getPaths()[0]) {
          filePath = path.join(projectPath, filePath);
        }
      }
      if (htmlFilePath = atom.showSaveDialogSync(filePath)) {
        return this.getHTML((function(_this) {
          return function(error, htmlBody) {
            var html;
            if (error != null) {
              return console.warn('Saving Markdown as HTML failed', error);
            } else {
              html = ("<!DOCTYPE html>\n<html>\n  <head>\n      <meta charset=\"utf-8\" />\n      <title>" + title + "</title>\n      <style>" + (_this.getMarkdownPreviewCSS()) + "</style>\n  </head>\n  <body class='markdown-preview' data-use-github-style>" + htmlBody + "</body>\n</html>") + "\n";
              fs.writeFileSync(htmlFilePath, html);
              return atom.workspace.open(htmlFilePath);
            }
          };
        })(this));
      }
    };

    return MarkdownPreviewView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
