'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _eventKit = require('event-kit');

/*
 * Register callbacks and construct Promises to wait for the next occurrence of specific events that occur throughout
 * the data refresh and rendering cycle.
 */
let Switchboard = class Switchboard {
  constructor() {
    this.promises = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  /*
   * Invoke a callback each time that a desired event is observed. Return a Disposable that can be used to
   * unsubscribe from events.
   *
   * In general, you should use the more specific `onDidXyz` methods.
   */
  onDid(eventName, callback) {
    return this.emitter.on(`did-${eventName}`, callback);
  }

  /*
   * Indicate that a named event has been observed, firing any callbacks and resolving any Promises that were created
   * for this event. Optionally provide a payload with more information.
   *
   * In general, you should prefer the more specific `didXyz()` methods.
   */
  did(eventName, payload) {
    this.emitter.emit(`did-${eventName}`, payload);
  }

  /*
   * Retrieve a Promise that will be resolved the next time a desired event is observed.
   *
   * In general, you should prefer the more specific `getXyzPromise()` methods.
   */
  getPromise(eventName) {
    const existing = this.promises.get(eventName);
    if (existing !== undefined) {
      return existing;
    }

    const created = new Promise((resolve, reject) => {
      const subscription = this.onDid(eventName, payload => {
        subscription.dispose();
        this.promises.delete(eventName);
        resolve(payload);
      });
    });
    this.promises.set(eventName, created);
    return created;
  }
};
exports.default = Switchboard;


['UpdateRepository', 'BeginStageOperation', 'FinishStageOperation', 'ChangePatch', 'ScheduleActiveContextUpdate', 'BeginActiveContextUpdate', 'FinishActiveContextUpdate', 'FinishRender', 'FinishContextChangeRender'].forEach(eventName => {
  Switchboard.prototype[`did${eventName}`] = function (payload) {
    this.did(eventName, payload);
  };

  Switchboard.prototype[`get${eventName}Promise`] = function () {
    return this.getPromise(eventName);
  };

  Switchboard.prototype[`onDid${eventName}`] = function (callback) {
    return this.onDid(eventName, callback);
  };
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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