'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
/*
 * Input adapter to facilitate parsing conflicts from text loaded into an Editor.
 */
let EditorAdapter = exports.EditorAdapter = class EditorAdapter {
  constructor(editor, startRow) {
    this.editor = editor;
    this.currentRow = startRow;
  }

  getCurrentRow() {
    return this.currentRow;
  }

  getCurrentLine() {
    return this.editor.lineTextForBufferRow(this.currentRow);
  }

  advanceRow() {
    this.currentRow++;
  }

  isAtEnd() {
    return this.currentRow > this.editor.getLastBufferRow();
  }
};

/*
 * Input adapter for parsing conflicts from a chunk of text arriving from a ReadStream.
 */

let ChunkAdapter = exports.ChunkAdapter = class ChunkAdapter {
  constructor(chunk) {
    this.chunk = chunk;

    this.lineEndRx = /\r?\n/g;

    this.lineStartPosition = 0;
    this.eof = false;
    this.advanceRow();
  }

  advanceTo(pattern) {
    if (this.eof) {
      return false;
    }

    const rx = new RegExp(pattern.source, 'gm');
    rx.lastIndex = this.lineStartPosition;

    const match = rx.exec(this.chunk);
    if (match) {
      this.lineEndRx.lastIndex = match.index;
      return true;
    } else {
      return false;
    }
  }

  getCurrentRow() {
    return undefined;
  }

  getCurrentLine() {
    return this.currentLine;
  }

  advanceRow() {
    this.lineStartPosition = this.lineEndRx.lastIndex;
    if (this.lineEndRx.test(this.chunk)) {
      this.currentLine = this.chunk.slice(this.lineStartPosition, this.lineEndRx.lastIndex);
    } else {
      this.currentLine = this.chunk.slice(this.lineStartPosition);
      this.eof = true;
    }
  }

  isAtEnd() {
    return this.eof;
  }

  getLastPartialMarker() {
    const match = /[<|>=]{1,7} ?[^\r\n]*\r?$/.exec(this.chunk);
    return match ? match[0] : '';
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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