(function() {
  var CompositeDisposable, Range, SelectNext, _, ref;

  _ = require('underscore-plus');

  ref = require('atom'), CompositeDisposable = ref.CompositeDisposable, Range = ref.Range;

  module.exports = SelectNext = (function() {
    SelectNext.prototype.selectionRanges = null;

    function SelectNext(editor) {
      this.editor = editor;
      this.selectionRanges = [];
    }

    SelectNext.prototype.findAndSelectNext = function() {
      if (this.editor.getLastSelection().isEmpty()) {
        return this.selectWord();
      } else {
        return this.selectNextOccurrence();
      }
    };

    SelectNext.prototype.findAndSelectAll = function() {
      if (this.editor.getLastSelection().isEmpty()) {
        this.selectWord();
      }
      return this.selectAllOccurrences();
    };

    SelectNext.prototype.undoLastSelection = function() {
      this.updateSavedSelections();
      if (this.selectionRanges.length < 1) {
        return;
      }
      if (this.selectionRanges.length > 1) {
        this.selectionRanges.pop();
        this.editor.setSelectedBufferRanges(this.selectionRanges);
      } else {
        this.editor.clearSelections();
      }
      return this.editor.scrollToCursorPosition();
    };

    SelectNext.prototype.skipCurrentSelection = function() {
      var lastSelection;
      this.updateSavedSelections();
      if (this.selectionRanges.length < 1) {
        return;
      }
      if (this.selectionRanges.length > 1) {
        lastSelection = this.selectionRanges.pop();
        this.editor.setSelectedBufferRanges(this.selectionRanges);
        return this.selectNextOccurrence({
          start: lastSelection.end
        });
      } else {
        this.selectNextOccurrence();
        this.selectionRanges.shift();
        if (this.selectionRanges.length < 1) {
          return;
        }
        return this.editor.setSelectedBufferRanges(this.selectionRanges);
      }
    };

    SelectNext.prototype.selectWord = function() {
      var clearWordSelected, disposables, lastSelection;
      this.editor.selectWordsContainingCursors();
      lastSelection = this.editor.getLastSelection();
      if (this.wordSelected = this.isWordSelected(lastSelection)) {
        disposables = new CompositeDisposable;
        clearWordSelected = (function(_this) {
          return function() {
            _this.wordSelected = null;
            return disposables.dispose();
          };
        })(this);
        disposables.add(lastSelection.onDidChangeRange(clearWordSelected));
        return disposables.add(lastSelection.onDidDestroy(clearWordSelected));
      }
    };

    SelectNext.prototype.selectAllOccurrences = function() {
      var range;
      range = [[0, 0], this.editor.getEofBufferPosition()];
      return this.scanForNextOccurrence(range, (function(_this) {
        return function(arg) {
          var range, stop;
          range = arg.range, stop = arg.stop;
          return _this.addSelection(range);
        };
      })(this));
    };

    SelectNext.prototype.selectNextOccurrence = function(options) {
      var range, ref1, startingRange;
      if (options == null) {
        options = {};
      }
      startingRange = (ref1 = options.start) != null ? ref1 : this.editor.getSelectedBufferRange().end;
      range = this.findNextOccurrence([startingRange, this.editor.getEofBufferPosition()]);
      if (range == null) {
        range = this.findNextOccurrence([[0, 0], this.editor.getSelections()[0].getBufferRange().start]);
      }
      if (range != null) {
        return this.addSelection(range);
      }
    };

    SelectNext.prototype.findNextOccurrence = function(scanRange) {
      var foundRange;
      foundRange = null;
      this.scanForNextOccurrence(scanRange, function(arg) {
        var range, stop;
        range = arg.range, stop = arg.stop;
        foundRange = range;
        return stop();
      });
      return foundRange;
    };

    SelectNext.prototype.addSelection = function(range) {
      var selection;
      selection = this.editor.addSelectionForBufferRange(range);
      return this.updateSavedSelections(selection);
    };

    SelectNext.prototype.scanForNextOccurrence = function(range, callback) {
      var nonWordCharacters, selection, text;
      selection = this.editor.getLastSelection();
      text = _.escapeRegExp(selection.getText());
      if (this.wordSelected) {
        nonWordCharacters = atom.config.get('editor.nonWordCharacters');
        text = "(^|[ \t" + (_.escapeRegExp(nonWordCharacters)) + "]+)" + text + "(?=$|[\\s" + (_.escapeRegExp(nonWordCharacters)) + "]+)";
      }
      return this.editor.scanInBufferRange(new RegExp(text, 'g'), range, function(result) {
        var prefix;
        if (prefix = result.match[1]) {
          result.range = result.range.translate([0, prefix.length], [0, 0]);
        }
        return callback(result);
      });
    };

    SelectNext.prototype.updateSavedSelections = function(selection) {
      var i, len, results, s, selectionRange, selections;
      if (selection == null) {
        selection = null;
      }
      selections = this.editor.getSelections();
      if (selections.length < 3) {
        this.selectionRanges = [];
      }
      if (this.selectionRanges.length === 0) {
        results = [];
        for (i = 0, len = selections.length; i < len; i++) {
          s = selections[i];
          results.push(this.selectionRanges.push(s.getBufferRange()));
        }
        return results;
      } else if (selection) {
        selectionRange = selection.getBufferRange();
        if (this.selectionRanges.some(function(existingRange) {
          return existingRange.isEqual(selectionRange);
        })) {
          return;
        }
        return this.selectionRanges.push(selectionRange);
      }
    };

    SelectNext.prototype.isNonWordCharacter = function(character) {
      var nonWordCharacters;
      nonWordCharacters = atom.config.get('editor.nonWordCharacters');
      return new RegExp("[ \t" + (_.escapeRegExp(nonWordCharacters)) + "]").test(character);
    };

    SelectNext.prototype.isNonWordCharacterToTheLeft = function(selection) {
      var range, selectionStart;
      selectionStart = selection.getBufferRange().start;
      range = Range.fromPointWithDelta(selectionStart, 0, -1);
      return this.isNonWordCharacter(this.editor.getTextInBufferRange(range));
    };

    SelectNext.prototype.isNonWordCharacterToTheRight = function(selection) {
      var range, selectionEnd;
      selectionEnd = selection.getBufferRange().end;
      range = Range.fromPointWithDelta(selectionEnd, 0, 1);
      return this.isNonWordCharacter(this.editor.getTextInBufferRange(range));
    };

    SelectNext.prototype.isWordSelected = function(selection) {
      var containsOnlyWordCharacters, lineRange, nonWordCharacterToTheLeft, nonWordCharacterToTheRight, selectionRange;
      if (selection.getBufferRange().isSingleLine()) {
        selectionRange = selection.getBufferRange();
        lineRange = this.editor.bufferRangeForBufferRow(selectionRange.start.row);
        nonWordCharacterToTheLeft = _.isEqual(selectionRange.start, lineRange.start) || this.isNonWordCharacterToTheLeft(selection);
        nonWordCharacterToTheRight = _.isEqual(selectionRange.end, lineRange.end) || this.isNonWordCharacterToTheRight(selection);
        containsOnlyWordCharacters = !this.isNonWordCharacter(selection.getText());
        return nonWordCharacterToTheLeft && nonWordCharacterToTheRight && containsOnlyWordCharacters;
      } else {
        return false;
      }
    };

    return SelectNext;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
