(function() {
  var BracketMatcher, CompositeDisposable, SelectorCache, _,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  CompositeDisposable = require('atom').CompositeDisposable;

  SelectorCache = require('./selector-cache');

  module.exports = BracketMatcher = (function() {
    function BracketMatcher(editor, editorElement, matchManager) {
      this.editor = editor;
      this.matchManager = matchManager;
      this.backspace = bind(this.backspace, this);
      this.insertNewline = bind(this.insertNewline, this);
      this.insertText = bind(this.insertText, this);
      this.subscriptions = new CompositeDisposable;
      this.bracketMarkers = [];
      _.adviseBefore(this.editor, 'insertText', this.insertText);
      _.adviseBefore(this.editor, 'insertNewline', this.insertNewline);
      _.adviseBefore(this.editor, 'backspace', this.backspace);
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:remove-brackets-from-selection', (function(_this) {
        return function(event) {
          if (!_this.removeBrackets()) {
            return event.abortKeyBinding();
          }
        };
      })(this)));
      this.subscriptions.add(this.editor.onDidDestroy((function(_this) {
        return function() {
          return _this.unsubscribe();
        };
      })(this)));
    }

    BracketMatcher.prototype.insertText = function(text, options) {
      var autoCompleteOpeningBracket, bracketMarker, cursorBufferPosition, hasEscapeCharacterBeforeCursor, hasEscapeSequenceBeforeCursor, hasQuoteBeforeCursor, hasWordAfterCursor, hasWordBeforeCursor, nextCharacter, pair, previousCharacter, previousCharacters, range, ref, ref1, ref2, skipOverExistingClosingBracket;
      if (!text) {
        return true;
      }
      if ((options != null ? options.select : void 0) || (options != null ? options.undo : void 0) === 'skip') {
        return true;
      }
      if (this.wrapSelectionInBrackets(text)) {
        return false;
      }
      if (this.editor.hasMultipleCursors()) {
        return true;
      }
      cursorBufferPosition = this.editor.getCursorBufferPosition();
      previousCharacters = this.editor.getTextInBufferRange([[cursorBufferPosition.row, 0], cursorBufferPosition]);
      nextCharacter = this.editor.getTextInBufferRange([cursorBufferPosition, cursorBufferPosition.traverse([0, 1])]);
      previousCharacter = previousCharacters.slice(-1);
      hasWordAfterCursor = /\w/.test(nextCharacter);
      hasWordBeforeCursor = /\w/.test(previousCharacter);
      hasQuoteBeforeCursor = this.isQuote(previousCharacter) && previousCharacter === text[0];
      hasEscapeCharacterBeforeCursor = ((ref = previousCharacters.match(/(\\+)$/)) != null ? ref[1].length : void 0) % 2 === 1;
      hasEscapeSequenceBeforeCursor = ((ref1 = previousCharacters.match(/(\\+)[^\\]$/)) != null ? ref1[1].length : void 0) % 2 === 1 || ((ref2 = previousCharacters.match(/(\\+)$/)) != null ? ref2[1].length : void 0) % 2 === 0;
      if (text === '#' && this.isCursorOnInterpolatedString()) {
        autoCompleteOpeningBracket = this.getScopedSetting('bracket-matcher.autocompleteBrackets') && !hasEscapeCharacterBeforeCursor;
        text += '{';
        pair = '}';
      } else {
        autoCompleteOpeningBracket = this.isOpeningBracket(text) && !hasWordAfterCursor && this.getScopedSetting('bracket-matcher.autocompleteBrackets') && !(this.isQuote(text) && (hasWordBeforeCursor || hasQuoteBeforeCursor || hasEscapeSequenceBeforeCursor)) && !hasEscapeCharacterBeforeCursor;
        pair = this.matchManager.pairedCharacters[text];
      }
      skipOverExistingClosingBracket = false;
      if (this.isClosingBracket(text) && nextCharacter === text && !hasEscapeCharacterBeforeCursor) {
        if (bracketMarker = _.find(this.bracketMarkers, function(marker) {
          return marker.isValid() && marker.getBufferRange().end.isEqual(cursorBufferPosition);
        })) {
          skipOverExistingClosingBracket = true;
        }
      }
      if (skipOverExistingClosingBracket) {
        bracketMarker.destroy();
        _.remove(this.bracketMarkers, bracketMarker);
        this.editor.moveRight();
        return false;
      } else if (autoCompleteOpeningBracket) {
        this.editor.insertText(text + pair);
        this.editor.moveLeft();
        range = [cursorBufferPosition, cursorBufferPosition.traverse([0, text.length])];
        this.bracketMarkers.push(this.editor.markBufferRange(range));
        return false;
      }
    };

    BracketMatcher.prototype.insertNewline = function() {
      var cursorBufferPosition, hasEscapeCharacterBeforeCursor, nextCharacter, previousCharacter, previousCharacters, ref;
      if (this.editor.hasMultipleCursors()) {
        return;
      }
      if (!this.editor.getLastSelection().isEmpty()) {
        return;
      }
      cursorBufferPosition = this.editor.getCursorBufferPosition();
      previousCharacters = this.editor.getTextInBufferRange([[cursorBufferPosition.row, 0], cursorBufferPosition]);
      nextCharacter = this.editor.getTextInBufferRange([cursorBufferPosition, cursorBufferPosition.traverse([0, 1])]);
      previousCharacter = previousCharacters.slice(-1);
      hasEscapeCharacterBeforeCursor = ((ref = previousCharacters.match(/(\\+)$/)) != null ? ref[1].length : void 0) % 2 === 1;
      if (this.matchManager.pairsWithExtraNewline[previousCharacter] === nextCharacter && !hasEscapeCharacterBeforeCursor) {
        this.editor.transact((function(_this) {
          return function() {
            var cursorRow;
            _this.editor.insertText("\n\n");
            _this.editor.moveUp();
            if (_this.getScopedSetting('editor.autoIndent')) {
              cursorRow = _this.editor.getCursorBufferPosition().row;
              return _this.editor.autoIndentBufferRows(cursorRow, cursorRow + 1);
            }
          };
        })(this));
        return false;
      }
    };

    BracketMatcher.prototype.backspace = function() {
      var cursorBufferPosition, hasEscapeCharacterBeforeCursor, nextCharacter, previousCharacter, previousCharacters, ref;
      if (this.editor.hasMultipleCursors()) {
        return;
      }
      if (!this.editor.getLastSelection().isEmpty()) {
        return;
      }
      cursorBufferPosition = this.editor.getCursorBufferPosition();
      previousCharacters = this.editor.getTextInBufferRange([[cursorBufferPosition.row, 0], cursorBufferPosition]);
      nextCharacter = this.editor.getTextInBufferRange([cursorBufferPosition, cursorBufferPosition.traverse([0, 1])]);
      previousCharacter = previousCharacters.slice(-1);
      hasEscapeCharacterBeforeCursor = ((ref = previousCharacters.match(/(\\+)$/)) != null ? ref[1].length : void 0) % 2 === 1;
      if ((this.matchManager.pairedCharacters[previousCharacter] === nextCharacter) && !hasEscapeCharacterBeforeCursor && this.getScopedSetting('bracket-matcher.autocompleteBrackets')) {
        this.editor.transact((function(_this) {
          return function() {
            _this.editor.moveLeft();
            _this.editor["delete"]();
            return _this.editor["delete"]();
          };
        })(this));
        return false;
      }
    };

    BracketMatcher.prototype.removeBrackets = function() {
      var bracketsRemoved;
      bracketsRemoved = false;
      this.editor.mutateSelectedText((function(_this) {
        return function(selection) {
          var options, range, selectionEnd, selectionStart, text;
          if (!_this.selectionIsWrappedByMatchingBrackets(selection)) {
            return;
          }
          range = selection.getBufferRange();
          options = {
            reversed: selection.isReversed()
          };
          selectionStart = range.start;
          if (range.start.row === range.end.row) {
            selectionEnd = range.end.traverse([0, -2]);
          } else {
            selectionEnd = range.end.traverse([0, -1]);
          }
          text = selection.getText();
          selection.insertText(text.substring(1, text.length - 1));
          selection.setBufferRange([selectionStart, selectionEnd], options);
          return bracketsRemoved = true;
        };
      })(this));
      return bracketsRemoved;
    };

    BracketMatcher.prototype.wrapSelectionInBrackets = function(bracket) {
      var pair, selectionWrapped;
      if (bracket === '#') {
        if (!this.isCursorOnInterpolatedString()) {
          return false;
        }
        bracket = '#{';
        pair = '}';
      } else {
        if (!this.isOpeningBracket(bracket)) {
          return false;
        }
        pair = this.matchManager.pairedCharacters[bracket];
      }
      if (!this.editor.selections.some(function(s) {
        return !s.isEmpty();
      })) {
        return false;
      }
      if (!this.getScopedSetting('bracket-matcher.wrapSelectionsInBrackets')) {
        return false;
      }
      selectionWrapped = false;
      this.editor.mutateSelectedText(function(selection) {
        var options, range, selectionEnd, selectionStart;
        if (selection.isEmpty()) {
          return;
        }
        if (bracket === '#{' && !selection.getBufferRange().isSingleLine()) {
          return;
        }
        selectionWrapped = true;
        range = selection.getBufferRange();
        options = {
          reversed: selection.isReversed()
        };
        selection.insertText("" + bracket + (selection.getText()) + pair);
        selectionStart = range.start.traverse([0, bracket.length]);
        if (range.start.row === range.end.row) {
          selectionEnd = range.end.traverse([0, bracket.length]);
        } else {
          selectionEnd = range.end;
        }
        return selection.setBufferRange([selectionStart, selectionEnd], options);
      });
      return selectionWrapped;
    };

    BracketMatcher.prototype.isQuote = function(string) {
      return /['"`]/.test(string);
    };

    BracketMatcher.prototype.isCursorOnInterpolatedString = function() {
      var segments;
      if (this.interpolatedStringSelector == null) {
        segments = ['*.*.*.interpolated.ruby', 'string.interpolated.ruby', 'string.regexp.interpolated.ruby', 'string.quoted.double.coffee', 'string.unquoted.heredoc.ruby', 'string.quoted.double.livescript', 'string.quoted.double.heredoc.livescript', 'string.quoted.double.elixir', 'string.quoted.double.heredoc.elixir', 'comment.documentation.heredoc.elixir'];
        this.interpolatedStringSelector = SelectorCache.get(segments.join(' | '));
      }
      return this.interpolatedStringSelector.matches(this.editor.getLastCursor().getScopeDescriptor().getScopesArray());
    };

    BracketMatcher.prototype.isOpeningBracket = function(string) {
      return this.matchManager.pairedCharacters.hasOwnProperty(string);
    };

    BracketMatcher.prototype.isClosingBracket = function(string) {
      return this.matchManager.pairedCharactersInverse.hasOwnProperty(string);
    };

    BracketMatcher.prototype.selectionIsWrappedByMatchingBrackets = function(selection) {
      var firstCharacter, lastCharacter, selectedText;
      if (selection.isEmpty()) {
        return false;
      }
      selectedText = selection.getText();
      firstCharacter = selectedText[0];
      lastCharacter = selectedText[selectedText.length - 1];
      return this.matchManager.pairedCharacters[firstCharacter] === lastCharacter;
    };

    BracketMatcher.prototype.unsubscribe = function() {
      return this.subscriptions.dispose();
    };

    BracketMatcher.prototype.getScopedSetting = function(key) {
      return atom.config.get(key, {
        scope: this.editor.getRootScopeDescriptor()
      });
    };

    return BracketMatcher;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
