(function() {
  var CompositeDisposable, Disposable, WindowEventHandler, listen, ref,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  ref = require('event-kit'), Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable;

  listen = require('./delegated-listener');

  module.exports = WindowEventHandler = (function() {
    function WindowEventHandler(arg) {
      this.atomEnvironment = arg.atomEnvironment, this.applicationDelegate = arg.applicationDelegate;
      this.handleDocumentContextmenu = bind(this.handleDocumentContextmenu, this);
      this.handleLinkClick = bind(this.handleLinkClick, this);
      this.handleWindowToggleMenuBar = bind(this.handleWindowToggleMenuBar, this);
      this.handleWindowToggleDevTools = bind(this.handleWindowToggleDevTools, this);
      this.handleWindowReload = bind(this.handleWindowReload, this);
      this.handleWindowClose = bind(this.handleWindowClose, this);
      this.handleWindowToggleFullScreen = bind(this.handleWindowToggleFullScreen, this);
      this.handleWindowBeforeunload = bind(this.handleWindowBeforeunload, this);
      this.handleLeaveFullScreen = bind(this.handleLeaveFullScreen, this);
      this.handleEnterFullScreen = bind(this.handleEnterFullScreen, this);
      this.handleWindowBlur = bind(this.handleWindowBlur, this);
      this.handleFocusPrevious = bind(this.handleFocusPrevious, this);
      this.handleFocusNext = bind(this.handleFocusNext, this);
      this.handleDocumentKeyEvent = bind(this.handleDocumentKeyEvent, this);
      this.reloadRequested = false;
      this.subscriptions = new CompositeDisposable;
      this.handleNativeKeybindings();
    }

    WindowEventHandler.prototype.initialize = function(window, document) {
      var ref1;
      this.window = window;
      this.document = document;
      this.subscriptions.add(this.atomEnvironment.commands.add(this.window, {
        'window:toggle-full-screen': this.handleWindowToggleFullScreen,
        'window:close': this.handleWindowClose,
        'window:reload': this.handleWindowReload,
        'window:toggle-dev-tools': this.handleWindowToggleDevTools
      }));
      if ((ref1 = process.platform) === 'win32' || ref1 === 'linux') {
        this.subscriptions.add(this.atomEnvironment.commands.add(this.window, {
          'window:toggle-menu-bar': this.handleWindowToggleMenuBar
        }));
      }
      this.subscriptions.add(this.atomEnvironment.commands.add(this.document, {
        'core:focus-next': this.handleFocusNext,
        'core:focus-previous': this.handleFocusPrevious
      }));
      this.addEventListener(this.window, 'beforeunload', this.handleWindowBeforeunload);
      this.addEventListener(this.window, 'focus', this.handleWindowFocus);
      this.addEventListener(this.window, 'blur', this.handleWindowBlur);
      this.addEventListener(this.document, 'keyup', this.handleDocumentKeyEvent);
      this.addEventListener(this.document, 'keydown', this.handleDocumentKeyEvent);
      this.addEventListener(this.document, 'drop', this.handleDocumentDrop);
      this.addEventListener(this.document, 'dragover', this.handleDocumentDragover);
      this.addEventListener(this.document, 'contextmenu', this.handleDocumentContextmenu);
      this.subscriptions.add(listen(this.document, 'click', 'a', this.handleLinkClick));
      this.subscriptions.add(listen(this.document, 'submit', 'form', this.handleFormSubmit));
      this.subscriptions.add(this.applicationDelegate.onDidEnterFullScreen(this.handleEnterFullScreen));
      return this.subscriptions.add(this.applicationDelegate.onDidLeaveFullScreen(this.handleLeaveFullScreen));
    };

    WindowEventHandler.prototype.handleNativeKeybindings = function() {
      var bindCommandToAction;
      bindCommandToAction = (function(_this) {
        return function(command, action) {
          return _this.subscriptions.add(_this.atomEnvironment.commands.add('.native-key-bindings', command, (function(event) {
            return _this.applicationDelegate.getCurrentWindow().webContents[action]();
          }), false));
        };
      })(this);
      bindCommandToAction('core:copy', 'copy');
      bindCommandToAction('core:paste', 'paste');
      bindCommandToAction('core:undo', 'undo');
      bindCommandToAction('core:redo', 'redo');
      bindCommandToAction('core:select-all', 'selectAll');
      return bindCommandToAction('core:cut', 'cut');
    };

    WindowEventHandler.prototype.unsubscribe = function() {
      return this.subscriptions.dispose();
    };

    WindowEventHandler.prototype.on = function(target, eventName, handler) {
      target.on(eventName, handler);
      return this.subscriptions.add(new Disposable(function() {
        return target.removeListener(eventName, handler);
      }));
    };

    WindowEventHandler.prototype.addEventListener = function(target, eventName, handler) {
      target.addEventListener(eventName, handler);
      return this.subscriptions.add(new Disposable(function() {
        return target.removeEventListener(eventName, handler);
      }));
    };

    WindowEventHandler.prototype.handleDocumentKeyEvent = function(event) {
      this.atomEnvironment.keymaps.handleKeyboardEvent(event);
      return event.stopImmediatePropagation();
    };

    WindowEventHandler.prototype.handleDrop = function(event) {
      event.preventDefault();
      return event.stopPropagation();
    };

    WindowEventHandler.prototype.handleDragover = function(event) {
      event.preventDefault();
      event.stopPropagation();
      return event.dataTransfer.dropEffect = 'none';
    };

    WindowEventHandler.prototype.eachTabIndexedElement = function(callback) {
      var element, i, len, ref1;
      ref1 = this.document.querySelectorAll('[tabindex]');
      for (i = 0, len = ref1.length; i < len; i++) {
        element = ref1[i];
        if (element.disabled) {
          continue;
        }
        if (!(element.tabIndex >= 0)) {
          continue;
        }
        callback(element, element.tabIndex);
      }
    };

    WindowEventHandler.prototype.handleFocusNext = function() {
      var focusedTabIndex, lowestElement, lowestTabIndex, nextElement, nextTabIndex, ref1;
      focusedTabIndex = (ref1 = this.document.activeElement.tabIndex) != null ? ref1 : -2e308;
      nextElement = null;
      nextTabIndex = 2e308;
      lowestElement = null;
      lowestTabIndex = 2e308;
      this.eachTabIndexedElement(function(element, tabIndex) {
        if (tabIndex < lowestTabIndex) {
          lowestTabIndex = tabIndex;
          lowestElement = element;
        }
        if ((focusedTabIndex < tabIndex && tabIndex < nextTabIndex)) {
          nextTabIndex = tabIndex;
          return nextElement = element;
        }
      });
      if (nextElement != null) {
        return nextElement.focus();
      } else if (lowestElement != null) {
        return lowestElement.focus();
      }
    };

    WindowEventHandler.prototype.handleFocusPrevious = function() {
      var focusedTabIndex, highestElement, highestTabIndex, previousElement, previousTabIndex, ref1;
      focusedTabIndex = (ref1 = this.document.activeElement.tabIndex) != null ? ref1 : 2e308;
      previousElement = null;
      previousTabIndex = -2e308;
      highestElement = null;
      highestTabIndex = -2e308;
      this.eachTabIndexedElement(function(element, tabIndex) {
        if (tabIndex > highestTabIndex) {
          highestTabIndex = tabIndex;
          highestElement = element;
        }
        if ((focusedTabIndex > tabIndex && tabIndex > previousTabIndex)) {
          previousTabIndex = tabIndex;
          return previousElement = element;
        }
      });
      if (previousElement != null) {
        return previousElement.focus();
      } else if (highestElement != null) {
        return highestElement.focus();
      }
    };

    WindowEventHandler.prototype.handleWindowFocus = function() {
      return this.document.body.classList.remove('is-blurred');
    };

    WindowEventHandler.prototype.handleWindowBlur = function() {
      this.document.body.classList.add('is-blurred');
      return this.atomEnvironment.storeWindowDimensions();
    };

    WindowEventHandler.prototype.handleEnterFullScreen = function() {
      return this.document.body.classList.add("fullscreen");
    };

    WindowEventHandler.prototype.handleLeaveFullScreen = function() {
      return this.document.body.classList.remove("fullscreen");
    };

    WindowEventHandler.prototype.handleWindowBeforeunload = function(event) {
      if (!this.reloadRequested && !this.atomEnvironment.inSpecMode() && this.atomEnvironment.getCurrentWindow().isWebViewFocused()) {
        this.atomEnvironment.hide();
      }
      this.reloadRequested = false;
      this.atomEnvironment.storeWindowDimensions();
      this.atomEnvironment.unloadEditorWindow();
      return this.atomEnvironment.destroy();
    };

    WindowEventHandler.prototype.handleWindowToggleFullScreen = function() {
      return this.atomEnvironment.toggleFullScreen();
    };

    WindowEventHandler.prototype.handleWindowClose = function() {
      return this.atomEnvironment.close();
    };

    WindowEventHandler.prototype.handleWindowReload = function() {
      this.reloadRequested = true;
      return this.atomEnvironment.reload();
    };

    WindowEventHandler.prototype.handleWindowToggleDevTools = function() {
      return this.atomEnvironment.toggleDevTools();
    };

    WindowEventHandler.prototype.handleWindowToggleMenuBar = function() {
      var detail;
      this.atomEnvironment.config.set('core.autoHideMenuBar', !this.atomEnvironment.config.get('core.autoHideMenuBar'));
      if (this.atomEnvironment.config.get('core.autoHideMenuBar')) {
        detail = "To toggle, press the Alt key or execute the window:toggle-menu-bar command";
        return this.atomEnvironment.notifications.addInfo('Menu bar hidden', {
          detail: detail
        });
      }
    };

    WindowEventHandler.prototype.handleLinkClick = function(event) {
      var ref1, uri;
      event.preventDefault();
      uri = (ref1 = event.currentTarget) != null ? ref1.getAttribute('href') : void 0;
      if (uri && uri[0] !== '#' && /^https?:\/\//.test(uri)) {
        return this.applicationDelegate.openExternal(uri);
      }
    };

    WindowEventHandler.prototype.handleFormSubmit = function(event) {
      return event.preventDefault();
    };

    WindowEventHandler.prototype.handleDocumentContextmenu = function(event) {
      event.preventDefault();
      return this.atomEnvironment.contextMenu.showForEvent(event);
    };

    return WindowEventHandler;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
