(function() {
  var CSON, Disposable, MenuHelpers, MenuManager, _, fs, ipcRenderer, path, platformMenu, ref, ref1;

  path = require('path');

  _ = require('underscore-plus');

  ipcRenderer = require('electron').ipcRenderer;

  CSON = require('season');

  fs = require('fs-plus');

  Disposable = require('event-kit').Disposable;

  MenuHelpers = require('./menu-helpers');

  platformMenu = (ref = require('../package.json')) != null ? (ref1 = ref._atomMenu) != null ? ref1.menu : void 0 : void 0;

  module.exports = MenuManager = (function() {
    function MenuManager(arg) {
      this.resourcePath = arg.resourcePath, this.keymapManager = arg.keymapManager, this.packageManager = arg.packageManager;
      this.initialized = false;
      this.pendingUpdateOperation = null;
      this.template = [];
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
      this.packageManager.onDidActivateInitialPackages((function(_this) {
        return function() {
          return _this.sortPackagesMenu();
        };
      })(this));
    }

    MenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath;
      this.keymapManager.onDidReloadKeymap((function(_this) {
        return function() {
          return _this.update();
        };
      })(this));
      this.update();
      return this.initialized = true;
    };

    MenuManager.prototype.add = function(items) {
      var i, item, len;
      items = _.deepClone(items);
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        this.merge(this.template, item);
      }
      this.update();
      return new Disposable((function(_this) {
        return function() {
          return _this.remove(items);
        };
      })(this));
    };

    MenuManager.prototype.remove = function(items) {
      var i, item, len;
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        this.unmerge(this.template, item);
      }
      return this.update();
    };

    MenuManager.prototype.clear = function() {
      this.template = [];
      return this.update();
    };

    MenuManager.prototype.includeSelector = function(selector) {
      var element, error, ref2, ref3, testBody, testDocument, testWorkspace, workspaceClasses;
      try {
        if (document.body.webkitMatchesSelector(selector)) {
          return true;
        }
      } catch (error1) {
        error = error1;
        return false;
      }
      if (this.testEditor == null) {
        testDocument = document.implementation.createDocument(document.namespaceURI, 'html');
        testBody = testDocument.createElement('body');
        (ref2 = testBody.classList).add.apply(ref2, this.classesForElement(document.body));
        testWorkspace = testDocument.createElement('atom-workspace');
        workspaceClasses = this.classesForElement(document.body.querySelector('atom-workspace'));
        if (workspaceClasses.length === 0) {
          workspaceClasses = ['workspace'];
        }
        (ref3 = testWorkspace.classList).add.apply(ref3, workspaceClasses);
        testBody.appendChild(testWorkspace);
        this.testEditor = testDocument.createElement('atom-text-editor');
        this.testEditor.classList.add('editor');
        testWorkspace.appendChild(this.testEditor);
      }
      element = this.testEditor;
      while (element) {
        if (element.webkitMatchesSelector(selector)) {
          return true;
        }
        element = element.parentElement;
      }
      return false;
    };

    MenuManager.prototype.update = function() {
      if (!this.initialized) {
        return;
      }
      if (this.pendingUpdateOperation != null) {
        clearImmediate(this.pendingUpdateOperation);
      }
      return this.pendingUpdateOperation = setImmediate((function(_this) {
        return function() {
          var binding, i, j, keystrokesByCommand, len, len1, name, ref2, ref3, unsetKeystrokes;
          unsetKeystrokes = new Set;
          ref2 = _this.keymapManager.getKeyBindings();
          for (i = 0, len = ref2.length; i < len; i++) {
            binding = ref2[i];
            if (binding.command === 'unset!') {
              unsetKeystrokes.add(binding.keystrokes);
            }
          }
          keystrokesByCommand = {};
          ref3 = _this.keymapManager.getKeyBindings();
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            binding = ref3[j];
            if (!_this.includeSelector(binding.selector)) {
              continue;
            }
            if (unsetKeystrokes.has(binding.keystrokes)) {
              continue;
            }
            if (binding.keystrokes.includes(' ')) {
              continue;
            }
            if (process.platform === 'darwin' && /^alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (process.platform === 'win32' && /^ctrl-alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (keystrokesByCommand[name = binding.command] == null) {
              keystrokesByCommand[name] = [];
            }
            keystrokesByCommand[binding.command].unshift(binding.keystrokes);
          }
          return _this.sendToBrowserProcess(_this.template, keystrokesByCommand);
        };
      })(this));
    };

    MenuManager.prototype.loadPlatformItems = function() {
      var menu, menusDirPath, platformMenuPath;
      if (platformMenu != null) {
        return this.add(platformMenu);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        menu = CSON.readFileSync(platformMenuPath).menu;
        return this.add(menu);
      }
    };

    MenuManager.prototype.merge = function(menu, item) {
      return MenuHelpers.merge(menu, item);
    };

    MenuManager.prototype.unmerge = function(menu, item) {
      return MenuHelpers.unmerge(menu, item);
    };

    MenuManager.prototype.sendToBrowserProcess = function(template, keystrokesByCommand) {
      return ipcRenderer.send('update-application-menu', template, keystrokesByCommand);
    };

    MenuManager.prototype.classesForElement = function(element) {
      var classList;
      if (classList = element != null ? element.classList : void 0) {
        return Array.prototype.slice.apply(classList);
      } else {
        return [];
      }
    };

    MenuManager.prototype.sortPackagesMenu = function() {
      var packagesMenu;
      packagesMenu = _.find(this.template, function(arg) {
        var label;
        label = arg.label;
        return MenuHelpers.normalizeLabel(label) === 'Packages';
      });
      if ((packagesMenu != null ? packagesMenu.submenu : void 0) == null) {
        return;
      }
      packagesMenu.submenu.sort(function(item1, item2) {
        if (item1.label && item2.label) {
          return MenuHelpers.normalizeLabel(item1.label).localeCompare(MenuHelpers.normalizeLabel(item2.label));
        } else {
          return 0;
        }
      });
      return this.update();
    };

    return MenuManager;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiL2hvbWUvdHJhdmlzL2J1aWxkL2F0b20vYXRvbS9vdXQvYXBwL3NyYy9tZW51LW1hbmFnZXIuY29mZmVlIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBO0FBQUEsTUFBQTs7RUFBQSxJQUFBLEdBQU8sT0FBQSxDQUFRLE1BQVI7O0VBRVAsQ0FBQSxHQUFJLE9BQUEsQ0FBUSxpQkFBUjs7RUFDSCxjQUFlLE9BQUEsQ0FBUSxVQUFSOztFQUNoQixJQUFBLEdBQU8sT0FBQSxDQUFRLFFBQVI7O0VBQ1AsRUFBQSxHQUFLLE9BQUEsQ0FBUSxTQUFSOztFQUNKLGFBQWMsT0FBQSxDQUFRLFdBQVI7O0VBRWYsV0FBQSxHQUFjLE9BQUEsQ0FBUSxnQkFBUjs7RUFFZCxZQUFBLHFGQUFvRCxDQUFFOztFQWlEdEQsTUFBTSxDQUFDLE9BQVAsR0FDTTtJQUNTLHFCQUFDLEdBQUQ7TUFBRSxJQUFDLENBQUEsbUJBQUEsY0FBYyxJQUFDLENBQUEsb0JBQUEsZUFBZSxJQUFDLENBQUEscUJBQUE7TUFDN0MsSUFBQyxDQUFBLFdBQUQsR0FBZTtNQUNmLElBQUMsQ0FBQSxzQkFBRCxHQUEwQjtNQUMxQixJQUFDLENBQUEsUUFBRCxHQUFZO01BQ1osSUFBQyxDQUFBLGFBQWEsQ0FBQyx1QkFBZixDQUF1QyxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUE7aUJBQUcsS0FBQyxDQUFBLGlCQUFELENBQUE7UUFBSDtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBdkM7TUFDQSxJQUFDLENBQUEsY0FBYyxDQUFDLDRCQUFoQixDQUE2QyxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUE7aUJBQUcsS0FBQyxDQUFBLGdCQUFELENBQUE7UUFBSDtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBN0M7SUFMVzs7MEJBT2IsVUFBQSxHQUFZLFNBQUMsR0FBRDtNQUFFLElBQUMsQ0FBQSxlQUFGLElBQUU7TUFDYixJQUFDLENBQUEsYUFBYSxDQUFDLGlCQUFmLENBQWlDLENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQTtpQkFBRyxLQUFDLENBQUEsTUFBRCxDQUFBO1FBQUg7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQWpDO01BQ0EsSUFBQyxDQUFBLE1BQUQsQ0FBQTthQUNBLElBQUMsQ0FBQSxXQUFELEdBQWU7SUFITDs7MEJBeUJaLEdBQUEsR0FBSyxTQUFDLEtBQUQ7QUFDSCxVQUFBO01BQUEsS0FBQSxHQUFRLENBQUMsQ0FBQyxTQUFGLENBQVksS0FBWjtBQUNSLFdBQUEsdUNBQUE7O1FBQUEsSUFBQyxDQUFBLEtBQUQsQ0FBTyxJQUFDLENBQUEsUUFBUixFQUFrQixJQUFsQjtBQUFBO01BQ0EsSUFBQyxDQUFBLE1BQUQsQ0FBQTthQUNJLElBQUEsVUFBQSxDQUFXLENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQTtpQkFBRyxLQUFDLENBQUEsTUFBRCxDQUFRLEtBQVI7UUFBSDtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBWDtJQUpEOzswQkFNTCxNQUFBLEdBQVEsU0FBQyxLQUFEO0FBQ04sVUFBQTtBQUFBLFdBQUEsdUNBQUE7O1FBQUEsSUFBQyxDQUFBLE9BQUQsQ0FBUyxJQUFDLENBQUEsUUFBVixFQUFvQixJQUFwQjtBQUFBO2FBQ0EsSUFBQyxDQUFBLE1BQUQsQ0FBQTtJQUZNOzswQkFJUixLQUFBLEdBQU8sU0FBQTtNQUNMLElBQUMsQ0FBQSxRQUFELEdBQVk7YUFDWixJQUFDLENBQUEsTUFBRCxDQUFBO0lBRks7OzBCQVVQLGVBQUEsR0FBaUIsU0FBQyxRQUFEO0FBQ2YsVUFBQTtBQUFBO1FBQ0UsSUFBZSxRQUFRLENBQUMsSUFBSSxDQUFDLHFCQUFkLENBQW9DLFFBQXBDLENBQWY7QUFBQSxpQkFBTyxLQUFQO1NBREY7T0FBQSxjQUFBO1FBRU07QUFFSixlQUFPLE1BSlQ7O01BUUEsSUFBTyx1QkFBUDtRQUVFLFlBQUEsR0FBZSxRQUFRLENBQUMsY0FBYyxDQUFDLGNBQXhCLENBQXVDLFFBQVEsQ0FBQyxZQUFoRCxFQUE4RCxNQUE5RDtRQUVmLFFBQUEsR0FBVyxZQUFZLENBQUMsYUFBYixDQUEyQixNQUEzQjtRQUNYLFFBQUEsUUFBUSxDQUFDLFNBQVQsQ0FBa0IsQ0FBQyxHQUFuQixhQUF1QixJQUFDLENBQUEsaUJBQUQsQ0FBbUIsUUFBUSxDQUFDLElBQTVCLENBQXZCO1FBRUEsYUFBQSxHQUFnQixZQUFZLENBQUMsYUFBYixDQUEyQixnQkFBM0I7UUFDaEIsZ0JBQUEsR0FBbUIsSUFBQyxDQUFBLGlCQUFELENBQW1CLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBZCxDQUE0QixnQkFBNUIsQ0FBbkI7UUFDbkIsSUFBb0MsZ0JBQWdCLENBQUMsTUFBakIsS0FBMkIsQ0FBL0Q7VUFBQSxnQkFBQSxHQUFtQixDQUFDLFdBQUQsRUFBbkI7O1FBQ0EsUUFBQSxhQUFhLENBQUMsU0FBZCxDQUF1QixDQUFDLEdBQXhCLGFBQTRCLGdCQUE1QjtRQUVBLFFBQVEsQ0FBQyxXQUFULENBQXFCLGFBQXJCO1FBRUEsSUFBQyxDQUFBLFVBQUQsR0FBYyxZQUFZLENBQUMsYUFBYixDQUEyQixrQkFBM0I7UUFDZCxJQUFDLENBQUEsVUFBVSxDQUFDLFNBQVMsQ0FBQyxHQUF0QixDQUEwQixRQUExQjtRQUNBLGFBQWEsQ0FBQyxXQUFkLENBQTBCLElBQUMsQ0FBQSxVQUEzQixFQWhCRjs7TUFrQkEsT0FBQSxHQUFVLElBQUMsQ0FBQTtBQUNYLGFBQU0sT0FBTjtRQUNFLElBQWUsT0FBTyxDQUFDLHFCQUFSLENBQThCLFFBQTlCLENBQWY7QUFBQSxpQkFBTyxLQUFQOztRQUNBLE9BQUEsR0FBVSxPQUFPLENBQUM7TUFGcEI7YUFJQTtJQWhDZTs7MEJBbUNqQixNQUFBLEdBQVEsU0FBQTtNQUNOLElBQUEsQ0FBYyxJQUFDLENBQUEsV0FBZjtBQUFBLGVBQUE7O01BRUEsSUFBMkMsbUNBQTNDO1FBQUEsY0FBQSxDQUFlLElBQUMsQ0FBQSxzQkFBaEIsRUFBQTs7YUFFQSxJQUFDLENBQUEsc0JBQUQsR0FBMEIsWUFBQSxDQUFhLENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQTtBQUNyQyxjQUFBO1VBQUEsZUFBQSxHQUFrQixJQUFJO0FBQ3RCO0FBQUEsZUFBQSxzQ0FBQTs7WUFDRSxJQUFHLE9BQU8sQ0FBQyxPQUFSLEtBQW1CLFFBQXRCO2NBQ0UsZUFBZSxDQUFDLEdBQWhCLENBQW9CLE9BQU8sQ0FBQyxVQUE1QixFQURGOztBQURGO1VBSUEsbUJBQUEsR0FBc0I7QUFDdEI7QUFBQSxlQUFBLHdDQUFBOztZQUNFLElBQUEsQ0FBZ0IsS0FBQyxDQUFBLGVBQUQsQ0FBaUIsT0FBTyxDQUFDLFFBQXpCLENBQWhCO0FBQUEsdUJBQUE7O1lBQ0EsSUFBWSxlQUFlLENBQUMsR0FBaEIsQ0FBb0IsT0FBTyxDQUFDLFVBQTVCLENBQVo7QUFBQSx1QkFBQTs7WUFDQSxJQUFZLE9BQU8sQ0FBQyxVQUFVLENBQUMsUUFBbkIsQ0FBNEIsR0FBNUIsQ0FBWjtBQUFBLHVCQUFBOztZQUNBLElBQVksT0FBTyxDQUFDLFFBQVIsS0FBb0IsUUFBcEIsSUFBaUMsa0JBQWtCLENBQUMsSUFBbkIsQ0FBd0IsT0FBTyxDQUFDLFVBQWhDLENBQTdDO0FBQUEsdUJBQUE7O1lBQ0EsSUFBWSxPQUFPLENBQUMsUUFBUixLQUFvQixPQUFwQixJQUFnQyx1QkFBdUIsQ0FBQyxJQUF4QixDQUE2QixPQUFPLENBQUMsVUFBckMsQ0FBNUM7QUFBQSx1QkFBQTs7O2NBQ0EsNEJBQXdDOztZQUN4QyxtQkFBb0IsQ0FBQSxPQUFPLENBQUMsT0FBUixDQUFnQixDQUFDLE9BQXJDLENBQTZDLE9BQU8sQ0FBQyxVQUFyRDtBQVBGO2lCQVNBLEtBQUMsQ0FBQSxvQkFBRCxDQUFzQixLQUFDLENBQUEsUUFBdkIsRUFBaUMsbUJBQWpDO1FBaEJxQztNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBYjtJQUxwQjs7MEJBdUJSLGlCQUFBLEdBQW1CLFNBQUE7QUFDakIsVUFBQTtNQUFBLElBQUcsb0JBQUg7ZUFDRSxJQUFDLENBQUEsR0FBRCxDQUFLLFlBQUwsRUFERjtPQUFBLE1BQUE7UUFHRSxZQUFBLEdBQWUsSUFBSSxDQUFDLElBQUwsQ0FBVSxJQUFDLENBQUEsWUFBWCxFQUF5QixPQUF6QjtRQUNmLGdCQUFBLEdBQW1CLEVBQUUsQ0FBQyxPQUFILENBQVcsWUFBWCxFQUF5QixPQUFPLENBQUMsUUFBakMsRUFBMkMsQ0FBQyxNQUFELEVBQVMsTUFBVCxDQUEzQztRQUNsQixPQUFRLElBQUksQ0FBQyxZQUFMLENBQWtCLGdCQUFsQjtlQUNULElBQUMsQ0FBQSxHQUFELENBQUssSUFBTCxFQU5GOztJQURpQjs7MEJBV25CLEtBQUEsR0FBTyxTQUFDLElBQUQsRUFBTyxJQUFQO2FBQ0wsV0FBVyxDQUFDLEtBQVosQ0FBa0IsSUFBbEIsRUFBd0IsSUFBeEI7SUFESzs7MEJBR1AsT0FBQSxHQUFTLFNBQUMsSUFBRCxFQUFPLElBQVA7YUFDUCxXQUFXLENBQUMsT0FBWixDQUFvQixJQUFwQixFQUEwQixJQUExQjtJQURPOzswQkFHVCxvQkFBQSxHQUFzQixTQUFDLFFBQUQsRUFBVyxtQkFBWDthQUNwQixXQUFXLENBQUMsSUFBWixDQUFpQix5QkFBakIsRUFBNEMsUUFBNUMsRUFBc0QsbUJBQXREO0lBRG9COzswQkFJdEIsaUJBQUEsR0FBbUIsU0FBQyxPQUFEO0FBQ2pCLFVBQUE7TUFBQSxJQUFHLFNBQUEscUJBQVksT0FBTyxDQUFFLGtCQUF4QjtlQUNFLEtBQUssQ0FBQSxTQUFFLENBQUEsS0FBSyxDQUFDLEtBQWIsQ0FBbUIsU0FBbkIsRUFERjtPQUFBLE1BQUE7ZUFHRSxHQUhGOztJQURpQjs7MEJBTW5CLGdCQUFBLEdBQWtCLFNBQUE7QUFDaEIsVUFBQTtNQUFBLFlBQUEsR0FBZSxDQUFDLENBQUMsSUFBRixDQUFPLElBQUMsQ0FBQSxRQUFSLEVBQWtCLFNBQUMsR0FBRDtBQUFhLFlBQUE7UUFBWCxRQUFEO2VBQVksV0FBVyxDQUFDLGNBQVosQ0FBMkIsS0FBM0IsQ0FBQSxLQUFxQztNQUFsRCxDQUFsQjtNQUNmLElBQWMsOERBQWQ7QUFBQSxlQUFBOztNQUVBLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBckIsQ0FBMEIsU0FBQyxLQUFELEVBQVEsS0FBUjtRQUN4QixJQUFHLEtBQUssQ0FBQyxLQUFOLElBQWdCLEtBQUssQ0FBQyxLQUF6QjtpQkFDRSxXQUFXLENBQUMsY0FBWixDQUEyQixLQUFLLENBQUMsS0FBakMsQ0FBdUMsQ0FBQyxhQUF4QyxDQUFzRCxXQUFXLENBQUMsY0FBWixDQUEyQixLQUFLLENBQUMsS0FBakMsQ0FBdEQsRUFERjtTQUFBLE1BQUE7aUJBR0UsRUFIRjs7TUFEd0IsQ0FBMUI7YUFLQSxJQUFDLENBQUEsTUFBRCxDQUFBO0lBVGdCOzs7OztBQXRNcEIiLCJzb3VyY2VzQ29udGVudCI6WyJwYXRoID0gcmVxdWlyZSAncGF0aCdcblxuXyA9IHJlcXVpcmUgJ3VuZGVyc2NvcmUtcGx1cydcbntpcGNSZW5kZXJlcn0gPSByZXF1aXJlICdlbGVjdHJvbidcbkNTT04gPSByZXF1aXJlICdzZWFzb24nXG5mcyA9IHJlcXVpcmUgJ2ZzLXBsdXMnXG57RGlzcG9zYWJsZX0gPSByZXF1aXJlICdldmVudC1raXQnXG5cbk1lbnVIZWxwZXJzID0gcmVxdWlyZSAnLi9tZW51LWhlbHBlcnMnXG5cbnBsYXRmb3JtTWVudSA9IHJlcXVpcmUoJy4uL3BhY2thZ2UuanNvbicpPy5fYXRvbU1lbnU/Lm1lbnVcblxuIyBFeHRlbmRlZDogUHJvdmlkZXMgYSByZWdpc3RyeSBmb3IgbWVudSBpdGVtcyB0aGF0IHlvdSdkIGxpa2UgdG8gYXBwZWFyIGluIHRoZVxuIyBhcHBsaWNhdGlvbiBtZW51LlxuI1xuIyBBbiBpbnN0YW5jZSBvZiB0aGlzIGNsYXNzIGlzIGFsd2F5cyBhdmFpbGFibGUgYXMgdGhlIGBhdG9tLm1lbnVgIGdsb2JhbC5cbiNcbiMgIyMgTWVudSBDU09OIEZvcm1hdFxuI1xuIyBIZXJlIGlzIGFuIGV4YW1wbGUgZnJvbSB0aGUgW3RyZWUtdmlld10oaHR0cHM6Ly9naXRodWIuY29tL2F0b20vdHJlZS12aWV3L2Jsb2IvbWFzdGVyL21lbnVzL3RyZWUtdmlldy5jc29uKTpcbiNcbiMgYGBgY29mZmVlXG4jIFtcbiMgICB7XG4jICAgICAnbGFiZWwnOiAnVmlldydcbiMgICAgICdzdWJtZW51JzogW1xuIyAgICAgICB7ICdsYWJlbCc6ICdUb2dnbGUgVHJlZSBWaWV3JywgJ2NvbW1hbmQnOiAndHJlZS12aWV3OnRvZ2dsZScgfVxuIyAgICAgXVxuIyAgIH1cbiMgICB7XG4jICAgICAnbGFiZWwnOiAnUGFja2FnZXMnXG4jICAgICAnc3VibWVudSc6IFtcbiMgICAgICAgJ2xhYmVsJzogJ1RyZWUgVmlldydcbiMgICAgICAgJ3N1Ym1lbnUnOiBbXG4jICAgICAgICAgeyAnbGFiZWwnOiAnRm9jdXMnLCAnY29tbWFuZCc6ICd0cmVlLXZpZXc6dG9nZ2xlLWZvY3VzJyB9XG4jICAgICAgICAgeyAnbGFiZWwnOiAnVG9nZ2xlJywgJ2NvbW1hbmQnOiAndHJlZS12aWV3OnRvZ2dsZScgfVxuIyAgICAgICAgIHsgJ2xhYmVsJzogJ1JldmVhbCBBY3RpdmUgRmlsZScsICdjb21tYW5kJzogJ3RyZWUtdmlldzpyZXZlYWwtYWN0aXZlLWZpbGUnIH1cbiMgICAgICAgICB7ICdsYWJlbCc6ICdUb2dnbGUgVHJlZSBTaWRlJywgJ2NvbW1hbmQnOiAndHJlZS12aWV3OnRvZ2dsZS1zaWRlJyB9XG4jICAgICAgIF1cbiMgICAgIF1cbiMgICB9XG4jIF1cbiMgYGBgXG4jXG4jIFVzZSBpbiB5b3VyIHBhY2thZ2UncyBtZW51IGAuY3NvbmAgZmlsZSByZXF1aXJlcyB0aGF0IHlvdSBwbGFjZSB5b3VyIG1lbnVcbiMgc3RydWN0dXJlIHVuZGVyIGEgYG1lbnVgIGtleS5cbiNcbiMgYGBgY29mZmVlXG4jICdtZW51JzogW1xuIyAgIHtcbiMgICAgICdsYWJlbCc6ICdWaWV3J1xuIyAgICAgJ3N1Ym1lbnUnOiBbXG4jICAgICAgIHsgJ2xhYmVsJzogJ1RvZ2dsZSBUcmVlIFZpZXcnLCAnY29tbWFuZCc6ICd0cmVlLXZpZXc6dG9nZ2xlJyB9XG4jICAgICBdXG4jICAgfVxuIyBdXG4jIGBgYFxuI1xuIyBTZWUgezo6YWRkfSBmb3IgbW9yZSBpbmZvIGFib3V0IGFkZGluZyBtZW51J3MgZGlyZWN0bHkuXG5tb2R1bGUuZXhwb3J0cyA9XG5jbGFzcyBNZW51TWFuYWdlclxuICBjb25zdHJ1Y3RvcjogKHtAcmVzb3VyY2VQYXRoLCBAa2V5bWFwTWFuYWdlciwgQHBhY2thZ2VNYW5hZ2VyfSkgLT5cbiAgICBAaW5pdGlhbGl6ZWQgPSBmYWxzZVxuICAgIEBwZW5kaW5nVXBkYXRlT3BlcmF0aW9uID0gbnVsbFxuICAgIEB0ZW1wbGF0ZSA9IFtdXG4gICAgQGtleW1hcE1hbmFnZXIub25EaWRMb2FkQnVuZGxlZEtleW1hcHMgPT4gQGxvYWRQbGF0Zm9ybUl0ZW1zKClcbiAgICBAcGFja2FnZU1hbmFnZXIub25EaWRBY3RpdmF0ZUluaXRpYWxQYWNrYWdlcyA9PiBAc29ydFBhY2thZ2VzTWVudSgpXG5cbiAgaW5pdGlhbGl6ZTogKHtAcmVzb3VyY2VQYXRofSkgLT5cbiAgICBAa2V5bWFwTWFuYWdlci5vbkRpZFJlbG9hZEtleW1hcCA9PiBAdXBkYXRlKClcbiAgICBAdXBkYXRlKClcbiAgICBAaW5pdGlhbGl6ZWQgPSB0cnVlXG5cbiAgIyBQdWJsaWM6IEFkZHMgdGhlIGdpdmVuIGl0ZW1zIHRvIHRoZSBhcHBsaWNhdGlvbiBtZW51LlxuICAjXG4gICMgIyMgRXhhbXBsZXNcbiAgIyBgYGBjb2ZmZWVcbiAgIyAgIGF0b20ubWVudS5hZGQgW1xuICAjICAgICB7XG4gICMgICAgICAgbGFiZWw6ICdIZWxsbydcbiAgIyAgICAgICBzdWJtZW51IDogW3tsYWJlbDogJ1dvcmxkIScsIGNvbW1hbmQ6ICdoZWxsbzp3b3JsZCd9XVxuICAjICAgICB9XG4gICMgICBdXG4gICMgYGBgXG4gICNcbiAgIyAqIGBpdGVtc2AgQW4ge0FycmF5fSBvZiBtZW51IGl0ZW0ge09iamVjdH1zIGNvbnRhaW5pbmcgdGhlIGtleXM6XG4gICMgICAqIGBsYWJlbGAgVGhlIHtTdHJpbmd9IG1lbnUgbGFiZWwuXG4gICMgICAqIGBzdWJtZW51YCBBbiBvcHRpb25hbCB7QXJyYXl9IG9mIHN1YiBtZW51IGl0ZW1zLlxuICAjICAgKiBgY29tbWFuZGAgQW4gb3B0aW9uYWwge1N0cmluZ30gY29tbWFuZCB0byB0cmlnZ2VyIHdoZW4gdGhlIGl0ZW0gaXNcbiAgIyAgICAgY2xpY2tlZC5cbiAgI1xuICAjIFJldHVybnMgYSB7RGlzcG9zYWJsZX0gb24gd2hpY2ggYC5kaXNwb3NlKClgIGNhbiBiZSBjYWxsZWQgdG8gcmVtb3ZlIHRoZVxuICAjIGFkZGVkIG1lbnUgaXRlbXMuXG4gIGFkZDogKGl0ZW1zKSAtPlxuICAgIGl0ZW1zID0gXy5kZWVwQ2xvbmUoaXRlbXMpXG4gICAgQG1lcmdlKEB0ZW1wbGF0ZSwgaXRlbSkgZm9yIGl0ZW0gaW4gaXRlbXNcbiAgICBAdXBkYXRlKClcbiAgICBuZXcgRGlzcG9zYWJsZSA9PiBAcmVtb3ZlKGl0ZW1zKVxuXG4gIHJlbW92ZTogKGl0ZW1zKSAtPlxuICAgIEB1bm1lcmdlKEB0ZW1wbGF0ZSwgaXRlbSkgZm9yIGl0ZW0gaW4gaXRlbXNcbiAgICBAdXBkYXRlKClcblxuICBjbGVhcjogLT5cbiAgICBAdGVtcGxhdGUgPSBbXVxuICAgIEB1cGRhdGUoKVxuXG4gICMgU2hvdWxkIHRoZSBiaW5kaW5nIGZvciB0aGUgZ2l2ZW4gc2VsZWN0b3IgYmUgaW5jbHVkZWQgaW4gdGhlIG1lbnVcbiAgIyBjb21tYW5kcy5cbiAgI1xuICAjICogYHNlbGVjdG9yYCBBIHtTdHJpbmd9IHNlbGVjdG9yIHRvIGNoZWNrLlxuICAjXG4gICMgUmV0dXJucyBhIHtCb29sZWFufSwgdHJ1ZSB0byBpbmNsdWRlIHRoZSBzZWxlY3RvciwgZmFsc2Ugb3RoZXJ3aXNlLlxuICBpbmNsdWRlU2VsZWN0b3I6IChzZWxlY3RvcikgLT5cbiAgICB0cnlcbiAgICAgIHJldHVybiB0cnVlIGlmIGRvY3VtZW50LmJvZHkud2Via2l0TWF0Y2hlc1NlbGVjdG9yKHNlbGVjdG9yKVxuICAgIGNhdGNoIGVycm9yXG4gICAgICAjIFNlbGVjdG9yIGlzbid0IHZhbGlkXG4gICAgICByZXR1cm4gZmFsc2VcblxuICAgICMgU2ltdWxhdGUgYW4gYXRvbS10ZXh0LWVkaXRvciBlbGVtZW50IGF0dGFjaGVkIHRvIGEgYXRvbS13b3Jrc3BhY2UgZWxlbWVudCBhdHRhY2hlZFxuICAgICMgdG8gYSBib2R5IGVsZW1lbnQgdGhhdCBoYXMgdGhlIHNhbWUgY2xhc3NlcyBhcyB0aGUgY3VycmVudCBib2R5IGVsZW1lbnQuXG4gICAgdW5sZXNzIEB0ZXN0RWRpdG9yP1xuICAgICAgIyBVc2UgbmV3IGRvY3VtZW50IHNvIHRoYXQgY3VzdG9tIGVsZW1lbnRzIGRvbid0IGFjdHVhbGx5IGdldCBjcmVhdGVkXG4gICAgICB0ZXN0RG9jdW1lbnQgPSBkb2N1bWVudC5pbXBsZW1lbnRhdGlvbi5jcmVhdGVEb2N1bWVudChkb2N1bWVudC5uYW1lc3BhY2VVUkksICdodG1sJylcblxuICAgICAgdGVzdEJvZHkgPSB0ZXN0RG9jdW1lbnQuY3JlYXRlRWxlbWVudCgnYm9keScpXG4gICAgICB0ZXN0Qm9keS5jbGFzc0xpc3QuYWRkKEBjbGFzc2VzRm9yRWxlbWVudChkb2N1bWVudC5ib2R5KS4uLilcblxuICAgICAgdGVzdFdvcmtzcGFjZSA9IHRlc3REb2N1bWVudC5jcmVhdGVFbGVtZW50KCdhdG9tLXdvcmtzcGFjZScpXG4gICAgICB3b3Jrc3BhY2VDbGFzc2VzID0gQGNsYXNzZXNGb3JFbGVtZW50KGRvY3VtZW50LmJvZHkucXVlcnlTZWxlY3RvcignYXRvbS13b3Jrc3BhY2UnKSlcbiAgICAgIHdvcmtzcGFjZUNsYXNzZXMgPSBbJ3dvcmtzcGFjZSddIGlmIHdvcmtzcGFjZUNsYXNzZXMubGVuZ3RoIGlzIDBcbiAgICAgIHRlc3RXb3Jrc3BhY2UuY2xhc3NMaXN0LmFkZCh3b3Jrc3BhY2VDbGFzc2VzLi4uKVxuXG4gICAgICB0ZXN0Qm9keS5hcHBlbmRDaGlsZCh0ZXN0V29ya3NwYWNlKVxuXG4gICAgICBAdGVzdEVkaXRvciA9IHRlc3REb2N1bWVudC5jcmVhdGVFbGVtZW50KCdhdG9tLXRleHQtZWRpdG9yJylcbiAgICAgIEB0ZXN0RWRpdG9yLmNsYXNzTGlzdC5hZGQoJ2VkaXRvcicpXG4gICAgICB0ZXN0V29ya3NwYWNlLmFwcGVuZENoaWxkKEB0ZXN0RWRpdG9yKVxuXG4gICAgZWxlbWVudCA9IEB0ZXN0RWRpdG9yXG4gICAgd2hpbGUgZWxlbWVudFxuICAgICAgcmV0dXJuIHRydWUgaWYgZWxlbWVudC53ZWJraXRNYXRjaGVzU2VsZWN0b3Ioc2VsZWN0b3IpXG4gICAgICBlbGVtZW50ID0gZWxlbWVudC5wYXJlbnRFbGVtZW50XG5cbiAgICBmYWxzZVxuXG4gICMgUHVibGljOiBSZWZyZXNoZXMgdGhlIGN1cnJlbnRseSB2aXNpYmxlIG1lbnUuXG4gIHVwZGF0ZTogLT5cbiAgICByZXR1cm4gdW5sZXNzIEBpbml0aWFsaXplZFxuXG4gICAgY2xlYXJJbW1lZGlhdGUoQHBlbmRpbmdVcGRhdGVPcGVyYXRpb24pIGlmIEBwZW5kaW5nVXBkYXRlT3BlcmF0aW9uP1xuXG4gICAgQHBlbmRpbmdVcGRhdGVPcGVyYXRpb24gPSBzZXRJbW1lZGlhdGUgPT5cbiAgICAgIHVuc2V0S2V5c3Ryb2tlcyA9IG5ldyBTZXRcbiAgICAgIGZvciBiaW5kaW5nIGluIEBrZXltYXBNYW5hZ2VyLmdldEtleUJpbmRpbmdzKClcbiAgICAgICAgaWYgYmluZGluZy5jb21tYW5kIGlzICd1bnNldCEnXG4gICAgICAgICAgdW5zZXRLZXlzdHJva2VzLmFkZChiaW5kaW5nLmtleXN0cm9rZXMpXG5cbiAgICAgIGtleXN0cm9rZXNCeUNvbW1hbmQgPSB7fVxuICAgICAgZm9yIGJpbmRpbmcgaW4gQGtleW1hcE1hbmFnZXIuZ2V0S2V5QmluZGluZ3MoKVxuICAgICAgICBjb250aW51ZSB1bmxlc3MgQGluY2x1ZGVTZWxlY3RvcihiaW5kaW5nLnNlbGVjdG9yKVxuICAgICAgICBjb250aW51ZSBpZiB1bnNldEtleXN0cm9rZXMuaGFzKGJpbmRpbmcua2V5c3Ryb2tlcylcbiAgICAgICAgY29udGludWUgaWYgYmluZGluZy5rZXlzdHJva2VzLmluY2x1ZGVzKCcgJylcbiAgICAgICAgY29udGludWUgaWYgcHJvY2Vzcy5wbGF0Zm9ybSBpcyAnZGFyd2luJyBhbmQgL15hbHQtKHNoaWZ0LSk/LiQvLnRlc3QoYmluZGluZy5rZXlzdHJva2VzKVxuICAgICAgICBjb250aW51ZSBpZiBwcm9jZXNzLnBsYXRmb3JtIGlzICd3aW4zMicgYW5kIC9eY3RybC1hbHQtKHNoaWZ0LSk/LiQvLnRlc3QoYmluZGluZy5rZXlzdHJva2VzKVxuICAgICAgICBrZXlzdHJva2VzQnlDb21tYW5kW2JpbmRpbmcuY29tbWFuZF0gPz0gW11cbiAgICAgICAga2V5c3Ryb2tlc0J5Q29tbWFuZFtiaW5kaW5nLmNvbW1hbmRdLnVuc2hpZnQgYmluZGluZy5rZXlzdHJva2VzXG5cbiAgICAgIEBzZW5kVG9Ccm93c2VyUHJvY2VzcyhAdGVtcGxhdGUsIGtleXN0cm9rZXNCeUNvbW1hbmQpXG5cbiAgbG9hZFBsYXRmb3JtSXRlbXM6IC0+XG4gICAgaWYgcGxhdGZvcm1NZW51P1xuICAgICAgQGFkZChwbGF0Zm9ybU1lbnUpXG4gICAgZWxzZVxuICAgICAgbWVudXNEaXJQYXRoID0gcGF0aC5qb2luKEByZXNvdXJjZVBhdGgsICdtZW51cycpXG4gICAgICBwbGF0Zm9ybU1lbnVQYXRoID0gZnMucmVzb2x2ZShtZW51c0RpclBhdGgsIHByb2Nlc3MucGxhdGZvcm0sIFsnY3NvbicsICdqc29uJ10pXG4gICAgICB7bWVudX0gPSBDU09OLnJlYWRGaWxlU3luYyhwbGF0Zm9ybU1lbnVQYXRoKVxuICAgICAgQGFkZChtZW51KVxuXG4gICMgTWVyZ2VzIGFuIGl0ZW0gaW4gYSBzdWJtZW51IGF3YXJlIHdheSBzdWNoIHRoYXQgbmV3IGl0ZW1zIGFyZSBhbHdheXNcbiAgIyBhcHBlbmRlZCB0byB0aGUgYm90dG9tIG9mIGV4aXN0aW5nIG1lbnVzIHdoZXJlIHBvc3NpYmxlLlxuICBtZXJnZTogKG1lbnUsIGl0ZW0pIC0+XG4gICAgTWVudUhlbHBlcnMubWVyZ2UobWVudSwgaXRlbSlcblxuICB1bm1lcmdlOiAobWVudSwgaXRlbSkgLT5cbiAgICBNZW51SGVscGVycy51bm1lcmdlKG1lbnUsIGl0ZW0pXG5cbiAgc2VuZFRvQnJvd3NlclByb2Nlc3M6ICh0ZW1wbGF0ZSwga2V5c3Ryb2tlc0J5Q29tbWFuZCkgLT5cbiAgICBpcGNSZW5kZXJlci5zZW5kICd1cGRhdGUtYXBwbGljYXRpb24tbWVudScsIHRlbXBsYXRlLCBrZXlzdHJva2VzQnlDb21tYW5kXG5cbiAgIyBHZXQgYW4ge0FycmF5fSBvZiB7U3RyaW5nfSBjbGFzc2VzIGZvciB0aGUgZ2l2ZW4gZWxlbWVudC5cbiAgY2xhc3Nlc0ZvckVsZW1lbnQ6IChlbGVtZW50KSAtPlxuICAgIGlmIGNsYXNzTGlzdCA9IGVsZW1lbnQ/LmNsYXNzTGlzdFxuICAgICAgQXJyYXk6OnNsaWNlLmFwcGx5KGNsYXNzTGlzdClcbiAgICBlbHNlXG4gICAgICBbXVxuXG4gIHNvcnRQYWNrYWdlc01lbnU6IC0+XG4gICAgcGFja2FnZXNNZW51ID0gXy5maW5kIEB0ZW1wbGF0ZSwgKHtsYWJlbH0pIC0+IE1lbnVIZWxwZXJzLm5vcm1hbGl6ZUxhYmVsKGxhYmVsKSBpcyAnUGFja2FnZXMnXG4gICAgcmV0dXJuIHVubGVzcyBwYWNrYWdlc01lbnU/LnN1Ym1lbnU/XG5cbiAgICBwYWNrYWdlc01lbnUuc3VibWVudS5zb3J0IChpdGVtMSwgaXRlbTIpIC0+XG4gICAgICBpZiBpdGVtMS5sYWJlbCBhbmQgaXRlbTIubGFiZWxcbiAgICAgICAgTWVudUhlbHBlcnMubm9ybWFsaXplTGFiZWwoaXRlbTEubGFiZWwpLmxvY2FsZUNvbXBhcmUoTWVudUhlbHBlcnMubm9ybWFsaXplTGFiZWwoaXRlbTIubGFiZWwpKVxuICAgICAgZWxzZVxuICAgICAgICAwXG4gICAgQHVwZGF0ZSgpXG4iXX0=
