(function() {
  var AutoUpdateManager, CheckingState, DownloadingState, ErrorState, EventEmitter, IdleState, NoUpdateAvailableState, UnsupportedState, UpdateAvailableState, autoUpdater, path,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  autoUpdater = null;

  EventEmitter = require('events').EventEmitter;

  path = require('path');

  IdleState = 'idle';

  CheckingState = 'checking';

  DownloadingState = 'downloading';

  UpdateAvailableState = 'update-available';

  NoUpdateAvailableState = 'no-update-available';

  UnsupportedState = 'unsupported';

  ErrorState = 'error';

  module.exports = AutoUpdateManager = (function() {
    Object.assign(AutoUpdateManager.prototype, EventEmitter.prototype);

    function AutoUpdateManager(version, testMode, config) {
      this.version = version;
      this.testMode = testMode;
      this.config = config;
      this.onUpdateError = bind(this.onUpdateError, this);
      this.onUpdateNotAvailable = bind(this.onUpdateNotAvailable, this);
      this.state = IdleState;
      this.iconPath = path.resolve(__dirname, '..', '..', 'resources', 'atom.png');
    }

    AutoUpdateManager.prototype.initialize = function() {
      var archSuffix;
      if (process.platform === 'win32') {
        archSuffix = process.arch === 'ia32' ? '' : '-' + process.arch;
        this.feedUrl = "https://atom.io/api/updates" + archSuffix + "?version=" + this.version;
        autoUpdater = require('./auto-updater-win32');
      } else {
        this.feedUrl = "https://atom.io/api/updates?version=" + this.version;
        autoUpdater = require('electron').autoUpdater;
      }
      autoUpdater.on('error', (function(_this) {
        return function(event, message) {
          _this.setState(ErrorState, message);
          _this.emitWindowEvent('update-error');
          return console.error("Error Downloading Update: " + message);
        };
      })(this));
      autoUpdater.setFeedURL(this.feedUrl);
      autoUpdater.on('checking-for-update', (function(_this) {
        return function() {
          _this.setState(CheckingState);
          return _this.emitWindowEvent('checking-for-update');
        };
      })(this));
      autoUpdater.on('update-not-available', (function(_this) {
        return function() {
          _this.setState(NoUpdateAvailableState);
          return _this.emitWindowEvent('update-not-available');
        };
      })(this));
      autoUpdater.on('update-available', (function(_this) {
        return function() {
          _this.setState(DownloadingState);
          _this.emitWindowEvent('did-begin-downloading-update');
          return _this.emit('did-begin-download');
        };
      })(this));
      autoUpdater.on('update-downloaded', (function(_this) {
        return function(event, releaseNotes, releaseVersion) {
          _this.releaseVersion = releaseVersion;
          _this.setState(UpdateAvailableState);
          return _this.emitUpdateAvailableEvent();
        };
      })(this));
      this.config.onDidChange('core.automaticallyUpdate', (function(_this) {
        return function(arg) {
          var newValue;
          newValue = arg.newValue;
          if (newValue) {
            return _this.scheduleUpdateCheck();
          } else {
            return _this.cancelScheduledUpdateCheck();
          }
        };
      })(this));
      if (this.config.get('core.automaticallyUpdate')) {
        this.scheduleUpdateCheck();
      }
      switch (process.platform) {
        case 'win32':
          if (!autoUpdater.supportsUpdates()) {
            return this.setState(UnsupportedState);
          }
          break;
        case 'linux':
          return this.setState(UnsupportedState);
      }
    };

    AutoUpdateManager.prototype.emitUpdateAvailableEvent = function() {
      if (this.releaseVersion == null) {
        return;
      }
      this.emitWindowEvent('update-available', {
        releaseVersion: this.releaseVersion
      });
    };

    AutoUpdateManager.prototype.emitWindowEvent = function(eventName, payload) {
      var atomWindow, i, len, ref;
      ref = this.getWindows();
      for (i = 0, len = ref.length; i < len; i++) {
        atomWindow = ref[i];
        atomWindow.sendMessage(eventName, payload);
      }
    };

    AutoUpdateManager.prototype.setState = function(state, errorMessage) {
      if (this.state === state) {
        return;
      }
      this.state = state;
      this.errorMessage = errorMessage;
      return this.emit('state-changed', this.state);
    };

    AutoUpdateManager.prototype.getState = function() {
      return this.state;
    };

    AutoUpdateManager.prototype.getErrorMessage = function() {
      return this.errorMessage;
    };

    AutoUpdateManager.prototype.scheduleUpdateCheck = function() {
      var checkForUpdates, fourHours;
      if (!(/\w{7}/.test(this.version) || this.checkForUpdatesIntervalID)) {
        checkForUpdates = (function(_this) {
          return function() {
            return _this.check({
              hidePopups: true
            });
          };
        })(this);
        fourHours = 1000 * 60 * 60 * 4;
        this.checkForUpdatesIntervalID = setInterval(checkForUpdates, fourHours);
        return checkForUpdates();
      }
    };

    AutoUpdateManager.prototype.cancelScheduledUpdateCheck = function() {
      if (this.checkForUpdatesIntervalID) {
        clearInterval(this.checkForUpdatesIntervalID);
        return this.checkForUpdatesIntervalID = null;
      }
    };

    AutoUpdateManager.prototype.check = function(arg) {
      var hidePopups;
      hidePopups = (arg != null ? arg : {}).hidePopups;
      if (!hidePopups) {
        autoUpdater.once('update-not-available', this.onUpdateNotAvailable);
        autoUpdater.once('error', this.onUpdateError);
      }
      if (!this.testMode) {
        return autoUpdater.checkForUpdates();
      }
    };

    AutoUpdateManager.prototype.install = function() {
      if (!this.testMode) {
        return autoUpdater.quitAndInstall();
      }
    };

    AutoUpdateManager.prototype.onUpdateNotAvailable = function() {
      var dialog;
      autoUpdater.removeListener('error', this.onUpdateError);
      dialog = require('electron').dialog;
      return dialog.showMessageBox({
        type: 'info',
        buttons: ['OK'],
        icon: this.iconPath,
        message: 'No update available.',
        title: 'No Update Available',
        detail: "Version " + this.version + " is the latest version."
      });
    };

    AutoUpdateManager.prototype.onUpdateError = function(event, message) {
      var dialog;
      autoUpdater.removeListener('update-not-available', this.onUpdateNotAvailable);
      dialog = require('electron').dialog;
      return dialog.showMessageBox({
        type: 'warning',
        buttons: ['OK'],
        icon: this.iconPath,
        message: 'There was an error checking for updates.',
        title: 'Update Error',
        detail: message
      });
    };

    AutoUpdateManager.prototype.getWindows = function() {
      return global.atomApplication.windows;
    };

    return AutoUpdateManager;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
