(function() {
  var ApplicationMenu, AtomApplication, AtomProtocolHandler, AtomWindow, AutoUpdateManager, BrowserWindow, CompositeDisposable, Config, ConfigSchema, Disposable, EventEmitter, FileRecoveryService, FindParentDir, LocationSuffixRegExp, Menu, Resolve, StorageFolder, _, app, dialog, fs, ipcHelpers, ipcMain, net, os, path, ref, ref1, screen, shell, url,
    slice = [].slice;

  AtomWindow = require('./atom-window');

  ApplicationMenu = require('./application-menu');

  AtomProtocolHandler = require('./atom-protocol-handler');

  AutoUpdateManager = require('./auto-update-manager');

  StorageFolder = require('../storage-folder');

  Config = require('../config');

  FileRecoveryService = require('./file-recovery-service');

  ipcHelpers = require('../ipc-helpers');

  ref = require('electron'), BrowserWindow = ref.BrowserWindow, Menu = ref.Menu, app = ref.app, dialog = ref.dialog, ipcMain = ref.ipcMain, shell = ref.shell, screen = ref.screen;

  ref1 = require('event-kit'), CompositeDisposable = ref1.CompositeDisposable, Disposable = ref1.Disposable;

  fs = require('fs-plus');

  path = require('path');

  os = require('os');

  net = require('net');

  url = require('url');

  EventEmitter = require('events').EventEmitter;

  _ = require('underscore-plus');

  FindParentDir = null;

  Resolve = null;

  ConfigSchema = require('../config-schema');

  LocationSuffixRegExp = /(:\d+)(:\d+)?$/;

  module.exports = AtomApplication = (function() {
    Object.assign(AtomApplication.prototype, EventEmitter.prototype);

    AtomApplication.open = function(options) {
      var client, userNameSafe;
      if (options.socketPath == null) {
        if (process.platform === 'win32') {
          userNameSafe = new Buffer(process.env.USERNAME).toString('base64');
          options.socketPath = "\\\\.\\pipe\\atom-" + options.version + "-" + userNameSafe + "-" + process.arch + "-sock";
        } else {
          options.socketPath = path.join(os.tmpdir(), "atom-" + options.version + "-" + process.env.USER + ".sock");
        }
      }
      if ((process.platform !== 'win32' && !fs.existsSync(options.socketPath)) || options.test || options.benchmark || options.benchmarkTest) {
        new AtomApplication(options).initialize(options);
        return;
      }
      client = net.connect({
        path: options.socketPath
      }, function() {
        return client.write(JSON.stringify(options), function() {
          client.end();
          return app.quit();
        });
      });
      return client.on('error', function() {
        return new AtomApplication(options).initialize(options);
      });
    };

    AtomApplication.prototype.windows = null;

    AtomApplication.prototype.applicationMenu = null;

    AtomApplication.prototype.atomProtocolHandler = null;

    AtomApplication.prototype.resourcePath = null;

    AtomApplication.prototype.version = null;

    AtomApplication.prototype.quitting = false;

    AtomApplication.prototype.exit = function(status) {
      return app.exit(status);
    };

    function AtomApplication(options) {
      this.resourcePath = options.resourcePath, this.devResourcePath = options.devResourcePath, this.version = options.version, this.devMode = options.devMode, this.safeMode = options.safeMode, this.socketPath = options.socketPath, this.logFile = options.logFile, this.userDataDir = options.userDataDir;
      if (options.test || options.benchmark || options.benchmarkTest) {
        this.socketPath = null;
      }
      this.pidsToOpenWindows = {};
      this.windows = [];
      this.config = new Config({
        enablePersistence: true
      });
      this.config.setSchema(null, {
        type: 'object',
        properties: _.clone(ConfigSchema)
      });
      ConfigSchema.projectHome = {
        type: 'string',
        "default": path.join(fs.getHomeDirectory(), 'github'),
        description: 'The directory where projects are assumed to be located. Packages created using the Package Generator will be stored here by default.'
      };
      this.config.initialize({
        configDirPath: process.env.ATOM_HOME,
        resourcePath: this.resourcePath,
        projectHomeSchema: ConfigSchema.projectHome
      });
      this.config.load();
      this.fileRecoveryService = new FileRecoveryService(path.join(process.env.ATOM_HOME, "recovery"));
      this.storageFolder = new StorageFolder(process.env.ATOM_HOME);
      this.autoUpdateManager = new AutoUpdateManager(this.version, options.test || options.benchmark || options.benchmarkTest, this.config);
      this.disposable = new CompositeDisposable;
      this.handleEvents();
    }

    AtomApplication.prototype.initialize = function(options) {
      global.atomApplication = this;
      if (process.platform === 'darwin' && this.config.get('core.useCustomTitleBar')) {
        this.config.unset('core.useCustomTitleBar');
        this.config.set('core.titleBar', 'custom');
      }
      this.config.onDidChange('core.titleBar', this.promptForRestart.bind(this));
      process.nextTick((function(_this) {
        return function() {
          return _this.autoUpdateManager.initialize();
        };
      })(this));
      this.applicationMenu = new ApplicationMenu(this.version, this.autoUpdateManager);
      this.atomProtocolHandler = new AtomProtocolHandler(this.resourcePath, this.safeMode);
      this.listenForArgumentsFromNewProcess();
      this.setupDockMenu();
      return this.launch(options);
    };

    AtomApplication.prototype.destroy = function() {
      var windowsClosePromises;
      windowsClosePromises = this.windows.map(function(window) {
        window.close();
        return window.closedPromise;
      });
      return Promise.all(windowsClosePromises).then((function(_this) {
        return function() {
          return _this.disposable.dispose();
        };
      })(this));
    };

    AtomApplication.prototype.launch = function(options) {
      var ref2, ref3;
      if (((ref2 = options.pathsToOpen) != null ? ref2.length : void 0) > 0 || ((ref3 = options.urlsToOpen) != null ? ref3.length : void 0) > 0 || options.test || options.benchmark || options.benchmarkTest) {
        if (this.config.get('core.restorePreviousWindowsOnStart') === 'always') {
          this.loadState(_.deepClone(options));
        }
        return this.openWithOptions(options);
      } else {
        return this.loadState(options) || this.openPath(options);
      }
    };

    AtomApplication.prototype.openWithOptions = function(options) {
      var addToLastWindow, benchmark, benchmarkTest, clearWindowState, devMode, env, executedFrom, i, initialPaths, len, logFile, newWindow, pathsToOpen, pidToKillWhenClosed, profileStartup, results, safeMode, test, timeout, urlToOpen, urlsToOpen;
      initialPaths = options.initialPaths, pathsToOpen = options.pathsToOpen, executedFrom = options.executedFrom, urlsToOpen = options.urlsToOpen, benchmark = options.benchmark, benchmarkTest = options.benchmarkTest, test = options.test, pidToKillWhenClosed = options.pidToKillWhenClosed, devMode = options.devMode, safeMode = options.safeMode, newWindow = options.newWindow, logFile = options.logFile, profileStartup = options.profileStartup, timeout = options.timeout, clearWindowState = options.clearWindowState, addToLastWindow = options.addToLastWindow, env = options.env;
      app.focus();
      if (test) {
        return this.runTests({
          headless: true,
          devMode: devMode,
          resourcePath: this.resourcePath,
          executedFrom: executedFrom,
          pathsToOpen: pathsToOpen,
          logFile: logFile,
          timeout: timeout,
          env: env
        });
      } else if (benchmark || benchmarkTest) {
        return this.runBenchmarks({
          headless: true,
          test: benchmarkTest,
          resourcePath: this.resourcePath,
          executedFrom: executedFrom,
          pathsToOpen: pathsToOpen,
          timeout: timeout,
          env: env
        });
      } else if (pathsToOpen.length > 0) {
        return this.openPaths({
          initialPaths: initialPaths,
          pathsToOpen: pathsToOpen,
          executedFrom: executedFrom,
          pidToKillWhenClosed: pidToKillWhenClosed,
          newWindow: newWindow,
          devMode: devMode,
          safeMode: safeMode,
          profileStartup: profileStartup,
          clearWindowState: clearWindowState,
          addToLastWindow: addToLastWindow,
          env: env
        });
      } else if (urlsToOpen.length > 0) {
        results = [];
        for (i = 0, len = urlsToOpen.length; i < len; i++) {
          urlToOpen = urlsToOpen[i];
          results.push(this.openUrl({
            urlToOpen: urlToOpen,
            devMode: devMode,
            safeMode: safeMode,
            env: env
          }));
        }
        return results;
      } else {
        return this.openPath({
          initialPaths: initialPaths,
          pidToKillWhenClosed: pidToKillWhenClosed,
          newWindow: newWindow,
          devMode: devMode,
          safeMode: safeMode,
          profileStartup: profileStartup,
          clearWindowState: clearWindowState,
          addToLastWindow: addToLastWindow,
          env: env
        });
      }
    };

    AtomApplication.prototype.removeWindow = function(window) {
      var ref2, ref3;
      this.windows.splice(this.windows.indexOf(window), 1);
      if (this.windows.length === 0) {
        if ((ref2 = this.applicationMenu) != null) {
          ref2.enableWindowSpecificItems(false);
        }
        if ((ref3 = process.platform) === 'win32' || ref3 === 'linux') {
          app.quit();
          return;
        }
      }
      if (!window.isSpec) {
        return this.saveState(true);
      }
    };

    AtomApplication.prototype.addWindow = function(window) {
      var blurHandler, focusHandler, ref2;
      this.windows.push(window);
      if ((ref2 = this.applicationMenu) != null) {
        ref2.addWindow(window.browserWindow);
      }
      window.once('window:loaded', (function(_this) {
        return function() {
          var ref3;
          return (ref3 = _this.autoUpdateManager) != null ? ref3.emitUpdateAvailableEvent(window) : void 0;
        };
      })(this));
      if (!window.isSpec) {
        focusHandler = (function(_this) {
          return function() {
            return _this.lastFocusedWindow = window;
          };
        })(this);
        blurHandler = (function(_this) {
          return function() {
            return _this.saveState(false);
          };
        })(this);
        window.browserWindow.on('focus', focusHandler);
        window.browserWindow.on('blur', blurHandler);
        window.browserWindow.once('closed', (function(_this) {
          return function() {
            if (window === _this.lastFocusedWindow) {
              _this.lastFocusedWindow = null;
            }
            window.browserWindow.removeListener('focus', focusHandler);
            return window.browserWindow.removeListener('blur', blurHandler);
          };
        })(this));
        return window.browserWindow.webContents.once('did-finish-load', (function(_this) {
          return function() {
            return _this.saveState(false);
          };
        })(this));
      }
    };

    AtomApplication.prototype.listenForArgumentsFromNewProcess = function() {
      var server;
      if (this.socketPath == null) {
        return;
      }
      this.deleteSocketFile();
      server = net.createServer((function(_this) {
        return function(connection) {
          var data;
          data = '';
          connection.on('data', function(chunk) {
            return data = data + chunk;
          });
          return connection.on('end', function() {
            var options;
            options = JSON.parse(data);
            return _this.openWithOptions(options);
          });
        };
      })(this));
      server.listen(this.socketPath);
      return server.on('error', function(error) {
        return console.error('Application server failed', error);
      });
    };

    AtomApplication.prototype.deleteSocketFile = function() {
      var error;
      if (process.platform === 'win32' || (this.socketPath == null)) {
        return;
      }
      if (fs.existsSync(this.socketPath)) {
        try {
          return fs.unlinkSync(this.socketPath);
        } catch (error1) {
          error = error1;
          if (error.code !== 'ENOENT') {
            throw error;
          }
        }
      }
    };

    AtomApplication.prototype.handleEvents = function() {
      var clipboard, getLoadSettings;
      getLoadSettings = (function(_this) {
        return function() {
          var ref2, ref3;
          return {
            devMode: (ref2 = _this.focusedWindow()) != null ? ref2.devMode : void 0,
            safeMode: (ref3 = _this.focusedWindow()) != null ? ref3.safeMode : void 0
          };
        };
      })(this);
      this.on('application:quit', function() {
        return app.quit();
      });
      this.on('application:new-window', function() {
        return this.openPath(getLoadSettings());
      });
      this.on('application:new-file', function() {
        var ref2;
        return ((ref2 = this.focusedWindow()) != null ? ref2 : this).openPath();
      });
      this.on('application:open-dev', function() {
        return this.promptForPathToOpen('all', {
          devMode: true
        });
      });
      this.on('application:open-safe', function() {
        return this.promptForPathToOpen('all', {
          safeMode: true
        });
      });
      this.on('application:inspect', function(arg) {
        var atomWindow, x, y;
        x = arg.x, y = arg.y, atomWindow = arg.atomWindow;
        if (atomWindow == null) {
          atomWindow = this.focusedWindow();
        }
        return atomWindow != null ? atomWindow.browserWindow.inspectElement(x, y) : void 0;
      });
      this.on('application:open-documentation', function() {
        return shell.openExternal('http://flight-manual.atom.io/');
      });
      this.on('application:open-discussions', function() {
        return shell.openExternal('https://discuss.atom.io');
      });
      this.on('application:open-faq', function() {
        return shell.openExternal('https://atom.io/faq');
      });
      this.on('application:open-terms-of-use', function() {
        return shell.openExternal('https://atom.io/terms');
      });
      this.on('application:report-issue', function() {
        return shell.openExternal('https://github.com/atom/atom/blob/master/CONTRIBUTING.md#reporting-bugs');
      });
      this.on('application:search-issues', function() {
        return shell.openExternal('https://github.com/search?q=+is%3Aissue+user%3Aatom');
      });
      this.on('application:install-update', (function(_this) {
        return function() {
          _this.quitting = true;
          return _this.autoUpdateManager.install();
        };
      })(this));
      this.on('application:check-for-update', (function(_this) {
        return function() {
          return _this.autoUpdateManager.check();
        };
      })(this));
      if (process.platform === 'darwin') {
        this.on('application:bring-all-windows-to-front', function() {
          return Menu.sendActionToFirstResponder('arrangeInFront:');
        });
        this.on('application:hide', function() {
          return Menu.sendActionToFirstResponder('hide:');
        });
        this.on('application:hide-other-applications', function() {
          return Menu.sendActionToFirstResponder('hideOtherApplications:');
        });
        this.on('application:minimize', function() {
          return Menu.sendActionToFirstResponder('performMiniaturize:');
        });
        this.on('application:unhide-all-applications', function() {
          return Menu.sendActionToFirstResponder('unhideAllApplications:');
        });
        this.on('application:zoom', function() {
          return Menu.sendActionToFirstResponder('zoom:');
        });
      } else {
        this.on('application:minimize', function() {
          var ref2;
          return (ref2 = this.focusedWindow()) != null ? ref2.minimize() : void 0;
        });
        this.on('application:zoom', function() {
          var ref2;
          return (ref2 = this.focusedWindow()) != null ? ref2.maximize() : void 0;
        });
      }
      this.openPathOnEvent('application:about', 'atom://about');
      this.openPathOnEvent('application:show-settings', 'atom://config');
      this.openPathOnEvent('application:open-your-config', 'atom://.atom/config');
      this.openPathOnEvent('application:open-your-init-script', 'atom://.atom/init-script');
      this.openPathOnEvent('application:open-your-keymap', 'atom://.atom/keymap');
      this.openPathOnEvent('application:open-your-snippets', 'atom://.atom/snippets');
      this.openPathOnEvent('application:open-your-stylesheet', 'atom://.atom/stylesheet');
      this.openPathOnEvent('application:open-license', path.join(process.resourcesPath, 'LICENSE.md'));
      this.disposable.add(ipcHelpers.on(app, 'before-quit', (function(_this) {
        return function(event) {
          var resolveBeforeQuitPromise, windowUnloadPromises;
          resolveBeforeQuitPromise = null;
          _this.lastBeforeQuitPromise = new Promise(function(resolve) {
            return resolveBeforeQuitPromise = resolve;
          });
          if (_this.quitting) {
            return resolveBeforeQuitPromise();
          } else {
            event.preventDefault();
            _this.quitting = true;
            windowUnloadPromises = _this.windows.map(function(window) {
              return window.prepareToUnload();
            });
            return Promise.all(windowUnloadPromises).then(function(windowUnloadedResults) {
              var didUnloadAllWindows;
              didUnloadAllWindows = windowUnloadedResults.every(function(didUnloadWindow) {
                return didUnloadWindow;
              });
              if (didUnloadAllWindows) {
                app.quit();
              }
              return resolveBeforeQuitPromise();
            });
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'will-quit', (function(_this) {
        return function() {
          _this.killAllProcesses();
          return _this.deleteSocketFile();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'open-file', (function(_this) {
        return function(event, pathToOpen) {
          event.preventDefault();
          return _this.openPath({
            pathToOpen: pathToOpen
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'open-url', (function(_this) {
        return function(event, urlToOpen) {
          event.preventDefault();
          return _this.openUrl({
            urlToOpen: urlToOpen,
            devMode: _this.devMode,
            safeMode: _this.safeMode
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(app, 'activate', (function(_this) {
        return function(event, hasVisibleWindows) {
          if (!hasVisibleWindows) {
            if (event != null) {
              event.preventDefault();
            }
            return _this.emit('application:new-window');
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'restart-application', (function(_this) {
        return function() {
          return _this.restart();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'resolve-proxy', function(event, requestId, url) {
        return event.sender.session.resolveProxy(url, function(proxy) {
          if (!event.sender.isDestroyed()) {
            return event.sender.send('did-resolve-proxy', requestId, proxy);
          }
        });
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'did-change-history-manager', (function(_this) {
        return function(event) {
          var atomWindow, i, len, ref2, results, webContents;
          ref2 = _this.windows;
          results = [];
          for (i = 0, len = ref2.length; i < len; i++) {
            atomWindow = ref2[i];
            webContents = atomWindow.browserWindow.webContents;
            if (webContents !== event.sender) {
              results.push(webContents.send('did-change-history-manager'));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'open', (function(_this) {
        return function(event, options) {
          var ref2, window;
          window = _this.atomWindowForEvent(event);
          if (options != null) {
            if (typeof options.pathsToOpen === 'string') {
              options.pathsToOpen = [options.pathsToOpen];
            }
            if (((ref2 = options.pathsToOpen) != null ? ref2.length : void 0) > 0) {
              options.window = window;
              return _this.openPaths(options);
            } else {
              return new AtomWindow(_this, _this.fileRecoveryService, options);
            }
          } else {
            return _this.promptForPathToOpen('all', {
              window: window
            });
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'update-application-menu', (function(_this) {
        return function(event, template, keystrokesByCommand) {
          var ref2, win;
          win = BrowserWindow.fromWebContents(event.sender);
          return (ref2 = _this.applicationMenu) != null ? ref2.update(win, template, keystrokesByCommand) : void 0;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'run-package-specs', (function(_this) {
        return function(event, packageSpecPath) {
          return _this.runTests({
            resourcePath: _this.devResourcePath,
            pathsToOpen: [packageSpecPath],
            headless: false
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'run-benchmarks', (function(_this) {
        return function(event, benchmarksPath) {
          return _this.runBenchmarks({
            resourcePath: _this.devResourcePath,
            pathsToOpen: [benchmarksPath],
            headless: false,
            test: false
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'command', (function(_this) {
        return function(event, command) {
          return _this.emit(command);
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'open-command', (function(_this) {
        return function() {
          var args, command, defaultPath, event;
          event = arguments[0], command = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
          if (args.length > 0) {
            defaultPath = args[0];
          }
          switch (command) {
            case 'application:open':
              return _this.promptForPathToOpen('all', getLoadSettings(), defaultPath);
            case 'application:open-file':
              return _this.promptForPathToOpen('file', getLoadSettings(), defaultPath);
            case 'application:open-folder':
              return _this.promptForPathToOpen('folder', getLoadSettings(), defaultPath);
            default:
              return console.log("Invalid open-command received: " + command);
          }
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'window-command', function() {
        var args, command, event, win;
        event = arguments[0], command = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
        win = BrowserWindow.fromWebContents(event.sender);
        return win.emit.apply(win, [command].concat(slice.call(args)));
      }));
      this.disposable.add(ipcHelpers.respondTo('window-method', (function(_this) {
        return function() {
          var args, browserWindow, method, ref2;
          browserWindow = arguments[0], method = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
          return (ref2 = _this.atomWindowForBrowserWindow(browserWindow)) != null ? ref2[method].apply(ref2, args) : void 0;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'pick-folder', (function(_this) {
        return function(event, responseChannel) {
          return _this.promptForPath("folder", function(selectedPaths) {
            return event.sender.send(responseChannel, selectedPaths);
          });
        };
      })(this)));
      this.disposable.add(ipcHelpers.respondTo('set-window-size', function(win, width, height) {
        return win.setSize(width, height);
      }));
      this.disposable.add(ipcHelpers.respondTo('set-window-position', function(win, x, y) {
        return win.setPosition(x, y);
      }));
      this.disposable.add(ipcHelpers.respondTo('center-window', function(win) {
        return win.center();
      }));
      this.disposable.add(ipcHelpers.respondTo('focus-window', function(win) {
        return win.focus();
      }));
      this.disposable.add(ipcHelpers.respondTo('show-window', function(win) {
        return win.show();
      }));
      this.disposable.add(ipcHelpers.respondTo('hide-window', function(win) {
        return win.hide();
      }));
      this.disposable.add(ipcHelpers.respondTo('get-temporary-window-state', function(win) {
        return win.temporaryState;
      }));
      this.disposable.add(ipcHelpers.respondTo('set-temporary-window-state', function(win, state) {
        return win.temporaryState = state;
      }));
      clipboard = require('../safe-clipboard');
      this.disposable.add(ipcHelpers.on(ipcMain, 'write-text-to-selection-clipboard', function(event, selectedText) {
        return clipboard.writeText(selectedText, 'selection');
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'write-to-stdout', function(event, output) {
        return process.stdout.write(output);
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'write-to-stderr', function(event, output) {
        return process.stderr.write(output);
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'add-recent-document', function(event, filename) {
        return app.addRecentDocument(filename);
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'execute-javascript-in-dev-tools', function(event, code) {
        var ref2;
        return (ref2 = event.sender.devToolsWebContents) != null ? ref2.executeJavaScript(code) : void 0;
      }));
      this.disposable.add(ipcHelpers.on(ipcMain, 'get-auto-update-manager-state', (function(_this) {
        return function(event) {
          return event.returnValue = _this.autoUpdateManager.getState();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'get-auto-update-manager-error', (function(_this) {
        return function(event) {
          return event.returnValue = _this.autoUpdateManager.getErrorMessage();
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'will-save-path', (function(_this) {
        return function(event, path) {
          _this.fileRecoveryService.willSavePath(_this.atomWindowForEvent(event), path);
          return event.returnValue = true;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'did-save-path', (function(_this) {
        return function(event, path) {
          _this.fileRecoveryService.didSavePath(_this.atomWindowForEvent(event), path);
          return event.returnValue = true;
        };
      })(this)));
      this.disposable.add(ipcHelpers.on(ipcMain, 'did-change-paths', (function(_this) {
        return function() {
          return _this.saveState(false);
        };
      })(this)));
      return this.disposable.add(this.disableZoomOnDisplayChange());
    };

    AtomApplication.prototype.setupDockMenu = function() {
      var dockMenu;
      if (process.platform === 'darwin') {
        dockMenu = Menu.buildFromTemplate([
          {
            label: 'New Window',
            click: (function(_this) {
              return function() {
                return _this.emit('application:new-window');
              };
            })(this)
          }
        ]);
        return app.dock.setMenu(dockMenu);
      }
    };

    AtomApplication.prototype.sendCommand = function() {
      var args, command, focusedWindow;
      command = arguments[0], args = 2 <= arguments.length ? slice.call(arguments, 1) : [];
      if (!this.emit.apply(this, [command].concat(slice.call(args)))) {
        focusedWindow = this.focusedWindow();
        if (focusedWindow != null) {
          return focusedWindow.sendCommand.apply(focusedWindow, [command].concat(slice.call(args)));
        } else {
          return this.sendCommandToFirstResponder(command);
        }
      }
    };

    AtomApplication.prototype.sendCommandToWindow = function() {
      var args, atomWindow, command;
      command = arguments[0], atomWindow = arguments[1], args = 3 <= arguments.length ? slice.call(arguments, 2) : [];
      if (!this.emit.apply(this, [command].concat(slice.call(args)))) {
        if (atomWindow != null) {
          return atomWindow.sendCommand.apply(atomWindow, [command].concat(slice.call(args)));
        } else {
          return this.sendCommandToFirstResponder(command);
        }
      }
    };

    AtomApplication.prototype.sendCommandToFirstResponder = function(command) {
      if (process.platform !== 'darwin') {
        return false;
      }
      switch (command) {
        case 'core:undo':
          Menu.sendActionToFirstResponder('undo:');
          break;
        case 'core:redo':
          Menu.sendActionToFirstResponder('redo:');
          break;
        case 'core:copy':
          Menu.sendActionToFirstResponder('copy:');
          break;
        case 'core:cut':
          Menu.sendActionToFirstResponder('cut:');
          break;
        case 'core:paste':
          Menu.sendActionToFirstResponder('paste:');
          break;
        case 'core:select-all':
          Menu.sendActionToFirstResponder('selectAll:');
          break;
        default:
          return false;
      }
      return true;
    };

    AtomApplication.prototype.openPathOnEvent = function(eventName, pathToOpen) {
      return this.on(eventName, function() {
        var window;
        if (window = this.focusedWindow()) {
          return window.openPath(pathToOpen);
        } else {
          return this.openPath({
            pathToOpen: pathToOpen
          });
        }
      });
    };

    AtomApplication.prototype.windowForPaths = function(pathsToOpen, devMode) {
      return _.find(this.windows, function(atomWindow) {
        return atomWindow.devMode === devMode && atomWindow.containsPaths(pathsToOpen);
      });
    };

    AtomApplication.prototype.atomWindowForEvent = function(arg) {
      var sender;
      sender = arg.sender;
      return this.atomWindowForBrowserWindow(BrowserWindow.fromWebContents(sender));
    };

    AtomApplication.prototype.atomWindowForBrowserWindow = function(browserWindow) {
      return this.windows.find(function(atomWindow) {
        return atomWindow.browserWindow === browserWindow;
      });
    };

    AtomApplication.prototype.focusedWindow = function() {
      return _.find(this.windows, function(atomWindow) {
        return atomWindow.isFocused();
      });
    };

    AtomApplication.prototype.getWindowOffsetForCurrentPlatform = function() {
      var offsetByPlatform, ref2;
      offsetByPlatform = {
        darwin: 22,
        win32: 26
      };
      return (ref2 = offsetByPlatform[process.platform]) != null ? ref2 : 0;
    };

    AtomApplication.prototype.getDimensionsForNewWindow = function() {
      var dimensions, offset, ref2, ref3, ref4, ref5;
      if ((ref2 = (ref3 = this.focusedWindow()) != null ? ref3 : this.lastFocusedWindow) != null ? ref2.isMaximized() : void 0) {
        return;
      }
      dimensions = (ref4 = (ref5 = this.focusedWindow()) != null ? ref5 : this.lastFocusedWindow) != null ? ref4.getDimensions() : void 0;
      offset = this.getWindowOffsetForCurrentPlatform();
      if ((dimensions != null) && (offset != null)) {
        dimensions.x += offset;
        dimensions.y += offset;
      }
      return dimensions;
    };

    AtomApplication.prototype.openPath = function(arg) {
      var addToLastWindow, clearWindowState, devMode, env, initialPaths, newWindow, pathToOpen, pidToKillWhenClosed, profileStartup, ref2, safeMode, window;
      ref2 = arg != null ? arg : {}, initialPaths = ref2.initialPaths, pathToOpen = ref2.pathToOpen, pidToKillWhenClosed = ref2.pidToKillWhenClosed, newWindow = ref2.newWindow, devMode = ref2.devMode, safeMode = ref2.safeMode, profileStartup = ref2.profileStartup, window = ref2.window, clearWindowState = ref2.clearWindowState, addToLastWindow = ref2.addToLastWindow, env = ref2.env;
      return this.openPaths({
        initialPaths: initialPaths,
        pathsToOpen: [pathToOpen],
        pidToKillWhenClosed: pidToKillWhenClosed,
        newWindow: newWindow,
        devMode: devMode,
        safeMode: safeMode,
        profileStartup: profileStartup,
        window: window,
        clearWindowState: clearWindowState,
        addToLastWindow: addToLastWindow,
        env: env
      });
    };

    AtomApplication.prototype.openPaths = function(arg) {
      var addToLastWindow, clearWindowState, currentWindow, devMode, env, executedFrom, existingWindow, initialPaths, locationToOpen, locationsToOpen, newWindow, openedWindow, pathToOpen, pathsToOpen, pidToKillWhenClosed, profileStartup, ref2, resourcePath, safeMode, stats, window, windowDimensions, windowInitializationScript;
      ref2 = arg != null ? arg : {}, initialPaths = ref2.initialPaths, pathsToOpen = ref2.pathsToOpen, executedFrom = ref2.executedFrom, pidToKillWhenClosed = ref2.pidToKillWhenClosed, newWindow = ref2.newWindow, devMode = ref2.devMode, safeMode = ref2.safeMode, windowDimensions = ref2.windowDimensions, profileStartup = ref2.profileStartup, window = ref2.window, clearWindowState = ref2.clearWindowState, addToLastWindow = ref2.addToLastWindow, env = ref2.env;
      if ((pathsToOpen == null) || pathsToOpen.length === 0) {
        return;
      }
      if (env == null) {
        env = process.env;
      }
      devMode = Boolean(devMode);
      safeMode = Boolean(safeMode);
      clearWindowState = Boolean(clearWindowState);
      locationsToOpen = (function() {
        var i, len, results;
        results = [];
        for (i = 0, len = pathsToOpen.length; i < len; i++) {
          pathToOpen = pathsToOpen[i];
          results.push(this.locationForPathToOpen(pathToOpen, executedFrom, addToLastWindow));
        }
        return results;
      }).call(this);
      pathsToOpen = (function() {
        var i, len, results;
        results = [];
        for (i = 0, len = locationsToOpen.length; i < len; i++) {
          locationToOpen = locationsToOpen[i];
          results.push(locationToOpen.pathToOpen);
        }
        return results;
      })();
      if (!(pidToKillWhenClosed || newWindow)) {
        existingWindow = this.windowForPaths(pathsToOpen, devMode);
        stats = (function() {
          var i, len, results;
          results = [];
          for (i = 0, len = pathsToOpen.length; i < len; i++) {
            pathToOpen = pathsToOpen[i];
            results.push(fs.statSyncNoException(pathToOpen));
          }
          return results;
        })();
        if (existingWindow == null) {
          if (currentWindow = window != null ? window : this.lastFocusedWindow) {
            if (addToLastWindow || currentWindow.devMode === devMode && (stats.every(function(stat) {
              return typeof stat.isFile === "function" ? stat.isFile() : void 0;
            }) || stats.some(function(stat) {
              return (typeof stat.isDirectory === "function" ? stat.isDirectory() : void 0) && !currentWindow.hasProjectPath();
            }))) {
              existingWindow = currentWindow;
            }
          }
        }
      }
      if (existingWindow != null) {
        openedWindow = existingWindow;
        openedWindow.openLocations(locationsToOpen);
        if (openedWindow.isMinimized()) {
          openedWindow.restore();
        } else {
          openedWindow.focus();
        }
        openedWindow.replaceEnvironment(env);
      } else {
        if (devMode) {
          try {
            windowInitializationScript = require.resolve(path.join(this.devResourcePath, 'src', 'initialize-application-window'));
            resourcePath = this.devResourcePath;
          } catch (error1) {}
        }
        if (windowInitializationScript == null) {
          windowInitializationScript = require.resolve('../initialize-application-window');
        }
        if (resourcePath == null) {
          resourcePath = this.resourcePath;
        }
        if (windowDimensions == null) {
          windowDimensions = this.getDimensionsForNewWindow();
        }
        openedWindow = new AtomWindow(this, this.fileRecoveryService, {
          initialPaths: initialPaths,
          locationsToOpen: locationsToOpen,
          windowInitializationScript: windowInitializationScript,
          resourcePath: resourcePath,
          devMode: devMode,
          safeMode: safeMode,
          windowDimensions: windowDimensions,
          profileStartup: profileStartup,
          clearWindowState: clearWindowState,
          env: env
        });
        openedWindow.focus();
        this.lastFocusedWindow = openedWindow;
      }
      if (pidToKillWhenClosed != null) {
        this.pidsToOpenWindows[pidToKillWhenClosed] = openedWindow;
      }
      openedWindow.browserWindow.once('closed', (function(_this) {
        return function() {
          return _this.killProcessForWindow(openedWindow);
        };
      })(this));
      return openedWindow;
    };

    AtomApplication.prototype.killAllProcesses = function() {
      var pid;
      for (pid in this.pidsToOpenWindows) {
        this.killProcess(pid);
      }
    };

    AtomApplication.prototype.killProcessForWindow = function(openedWindow) {
      var pid, ref2, trackedWindow;
      ref2 = this.pidsToOpenWindows;
      for (pid in ref2) {
        trackedWindow = ref2[pid];
        if (trackedWindow === openedWindow) {
          this.killProcess(pid);
        }
      }
    };

    AtomApplication.prototype.killProcess = function(pid) {
      var error, parsedPid, ref2;
      try {
        parsedPid = parseInt(pid);
        if (isFinite(parsedPid)) {
          process.kill(parsedPid);
        }
      } catch (error1) {
        error = error1;
        if (error.code !== 'ESRCH') {
          console.log("Killing process " + pid + " failed: " + ((ref2 = error.code) != null ? ref2 : error.message));
        }
      }
      return delete this.pidsToOpenWindows[pid];
    };

    AtomApplication.prototype.saveState = function(allowEmpty) {
      var i, len, ref2, states, window;
      if (allowEmpty == null) {
        allowEmpty = false;
      }
      if (this.quitting) {
        return;
      }
      states = [];
      ref2 = this.windows;
      for (i = 0, len = ref2.length; i < len; i++) {
        window = ref2[i];
        if (!window.isSpec) {
          states.push({
            initialPaths: window.representedDirectoryPaths
          });
        }
      }
      if (states.length > 0 || allowEmpty) {
        this.storageFolder.storeSync('application.json', states);
        return this.emit('application:did-save-state');
      }
    };

    AtomApplication.prototype.loadState = function(options) {
      var i, len, ref2, ref3, results, state, states;
      if (((ref2 = this.config.get('core.restorePreviousWindowsOnStart')) === 'yes' || ref2 === 'always') && ((ref3 = (states = this.storageFolder.load('application.json'))) != null ? ref3.length : void 0) > 0) {
        results = [];
        for (i = 0, len = states.length; i < len; i++) {
          state = states[i];
          results.push(this.openWithOptions(Object.assign(options, {
            initialPaths: state.initialPaths,
            pathsToOpen: state.initialPaths.filter(function(directoryPath) {
              return fs.isDirectorySync(directoryPath);
            }),
            urlsToOpen: [],
            devMode: this.devMode,
            safeMode: this.safeMode
          })));
        }
        return results;
      } else {
        return null;
      }
    };

    AtomApplication.prototype.openUrl = function(arg) {
      var PackageManager, devMode, env, pack, packageName, packagePath, safeMode, urlToOpen, windowDimensions, windowInitializationScript;
      urlToOpen = arg.urlToOpen, devMode = arg.devMode, safeMode = arg.safeMode, env = arg.env;
      if (this.packages == null) {
        PackageManager = require('../package-manager');
        this.packages = new PackageManager({});
        this.packages.initialize({
          configDirPath: process.env.ATOM_HOME,
          devMode: devMode,
          resourcePath: this.resourcePath
        });
      }
      packageName = url.parse(urlToOpen).host;
      pack = _.find(this.packages.getAvailablePackageMetadata(), function(arg1) {
        var name;
        name = arg1.name;
        return name === packageName;
      });
      if (pack != null) {
        if (pack.urlMain) {
          packagePath = this.packages.resolvePackagePath(packageName);
          windowInitializationScript = path.resolve(packagePath, pack.urlMain);
          windowDimensions = this.getDimensionsForNewWindow();
          return new AtomWindow(this, this.fileRecoveryService, {
            windowInitializationScript: windowInitializationScript,
            resourcePath: this.resourcePath,
            devMode: devMode,
            safeMode: safeMode,
            urlToOpen: urlToOpen,
            windowDimensions: windowDimensions,
            env: env
          });
        } else {
          return console.log("Package '" + pack.name + "' does not have a url main: " + urlToOpen);
        }
      } else {
        return console.log("Opening unknown url: " + urlToOpen);
      }
    };

    AtomApplication.prototype.runTests = function(arg) {
      var devMode, env, error, executedFrom, headless, i, isSpec, legacyTestRunnerPath, len, logFile, pathToOpen, pathsToOpen, resourcePath, safeMode, testPaths, testRunnerPath, timeout, timeoutHandler, timeoutInSeconds, windowInitializationScript;
      headless = arg.headless, resourcePath = arg.resourcePath, executedFrom = arg.executedFrom, pathsToOpen = arg.pathsToOpen, logFile = arg.logFile, safeMode = arg.safeMode, timeout = arg.timeout, env = arg.env;
      if (resourcePath !== this.resourcePath && !fs.existsSync(resourcePath)) {
        resourcePath = this.resourcePath;
      }
      timeoutInSeconds = Number.parseFloat(timeout);
      if (!Number.isNaN(timeoutInSeconds)) {
        timeoutHandler = function() {
          console.log("The test suite has timed out because it has been running for more than " + timeoutInSeconds + " seconds.");
          return process.exit(124);
        };
        setTimeout(timeoutHandler, timeoutInSeconds * 1000);
      }
      try {
        windowInitializationScript = require.resolve(path.resolve(this.devResourcePath, 'src', 'initialize-test-window'));
      } catch (error1) {
        error = error1;
        windowInitializationScript = require.resolve(path.resolve(__dirname, '..', '..', 'src', 'initialize-test-window'));
      }
      testPaths = [];
      if (pathsToOpen != null) {
        for (i = 0, len = pathsToOpen.length; i < len; i++) {
          pathToOpen = pathsToOpen[i];
          testPaths.push(path.resolve(executedFrom, fs.normalize(pathToOpen)));
        }
      }
      if (testPaths.length === 0) {
        process.stderr.write('Error: Specify at least one test path\n\n');
        process.exit(1);
      }
      legacyTestRunnerPath = this.resolveLegacyTestRunnerPath();
      testRunnerPath = this.resolveTestRunnerPath(testPaths[0]);
      devMode = true;
      isSpec = true;
      if (safeMode == null) {
        safeMode = false;
      }
      return new AtomWindow(this, this.fileRecoveryService, {
        windowInitializationScript: windowInitializationScript,
        resourcePath: resourcePath,
        headless: headless,
        isSpec: isSpec,
        devMode: devMode,
        testRunnerPath: testRunnerPath,
        legacyTestRunnerPath: legacyTestRunnerPath,
        testPaths: testPaths,
        logFile: logFile,
        safeMode: safeMode,
        env: env
      });
    };

    AtomApplication.prototype.runBenchmarks = function(arg) {
      var benchmarkPaths, devMode, env, error, executedFrom, headless, i, isSpec, len, pathToOpen, pathsToOpen, resourcePath, safeMode, test, windowInitializationScript;
      headless = arg.headless, test = arg.test, resourcePath = arg.resourcePath, executedFrom = arg.executedFrom, pathsToOpen = arg.pathsToOpen, env = arg.env;
      if (resourcePath !== this.resourcePath && !fs.existsSync(resourcePath)) {
        resourcePath = this.resourcePath;
      }
      try {
        windowInitializationScript = require.resolve(path.resolve(this.devResourcePath, 'src', 'initialize-benchmark-window'));
      } catch (error1) {
        error = error1;
        windowInitializationScript = require.resolve(path.resolve(__dirname, '..', '..', 'src', 'initialize-benchmark-window'));
      }
      benchmarkPaths = [];
      if (pathsToOpen != null) {
        for (i = 0, len = pathsToOpen.length; i < len; i++) {
          pathToOpen = pathsToOpen[i];
          benchmarkPaths.push(path.resolve(executedFrom, fs.normalize(pathToOpen)));
        }
      }
      if (benchmarkPaths.length === 0) {
        process.stderr.write('Error: Specify at least one benchmark path.\n\n');
        process.exit(1);
      }
      devMode = true;
      isSpec = true;
      safeMode = false;
      return new AtomWindow(this, this.fileRecoveryService, {
        windowInitializationScript: windowInitializationScript,
        resourcePath: resourcePath,
        headless: headless,
        test: test,
        isSpec: isSpec,
        devMode: devMode,
        benchmarkPaths: benchmarkPaths,
        safeMode: safeMode,
        env: env
      });
    };

    AtomApplication.prototype.resolveTestRunnerPath = function(testPath) {
      var packageMetadata, packageRoot, testRunnerPath;
      if (FindParentDir == null) {
        FindParentDir = require('find-parent-dir');
      }
      if (packageRoot = FindParentDir.sync(testPath, 'package.json')) {
        packageMetadata = require(path.join(packageRoot, 'package.json'));
        if (packageMetadata.atomTestRunner) {
          if (Resolve == null) {
            Resolve = require('resolve');
          }
          if (testRunnerPath = Resolve.sync(packageMetadata.atomTestRunner, {
            basedir: packageRoot,
            extensions: Object.keys(require.extensions)
          })) {
            return testRunnerPath;
          } else {
            process.stderr.write("Error: Could not resolve test runner path '" + packageMetadata.atomTestRunner + "'");
            process.exit(1);
          }
        }
      }
      return this.resolveLegacyTestRunnerPath();
    };

    AtomApplication.prototype.resolveLegacyTestRunnerPath = function() {
      var error;
      try {
        return require.resolve(path.resolve(this.devResourcePath, 'spec', 'jasmine-test-runner'));
      } catch (error1) {
        error = error1;
        return require.resolve(path.resolve(__dirname, '..', '..', 'spec', 'jasmine-test-runner'));
      }
    };

    AtomApplication.prototype.locationForPathToOpen = function(pathToOpen, executedFrom, forceAddToWindow) {
      var initialColumn, initialLine, match;
      if (executedFrom == null) {
        executedFrom = '';
      }
      if (!pathToOpen) {
        return {
          pathToOpen: pathToOpen
        };
      }
      pathToOpen = pathToOpen.replace(/[:\s]+$/, '');
      match = pathToOpen.match(LocationSuffixRegExp);
      if (match != null) {
        pathToOpen = pathToOpen.slice(0, -match[0].length);
        if (match[1]) {
          initialLine = Math.max(0, parseInt(match[1].slice(1)) - 1);
        }
        if (match[2]) {
          initialColumn = Math.max(0, parseInt(match[2].slice(1)) - 1);
        }
      } else {
        initialLine = initialColumn = null;
      }
      if (url.parse(pathToOpen).protocol == null) {
        pathToOpen = path.resolve(executedFrom, fs.normalize(pathToOpen));
      }
      return {
        pathToOpen: pathToOpen,
        initialLine: initialLine,
        initialColumn: initialColumn,
        forceAddToWindow: forceAddToWindow
      };
    };

    AtomApplication.prototype.promptForPathToOpen = function(type, arg, path) {
      var devMode, safeMode, window;
      devMode = arg.devMode, safeMode = arg.safeMode, window = arg.window;
      if (path == null) {
        path = null;
      }
      return this.promptForPath(type, ((function(_this) {
        return function(pathsToOpen) {
          return _this.openPaths({
            pathsToOpen: pathsToOpen,
            devMode: devMode,
            safeMode: safeMode,
            window: window
          });
        };
      })(this)), path);
    };

    AtomApplication.prototype.promptForPath = function(type, callback, path) {
      var openOptions, parentWindow, properties;
      properties = (function() {
        switch (type) {
          case 'file':
            return ['openFile'];
          case 'folder':
            return ['openDirectory'];
          case 'all':
            return ['openFile', 'openDirectory'];
          default:
            throw new Error(type + " is an invalid type for promptForPath");
        }
      })();
      parentWindow = process.platform === 'darwin' ? null : BrowserWindow.getFocusedWindow();
      openOptions = {
        properties: properties.concat(['multiSelections', 'createDirectory']),
        title: (function() {
          switch (type) {
            case 'file':
              return 'Open File';
            case 'folder':
              return 'Open Folder';
            default:
              return 'Open';
          }
        })()
      };
      if (path != null) {
        openOptions.defaultPath = path;
      }
      return dialog.showOpenDialog(parentWindow, openOptions, callback);
    };

    AtomApplication.prototype.promptForRestart = function() {
      var chosen;
      chosen = dialog.showMessageBox(BrowserWindow.getFocusedWindow(), {
        type: 'warning',
        title: 'Restart required',
        message: "You will need to restart Atom for this change to take effect.",
        buttons: ['Restart Atom', 'Cancel']
      });
      if (chosen === 0) {
        return this.restart();
      }
    };

    AtomApplication.prototype.restart = function() {
      var args;
      args = [];
      if (this.safeMode) {
        args.push("--safe");
      }
      if (this.logFile != null) {
        args.push("--log-file=" + this.logFile);
      }
      if (this.socketPath != null) {
        args.push("--socket-path=" + this.socketPath);
      }
      if (this.userDataDir != null) {
        args.push("--user-data-dir=" + this.userDataDir);
      }
      if (this.devMode) {
        args.push('--dev');
        args.push("--resource-path=" + this.resourcePath);
      }
      app.relaunch({
        args: args
      });
      return app.quit();
    };

    AtomApplication.prototype.disableZoomOnDisplayChange = function() {
      var outerCallback;
      outerCallback = (function(_this) {
        return function() {
          var i, len, ref2, results, window;
          ref2 = _this.windows;
          results = [];
          for (i = 0, len = ref2.length; i < len; i++) {
            window = ref2[i];
            results.push(window.disableZoom());
          }
          return results;
        };
      })(this);
      screen.on('display-added', outerCallback);
      screen.on('display-removed', outerCallback);
      return new Disposable(function() {
        screen.removeListener('display-added', outerCallback);
        return screen.removeListener('display-removed', outerCallback);
      });
    };

    return AtomApplication;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
