(function() {
  var CompositeDisposable, IgnoredCommands, PathRE, Reporter, fs, grim, path, stripPath;

  CompositeDisposable = require('atom').CompositeDisposable;

  path = require('path');

  Reporter = require('./reporter');

  fs = require('fs-plus');

  grim = require('grim');

  IgnoredCommands = {
    'vim-mode:move-up': true,
    'vim-mode:move-down': true,
    'vim-mode:move-left': true,
    'vim-mode:move-right': true
  };

  module.exports = {
    activate: function(arg) {
      var sessionLength;
      sessionLength = arg.sessionLength;
      this.subscriptions = new CompositeDisposable;
      this.shouldIncludePanesAndCommands = Math.random() < 0.05;
      return this.ensureClientId((function(_this) {
        return function() {
          return _this.begin(sessionLength);
        };
      })(this));
    },
    deactivate: function() {
      var ref;
      return (ref = this.subscriptions) != null ? ref.dispose() : void 0;
    },
    serialize: function() {
      return {
        sessionLength: Date.now() - this.sessionStart
      };
    },
    provideReporter: function() {
      return {
        sendEvent: Reporter.sendEvent.bind(Reporter),
        sendTiming: Reporter.sendTiming.bind(Reporter),
        sendException: Reporter.sendException.bind(Reporter)
      };
    },
    begin: function(sessionLength) {
      this.sessionStart = Date.now();
      if (sessionLength) {
        Reporter.sendEvent('window', 'ended', null, sessionLength);
      }
      Reporter.sendEvent('window', 'started');
      this.subscriptions.add(atom.onDidThrowError(function(event) {
        var errorMessage;
        errorMessage = event;
        if (typeof event !== 'string') {
          errorMessage = event.message;
        }
        errorMessage = stripPath(errorMessage) || 'Unknown';
        errorMessage = errorMessage.replace('Uncaught ', '').slice(0, 150);
        return Reporter.sendException(errorMessage);
      }));
      this.subscriptions.add(atom.textEditors.observe(function(editor) {
        var grammar;
        grammar = editor.getGrammar();
        if (grammar) {
          return Reporter.sendEvent('file', 'open', grammar.scopeName);
        }
      }));
      this.subscriptions.add(atom.config.onDidChange('core.telemetryConsent', function(arg) {
        var newValue, oldValue;
        newValue = arg.newValue, oldValue = arg.oldValue;
        if (newValue !== 'undecided') {
          return Reporter.sendEvent('setting', 'core.telemetryConsent', newValue);
        }
      }));
      this.watchPaneItems();
      this.watchCommands();
      this.watchDeprecations();
      if (atom.getLoadSettings().shellLoadTime != null) {
        Reporter.sendTiming('shell', 'load', atom.getLoadSettings().shellLoadTime);
      }
      return process.nextTick(function() {
        return Reporter.sendTiming('core', 'load', atom.getWindowLoadTime());
      });
    },
    ensureClientId: function(callback) {
      if (localStorage.getItem('metrics.userId')) {
        return callback();
      } else if (atom.config.get('metrics.userId')) {
        localStorage.setItem('metrics.userId', atom.config.get('metrics.userId'));
        return callback();
      } else {
        return this.createClientId(function(clientId) {
          localStorage.setItem('metrics.userId', clientId);
          return callback();
        });
      }
    },
    createClientId: function(callback) {
      return callback(require('node-uuid').v4());
    },
    getClientId: function() {
      return localStorage.getItem('metrics.userId');
    },
    watchPaneItems: function() {
      return this.subscriptions.add(atom.workspace.onDidAddPaneItem((function(_this) {
        return function(arg) {
          var item;
          item = arg.item;
          if (!_this.shouldIncludePanesAndCommands) {
            return;
          }
          return Reporter.sendPaneItem(item);
        };
      })(this)));
    },
    watchCommands: function() {
      return this.subscriptions.add(atom.commands.onWillDispatch((function(_this) {
        return function(commandEvent) {
          var eventName, ref;
          if (!_this.shouldIncludePanesAndCommands) {
            return;
          }
          eventName = commandEvent.type;
          if ((ref = commandEvent.detail) != null ? ref.jQueryTrigger : void 0) {
            return;
          }
          if (eventName.startsWith('core:') || eventName.startsWith('editor:')) {
            return;
          }
          if (!(eventName.indexOf(':') > -1)) {
            return;
          }
          if (eventName in IgnoredCommands) {
            return;
          }
          return Reporter.sendCommand(eventName);
        };
      })(this)));
    },
    watchDeprecations: function() {
      this.deprecationCache = {};
      this.packageVersionCache = {};
      atom.packages.onDidActivateInitialPackages((function(_this) {
        return function() {
          var j, len, pack, packages, ref;
          packages = atom.packages.getLoadedPackages();
          for (j = 0, len = packages.length; j < len; j++) {
            pack = packages[j];
            _this.packageVersionCache[pack.name] = (pack != null ? (ref = pack.metadata) != null ? ref.version : void 0 : void 0) || 'unknown';
          }
          setImmediate(function() {
            var deprecation, k, len1, ref1;
            ref1 = grim.getDeprecations();
            for (k = 0, len1 = ref1.length; k < len1; k++) {
              deprecation = ref1[k];
              _this.reportDeprecation(deprecation);
            }
          });
        };
      })(this));
      atom.packages.onDidLoadPackage((function(_this) {
        return function(pack) {
          var ref;
          if (!_this.packageVersionCache[pack.name]) {
            return _this.packageVersionCache[pack.name] = (pack != null ? (ref = pack.metadata) != null ? ref.version : void 0 : void 0) || 'unknown';
          }
        };
      })(this));
      return grim.on('updated', (function(_this) {
        return function(deprecation) {
          return setImmediate(function() {
            return _this.reportDeprecation(deprecation);
          });
        };
      })(this));
    },
    reportDeprecation: function(deprecation) {
      var __, message, nameAndVersion, pack, packageName, ref, ref1, ref2, ref3, stack, version;
      message = deprecation.getMessage().slice(0, 500);
      ref = deprecation.stacks;
      for (__ in ref) {
        stack = ref[__];
        packageName = (ref1 = (ref2 = stack.metadata) != null ? ref2.packageName : void 0) != null ? ref1 : (this.getPackageName(stack) || '').toLowerCase();
        if (!packageName) {
          continue;
        }
        if (!this.packageVersionCache[packageName]) {
          pack = atom.packages.getLoadedPackage(packageName);
          this.packageVersionCache[packageName] = (pack != null ? (ref3 = pack.metadata) != null ? ref3.version : void 0 : void 0) || 'unknown';
        }
        version = this.packageVersionCache[packageName];
        nameAndVersion = packageName + "@" + version;
        if (this.deprecationCache[nameAndVersion + message] == null) {
          this.deprecationCache[nameAndVersion + message] = true;
          Reporter.sendEvent('deprecation-v3', nameAndVersion, message);
        }
      }
    },
    getFileNameFromCallSite: function(callsite) {
      var ref;
      return (ref = callsite.fileName) != null ? ref : callsite.getFileName();
    },
    getPackageName: function(stack) {
      var fileName, i, j, packageName, packagePath, packagePaths, ref, relativePath;
      packagePaths = this.getPackagePathsByPackageName();
      for (i = j = 1, ref = stack.length; 1 <= ref ? j < ref : j > ref; i = 1 <= ref ? ++j : --j) {
        fileName = this.getFileNameFromCallSite(stack[i]);
        if (!fileName) {
          return;
        }
        if (fileName.includes(path.sep + "node_modules" + path.sep)) {
          continue;
        }
        for (packageName in packagePaths) {
          packagePath = packagePaths[packageName];
          relativePath = path.relative(packagePath, fileName);
          if (!/^\.\./.test(relativePath)) {
            return packageName;
          }
        }
        if (atom.getUserInitScriptPath() === fileName) {
          return "init-script";
        }
      }
    },
    getPackagePathsByPackageName: function() {
      var j, len, pack, ref;
      if (this.packagePathsByPackageName != null) {
        return this.packagePathsByPackageName;
      }
      this.packagePathsByPackageName = {};
      ref = atom.packages.getLoadedPackages();
      for (j = 0, len = ref.length; j < len; j++) {
        pack = ref[j];
        this.packagePathsByPackageName[pack.name] = pack.path;
        if (pack.path.indexOf('.atom/dev/packages') > -1 || pack.path.indexOf('.atom/packages') > -1) {
          this.packagePathsByPackageName[pack.name] = fs.absolute(pack.path);
        }
      }
      return this.packagePathsByPackageName;
    }
  };

  PathRE = /'?((\/|\\|[a-z]:\\)[^\s']+)+'?/ig;

  stripPath = function(message) {
    return message.replace(PathRE, '<path>');
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
