(function() {
  var Highlights, _, cheerio, convertCodeBlocksToAtomEditors, fs, highlighter, packagePath, path, render, resolveImagePaths, resourcePath, roaster, sanitize, scopeForFenceName, tokenizeCodeBlocks;

  path = require('path');

  _ = require('underscore-plus');

  cheerio = require('cheerio');

  fs = require('fs-plus');

  Highlights = require('highlights');

  roaster = null;

  scopeForFenceName = require('./extension-helper').scopeForFenceName;

  highlighter = null;

  resourcePath = atom.getLoadSettings().resourcePath;

  packagePath = path.dirname(__dirname);

  exports.toDOMFragment = function(text, filePath, grammar, callback) {
    if (text == null) {
      text = '';
    }
    return render(text, filePath, function(error, html) {
      var defaultCodeLanguage, domFragment, template;
      if (error != null) {
        return callback(error);
      }
      template = document.createElement('template');
      template.innerHTML = html;
      domFragment = template.content.cloneNode(true);
      if ((grammar != null ? grammar.scopeName : void 0) === 'source.litcoffee') {
        defaultCodeLanguage = 'coffee';
      }
      convertCodeBlocksToAtomEditors(domFragment, defaultCodeLanguage);
      return callback(null, domFragment);
    });
  };

  exports.toHTML = function(text, filePath, grammar, callback) {
    if (text == null) {
      text = '';
    }
    return render(text, filePath, function(error, html) {
      var defaultCodeLanguage;
      if (error != null) {
        return callback(error);
      }
      if ((grammar != null ? grammar.scopeName : void 0) === 'source.litcoffee') {
        defaultCodeLanguage = 'coffee';
      }
      html = tokenizeCodeBlocks(html, defaultCodeLanguage);
      return callback(null, html);
    });
  };

  render = function(text, filePath, callback) {
    var options;
    if (roaster == null) {
      roaster = require('roaster');
    }
    options = {
      sanitize: false,
      breaks: atom.config.get('markdown-preview.breakOnSingleNewline')
    };
    text = text.replace(/^\s*<!doctype(\s+.*)?>\s*/i, '');
    return roaster(text, options, function(error, html) {
      if (error != null) {
        return callback(error);
      }
      html = sanitize(html);
      html = resolveImagePaths(html, filePath);
      return callback(null, html.trim());
    });
  };

  sanitize = function(html) {
    var attribute, attributesToRemove, i, len, o;
    o = cheerio.load(html);
    o('script').remove();
    attributesToRemove = ['onabort', 'onblur', 'onchange', 'onclick', 'ondbclick', 'onerror', 'onfocus', 'onkeydown', 'onkeypress', 'onkeyup', 'onload', 'onmousedown', 'onmousemove', 'onmouseover', 'onmouseout', 'onmouseup', 'onreset', 'onresize', 'onscroll', 'onselect', 'onsubmit', 'onunload'];
    for (i = 0, len = attributesToRemove.length; i < len; i++) {
      attribute = attributesToRemove[i];
      o('*').removeAttr(attribute);
    }
    return o.html();
  };

  resolveImagePaths = function(html, filePath) {
    var i, img, imgElement, len, o, ref, rootDirectory, src;
    rootDirectory = atom.project.relativizePath(filePath)[0];
    o = cheerio.load(html);
    ref = o('img');
    for (i = 0, len = ref.length; i < len; i++) {
      imgElement = ref[i];
      img = o(imgElement);
      if (src = img.attr('src')) {
        if (src.match(/^(https?|atom):\/\//)) {
          continue;
        }
        if (src.startsWith(process.resourcesPath)) {
          continue;
        }
        if (src.startsWith(resourcePath)) {
          continue;
        }
        if (src.startsWith(packagePath)) {
          continue;
        }
        if (src[0] === '/') {
          if (!fs.isFileSync(src)) {
            if (rootDirectory) {
              img.attr('src', path.join(rootDirectory, src.substring(1)));
            }
          }
        } else {
          img.attr('src', path.resolve(path.dirname(filePath), src));
        }
      }
    }
    return o.html();
  };

  convertCodeBlocksToAtomEditors = function(domFragment, defaultLanguage) {
    var codeBlock, codeElement, cursorLineDecoration, editor, editorElement, fenceName, fontFamily, grammar, i, j, k, len, len1, len2, preElement, ref, ref1, ref2, ref3, ref4, ref5;
    if (defaultLanguage == null) {
      defaultLanguage = 'text';
    }
    if (fontFamily = atom.config.get('editor.fontFamily')) {
      ref = domFragment.querySelectorAll('code');
      for (i = 0, len = ref.length; i < len; i++) {
        codeElement = ref[i];
        codeElement.style.fontFamily = fontFamily;
      }
    }
    ref1 = domFragment.querySelectorAll('pre');
    for (j = 0, len1 = ref1.length; j < len1; j++) {
      preElement = ref1[j];
      codeBlock = (ref2 = preElement.firstElementChild) != null ? ref2 : preElement;
      fenceName = (ref3 = (ref4 = codeBlock.getAttribute('class')) != null ? ref4.replace(/^lang-/, '') : void 0) != null ? ref3 : defaultLanguage;
      editorElement = document.createElement('atom-text-editor');
      editorElement.setAttributeNode(document.createAttribute('gutter-hidden'));
      editorElement.removeAttribute('tabindex');
      preElement.parentNode.insertBefore(editorElement, preElement);
      preElement.remove();
      editor = editorElement.getModel();
      editor.setText(codeBlock.textContent);
      if (grammar = atom.grammars.grammarForScopeName(scopeForFenceName(fenceName))) {
        editor.setGrammar(grammar);
      }
      if (editor.cursorLineDecorations != null) {
        ref5 = editor.cursorLineDecorations;
        for (k = 0, len2 = ref5.length; k < len2; k++) {
          cursorLineDecoration = ref5[k];
          cursorLineDecoration.destroy();
        }
      } else {
        editor.getDecorations({
          "class": 'cursor-line',
          type: 'line'
        })[0].destroy();
      }
    }
    return domFragment;
  };

  tokenizeCodeBlocks = function(html, defaultLanguage) {
    var codeBlock, fenceName, fontFamily, highlightedBlock, highlightedHtml, i, len, o, preElement, ref, ref1, ref2;
    if (defaultLanguage == null) {
      defaultLanguage = 'text';
    }
    o = cheerio.load(html);
    if (fontFamily = atom.config.get('editor.fontFamily')) {
      o('code').css('font-family', fontFamily);
    }
    ref = o("pre");
    for (i = 0, len = ref.length; i < len; i++) {
      preElement = ref[i];
      codeBlock = o(preElement).children().first();
      fenceName = (ref1 = (ref2 = codeBlock.attr('class')) != null ? ref2.replace(/^lang-/, '') : void 0) != null ? ref1 : defaultLanguage;
      if (highlighter == null) {
        highlighter = new Highlights({
          registry: atom.grammars,
          scopePrefix: 'syntax--'
        });
      }
      highlightedHtml = highlighter.highlightSync({
        fileContents: codeBlock.text(),
        scopeName: scopeForFenceName(fenceName)
      });
      highlightedBlock = o(highlightedHtml);
      highlightedBlock.removeClass('editor').addClass("lang-" + fenceName);
      o(preElement).replaceWith(highlightedBlock);
    }
    return o.html();
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
