#! /bin/bash
#

#### Check to see if OpenRC is PID 1 - otherwise just use sysvinit #### 
#

if [ $(ps -p1 | grep -ic openrc) -eq 1 ] || [ $(cat /proc/cmdline | grep -ic "openrc") -eq 1 ] || ( [ $(cat /proc/cmdline | grep -ic "init=/") -eq 0 ] && [ $(readlink -e /sbin/init | grep -ic "openrc") -eq 1 ] ); then

#######################################################################
#-----Contents of openRC's original rc script below-----

#! /bin/sh
# Wrapper of OpenRC called from inittab

set -e

# map numbered to named runlevel
case $1 in
	0 )
		/usr/sbin/openrc off && /usr/sbin/openrc shutdown && \
			/etc/init.d/halt stop
		;;
	1 )
		exec /usr/sbin/openrc recovery
		;;
	6 )
		/usr/sbin/openrc off && /usr/sbin/openrc reboot && \
			/etc/init.d/reboot stop
		;;
	* )
		exec /usr/sbin/openrc default
		;;
esac
else
#######################################################################
#-----Contents of syvinit's original rc script below-----

#! /bin/sh
#
# rc
#
# Starts/stops services on runlevel changes.
#
# Authors:
# 	Miquel van Smoorenburg <miquels@cistron.nl>
# 	Bruce Perens <Bruce@Pixar.com>

PATH=/sbin:/usr/sbin:/bin:/usr/bin
export PATH

# Un-comment the following for interactive debugging. Do not un-comment
# this for debugging a real boot process as no scripts will be executed.

# Make sure the name survive changing the argument list
scriptname="$0"

umask 022

on_exit() {
}
trap on_exit EXIT # Enable emergency handler

# Ignore CTRL-C only in this shell, so we can interrupt subprocesses.
trap ":" INT QUIT TSTP

# Set onlcr to avoid staircase effect.
stty onlcr 0>&1

# Now find out what the current and what the previous runlevel are.

# shellcheck disable=SC2153 # this variable is set by /sbin/init
runlevel=$RUNLEVEL

# Get first argument. Set new runlevel to this argument.
[ "$1" != "" ] && runlevel=$1
if [ "$runlevel" = "" ]
then
	exit 1
fi
previous=$PREVLEVEL
[ "$previous" = "" ] && previous=N

export runlevel previous

if [ -f /etc/default/rcS ] ; then
	. /etc/default/rcS
fi
export VERBOSE

. /lib/lsb/init-functions

#
# Check if we are able to use make like booting.  It require the
# insserv package to be enabled. Boot concurrency also requires
# startpar to be installed.
#
CONCURRENCY=makefile
test -s /etc/init.d/.depend.boot  || CONCURRENCY="none"
test -s /etc/init.d/.depend.start || CONCURRENCY="none"
test -s /etc/init.d/.depend.stop  || CONCURRENCY="none"
if test -e /etc/init.d/.legacy-bootordering; then
	CONCURRENCY="none"
fi
if grep -wqs concurrency=none /proc/cmdline; then
	CONCURRENCY="none"
fi
if ! test -e /proc/stat; then
	# startpar requires /proc/stat
	if [ "$(uname)" = "GNU/kFreeBSD" ] ; then
		mount -t linprocfs linprocfs /proc
	elif [ "$(uname)" = "GNU" ] ; then
		mount -t proc none /proc -ocompatible
	else
		mount -t proc none /proc
	fi
fi

#
# Start script or program.
#
case "$CONCURRENCY" in
	makefile|startpar|shell) # startpar and shell are obsolete
		CONCURRENCY=makefile
		log_action_msg "Using makefile-style concurrent boot in runlevel $runlevel"
		startup() {
			eval "$(startpar -p 4 -t 20 -T 3 -M $1 -P $previous -R $runlevel)"

			if [ -n "$failed_service" ]
			then
				log_failure_msg "startpar: service(s) returned failure: $failed_service"
			fi

			if [ -n "$skipped_service_not_installed" ]
			then
				log_warning_msg "startpar: service(s) skipped, program is not installed: $skipped_service_not_installed"
			fi

			if [ -n "$skipped_service_not_configured" ]
			then
				log_warning_msg "startpar: service(s) skipped, program is not configured: $skipped_service_not_configured"
			fi

			unset failed_service skipped_service_not_installed skipped_service_not_configured
		}
		;;
	none|*)
		startup() {
			action=$1
			shift
			for script ; do
				"$script" "$action"
			done
		}
		;;
esac

# Is there an rc directory for this new runlevel?
if [ -d "/etc/rc$runlevel.d" ]
then
	case "$runlevel" in
		0|6)
			ACTION=stop
			;;
		S)
			ACTION=start
			;;
		*)
			ACTION=start
			;;
	esac

	# First, run the KILL scripts.
	if [ makefile = "$CONCURRENCY" ]
	then
		if [ "$ACTION" = "start" ] && [ "$previous" != N ]
		then
			startup stop
		fi
	elif [ "$previous" != N ]
	then
		CURLEVEL=""
		for s in "/etc/rc$runlevel.d/K"*
		do
			# Extract order value from symlink
			level=${s#/etc/rc$runlevel.d/K}
			level=${level%%[a-zA-Z]*}
			if [ "$level" = "$CURLEVEL" ]
			then
				continue
			fi
			CURLEVEL=$level
			SCRIPTS=""
			for i in "/etc/rc$runlevel.d/K$level"*
			do
				# Check if the script is there.
				[ ! -f "$i" ] && continue

				# Stop the service.
				SCRIPTS="$SCRIPTS $i"
			done
			startup stop $SCRIPTS
		done
	fi

	if [ makefile = "$CONCURRENCY" ]
	then
		if [ S = "$runlevel" ]
		then
			startup boot
		else
			startup $ACTION
		fi
	else
		# Now run the START scripts for this runlevel.
		CURLEVEL=""
		for s in "/etc/rc$runlevel.d/S"*
		do
			# Extract order value from symlink
			level=${s#/etc/rc$runlevel.d/S}
			level=${level%%[a-zA-Z]*}
			if [ "$level" = "$CURLEVEL" ]
			then
				continue
			fi
			CURLEVEL=$level
			SCRIPTS=""
			for i in "/etc/rc$runlevel.d/S$level"*
			do
				[ ! -f "$i" ] && continue

				SCRIPTS="$SCRIPTS $i"
			done
			startup $ACTION $SCRIPTS
		done
	fi
fi

trap - EXIT # Disable emergency handler
fi
exit 0
