#!/usr/bin/env python3

"""newday_custom

Create custom NewDay theme with accent color of choice.

"""

__author__ = "fladd"
__version__ = "0.3.0"
__date__ = "2023-04-14"


import os
import sys
import shutil
import argparse
import fileinput


# Parse arguments
parser = argparse.ArgumentParser(
        description="Create custom NewDay theme with accent color of choice.")

parser.add_argument(
        "-V", "--version",
        action="version",
        version=__version__,
        help="show version information and exit")
parser.add_argument(
        "-c", "--color",
        required=False,
        help="the accent color as 6-digit HEX string (default='5294E2')",
        default="5294E2")
parser.add_argument(
        "-o", "--output-directory",
        required=False,
        help="the directory to save the custom theme to \n" +\
              "(default='~/.icewm/themes/NewDay-Custom')")
parser.add_argument(
        "-l", "--look",
        choices=["flat", "3d"],
        default="flat",
        help="the theme look (default='flat')")
parser.add_argument(
        "-m", "--mode",
        choices=["dark", "light"],
        default="dark",
        help="the theme mode (default='dark')")
parser.add_argument(
        "-v", "--variant",
        choices=["small", "medium", "large", "huge"],
        default="medium",
        help="the theme variant to customize (default='medium')")
parser.add_argument(
        "-a", "--application-icon",
        action="store_true",
        help="show application icon in window titlebar")
parser.add_argument(
        "-d", "--desktop-alternative",
        action="store_true",
        help="use alternative 'desktop' icon")
parser.add_argument(
        "-s", "--start-alternative",
        action="store_true",
        help="use alternative 'start' icon")
parser.add_argument(
        "-t", "--taskbuttonactive-alternative",
        action="store_true",
        help="use alternative 'taskbuttonactive' icon")
args = parser.parse_args()

if len(args.color) != 6:
    parser.print_help()
    sys.exit()
for x in args.color.upper():
    if x not in "0123456789ABCDEF":
        parser.print_help()
        sys.exit()

# Get source and output directory
source_dir = os.path.join(os.path.abspath(os.path.split(__file__)[0]),
                          f"NewDay-{args.variant.capitalize()}")

if args.output_directory is None:
    output_dir = os.path.expanduser("~/.icewm/themes/NewDay-Custom")
else:
    output_dir = os.path.abspath(args.output_directory)

# Copy theme files and replace colors
shutil.copytree(source_dir, output_dir, dirs_exist_ok=True)

files = ("default.theme",
         "closeA.xpm",
         "maximizeA.xpm",
         "menuButtonA.xpm",
         "menusel.xpm",
         "minimizeA.xpm",
         "restoreA.xpm",
         "titleAB.xpm",
         "titleAS.xpm",
         "titleAT.xpm",
         "taskbar/desktop.xpm",
         "taskbar/desktop_alternative.xpm",
         "taskbar/start.xpm",
         "taskbar/start_alternative.xpm",
         "taskbar/taskbuttonactive.xpm",
         "taskbar/taskbuttonactive_alternative.xpm")

for file in files:
    # replace accent color
    with open(os.path.join(output_dir, file)) as f:
        file_content = f.read()
    file_content = file_content.replace('5294E2', args.color)

    # replace text color from white to black, based on accent color
    rgb = tuple(int(args.color[i:i+2], 16) for i in (0, 2, 4))
    if rgb[0] * 0.299 + rgb[1] * 0.587 + rgb[2] * 0.144 > 186:
        if file == "default.theme":
            file_content = file_content.replace(
                    'ColorActiveTitleBarText="#FFFFFF"',
                    'ColorActiveTitleBarText="#000000"')
            file_content = file_content.replace(
                    'ColorActiveMenuItemText="#FFFFFF"',
                    'ColorActiveMenuItemText="#000000"')
            file_content = file_content.replace(
                    'ColorActiveWorkspaceButtonText="#FFFFFF"',
                    'ColorActiveWorkspaceButtonText="#000000"')
            file_content = file_content.replace(
                    'ColorInputSelectionText="#FFFFFF"',
                    'ColorInputSelectionText="#000000"')
            file_content = file_content.replace(
                    'ColorListBoxSelectionText="#FFFFFF"',
                    'ColorListBoxSelectionText="#000000"')
            file_content = file_content.replace(
                    'ColorQuickSwitchSelectionText="#FFFFFF"',
                    'ColorQuickSwitchSelectionText="#000000"')
            if args.taskbuttonactive_alternative:
                file_content = file_content.replace(
                        'ColorActiveTaskBarAppText="#FFFFFF"',
                        'ColorActiveTaskBarAppText="#000000"')

        if file in ("maximizeA.xpm", "minimizeA.xpm", "restoreA.xpm"):
            file_content = file_content.replace('FFFFFF', '000000')

    if file == "default.theme":
        if args.look == "3d":
            file_content = file_content.replace(
                    'Look=flat',
                    'Look=metal')
        if args.application_icon:
            file_content = file_content.replace(
                    'ShowMenuButtonIcon=0',
                    'ShowMenuButtonIcon=1')

    # write changes
    with open(os.path.join(output_dir, file), 'w') as f:
        f.write(file_content)

# Set alternative items if requested
if args.desktop_alternative:
    shutil.move(os.path.join(output_dir, "taskbar/desktop_alternative.xpm"),
                os.path.join(output_dir, "taskbar/desktop.xpm"))
else:
    os.remove(os.path.join(output_dir, "taskbar/desktop_alternative.xpm"))
if args.start_alternative:
    shutil.move(os.path.join(output_dir, "taskbar/start_alternative.xpm"),
                os.path.join(output_dir, "taskbar/start.xpm"))
else:
    os.remove(os.path.join(output_dir, "taskbar/start_alternative.xpm"))
if args.taskbuttonactive_alternative:
    shutil.move(os.path.join(output_dir,
                             "taskbar/taskbuttonactive_alternative.xpm"),
                os.path.join(output_dir, "taskbar/taskbuttonactive.xpm"))
else:
    os.remove(os.path.join(output_dir,
                           "taskbar/taskbuttonactive_alternative.xpm"))
os.remove(os.path.join(output_dir, "taskbar/how-to-use.txt"))

# Handle light mode
if args.mode == "light":
    files = ("default.theme",
             "closeI.xpm",
             "closeO.xpm",
             "maximizeI.xpm",
             "maximizeO.xpm",
             "menuButtonI.xpm",
             "minimizeI.xpm",
             "minimizeO.xpm",
             "restoreI.xpm",
             "restoreO.xpm",
             "titleIB.xpm",
             "titleIS.xpm",
             "titleIT.xpm",
             "taskbar/collapse.xpm",
             "taskbar/desktop.xpm",
             "taskbar/expand.xpm",
             "taskbar/taskbuttonactive.xpm",
             "taskbar/workspacebuttonactive.xpm")

    for file in files:
        # replace colors
        with open(os.path.join(output_dir, file)) as f:
            file_content = f.read()
        file_content = file_content.replace('3B3C37', 'C4C3C8')
        file_content = file_content.replace('807D78', '7F8287')
        file_content = file_content.replace('DFDBD2', '20242D')
        file_content = file_content.replace('7A7A7A', '858585')
        file_content = file_content.replace('3C3B37', 'C3C4C8')
        file_content = file_content.replace('2A2A28', 'D5D5D7')
        file_content = file_content.replace('909090', '6F6F6F')

        # replace text color from white to black
        if file[:3] in ("max", "min", "res"):
            file_content = file_content.replace('FFFFFF', '000000')

        if file == "default.theme":
            if not args.taskbuttonactive_alternative:
                file_content = file_content.replace(
                            'ColorActiveTaskBarAppText="#FFFFFF"',
                            'ColorActiveTaskBarAppText="#000000"')
            if rgb[0] * 0.299 + rgb[1] * 0.587 + rgb[2] * 0.144 > 186:
                file_content = file_content.replace(
                        'ColorActiveWorkspaceButtonText="#FFFFFF"',
                        'ColorActiveWorkspaceButtonText="#000000"')
            file_content = file_content.replace(
                    'ColorToolTip="#000000"',
                    'ColorToolTip="#FFFFFF"')
            file_content = file_content.replace(
                    'ColorToolTipText="#FFFFFF"',
                    'ColorToolTipText="#000000"')
            file_content = file_content.replace(
                    'ColorMoveSizeStatus="#000000"',
                    'ColorMoveSizeStatus="#FFFFFF"')
            file_content = file_content.replace(
                    'ColorMoveSizeStatusText="#FFFFFF"',
                    'ColorMoveSizeStatusText="#000000"')

        # write changes
        with open(os.path.join(output_dir, file), 'w') as f:
            f.write(file_content)

# Give some feedback to user upon success
print(f"Custom theme saved to '{output_dir}'")
