/*
   Unix SMB/CIFS implementation.

   Provide parent->child communication based on NDR marshalling

   Copyright (C) Volker Lendecke 2009

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * This file implements an RPC between winbind parent and child processes,
 * leveraging the autogenerated marshalling routines for MSRPC. This is not
 * MSRPC, as it does not go through the whole DCERPC fragmentation, we just
 * leverage much the same infrastructure we already have for it.
 */

#include "includes.h"
#include "winbindd/winbindd.h"
#include "winbindd/winbindd_proto.h"
#include "ntdomain.h"
#include "librpc/gen_ndr/srv_winbind.h"

struct wbint_bh_state {
	struct winbindd_domain *domain;
	struct winbindd_child *child;
};

static bool wbint_bh_is_connected(struct dcerpc_binding_handle *h)
{
	struct wbint_bh_state *hs = dcerpc_binding_handle_data(h,
				     struct wbint_bh_state);

	if (!hs->child) {
		return false;
	}

	return true;
}

static uint32_t wbint_bh_set_timeout(struct dcerpc_binding_handle *h,
				     uint32_t timeout)
{
	/* TODO: implement timeouts */
	return UINT32_MAX;
}

struct wbint_bh_raw_call_state {
	struct winbindd_domain *domain;
	uint32_t opnum;
	DATA_BLOB in_data;
	struct winbindd_request request;
	struct winbindd_response *response;
	DATA_BLOB out_data;
};

static void wbint_bh_raw_call_done(struct tevent_req *subreq);

static struct tevent_req *wbint_bh_raw_call_send(TALLOC_CTX *mem_ctx,
						  struct tevent_context *ev,
						  struct dcerpc_binding_handle *h,
						  const struct GUID *object,
						  uint32_t opnum,
						  uint32_t in_flags,
						  const uint8_t *in_data,
						  size_t in_length)
{
	struct wbint_bh_state *hs =
		dcerpc_binding_handle_data(h,
		struct wbint_bh_state);
	struct tevent_req *req;
	struct wbint_bh_raw_call_state *state;
	bool ok;
	struct tevent_req *subreq;

	req = tevent_req_create(mem_ctx, &state,
				struct wbint_bh_raw_call_state);
	if (req == NULL) {
		return NULL;
	}
	state->domain = hs->domain;
	state->opnum = opnum;
	state->in_data.data = discard_const_p(uint8_t, in_data);
	state->in_data.length = in_length;

	ok = wbint_bh_is_connected(h);
	if (!ok) {
		tevent_req_nterror(req, NT_STATUS_CONNECTION_DISCONNECTED);
		return tevent_req_post(req, ev);
	}

	if ((state->domain != NULL)
	    && wcache_fetch_ndr(state, state->domain, state->opnum,
				&state->in_data, &state->out_data)) {
		tevent_req_done(req);
		return tevent_req_post(req, ev);
	}

	state->request.cmd = WINBINDD_DUAL_NDRCMD;
	state->request.data.ndrcmd = state->opnum;
	state->request.extra_data.data = (char *)state->in_data.data;
	state->request.extra_len = state->in_data.length;

	subreq = wb_child_request_send(state, ev, hs->child,
				       &state->request);
	if (tevent_req_nomem(subreq, req)) {
		return tevent_req_post(req, ev);
	}
	tevent_req_set_callback(subreq, wbint_bh_raw_call_done, req);

	return req;
}

static void wbint_bh_raw_call_done(struct tevent_req *subreq)
{
	struct tevent_req *req =
		tevent_req_callback_data(subreq,
		struct tevent_req);
	struct wbint_bh_raw_call_state *state =
		tevent_req_data(req,
		struct wbint_bh_raw_call_state);
	int ret, err;

	ret = wb_child_request_recv(subreq, state, &state->response, &err);
	TALLOC_FREE(subreq);
	if (ret == -1) {
		NTSTATUS status = map_nt_error_from_unix(err);
		tevent_req_nterror(req, status);
		return;
	}

	state->out_data = data_blob_talloc(state,
		state->response->extra_data.data,
		state->response->length - sizeof(struct winbindd_response));
	if (state->response->extra_data.data && !state->out_data.data) {
		tevent_req_oom(req);
		return;
	}

	if (state->domain != NULL) {
		wcache_store_ndr(state->domain, state->opnum,
				 &state->in_data, &state->out_data);
	}

	tevent_req_done(req);
}

static NTSTATUS wbint_bh_raw_call_recv(struct tevent_req *req,
					TALLOC_CTX *mem_ctx,
					uint8_t **out_data,
					size_t *out_length,
					uint32_t *out_flags)
{
	struct wbint_bh_raw_call_state *state =
		tevent_req_data(req,
		struct wbint_bh_raw_call_state);
	NTSTATUS status;

	if (tevent_req_is_nterror(req, &status)) {
		tevent_req_received(req);
		return status;
	}

	*out_data = talloc_move(mem_ctx, &state->out_data.data);
	*out_length = state->out_data.length;
	*out_flags = 0;
	tevent_req_received(req);
	return NT_STATUS_OK;
}

struct wbint_bh_disconnect_state {
	uint8_t _dummy;
};

static struct tevent_req *wbint_bh_disconnect_send(TALLOC_CTX *mem_ctx,
						struct tevent_context *ev,
						struct dcerpc_binding_handle *h)
{
	struct wbint_bh_state *hs = dcerpc_binding_handle_data(h,
				     struct wbint_bh_state);
	struct tevent_req *req;
	struct wbint_bh_disconnect_state *state;
	bool ok;

	req = tevent_req_create(mem_ctx, &state,
				struct wbint_bh_disconnect_state);
	if (req == NULL) {
		return NULL;
	}

	ok = wbint_bh_is_connected(h);
	if (!ok) {
		tevent_req_nterror(req, NT_STATUS_CONNECTION_DISCONNECTED);
		return tevent_req_post(req, ev);
	}

	/*
	 * TODO: do a real async disconnect ...
	 *
	 * For now the caller needs to free rpc_cli
	 */
	hs->child = NULL;

	tevent_req_done(req);
	return tevent_req_post(req, ev);
}

static NTSTATUS wbint_bh_disconnect_recv(struct tevent_req *req)
{
	NTSTATUS status;

	if (tevent_req_is_nterror(req, &status)) {
		tevent_req_received(req);
		return status;
	}

	tevent_req_received(req);
	return NT_STATUS_OK;
}

static bool wbint_bh_ref_alloc(struct dcerpc_binding_handle *h)
{
	return true;
}

static void wbint_bh_do_ndr_print(struct dcerpc_binding_handle *h,
				  int ndr_flags,
				  const void *_struct_ptr,
				  const struct ndr_interface_call *call)
{
	void *struct_ptr = discard_const(_struct_ptr);

	if (DEBUGLEVEL < 10) {
		return;
	}

	if (ndr_flags & NDR_IN) {
		ndr_print_function_debug(call->ndr_print,
					 call->name,
					 ndr_flags,
					 struct_ptr);
	}
	if (ndr_flags & NDR_OUT) {
		ndr_print_function_debug(call->ndr_print,
					 call->name,
					 ndr_flags,
					 struct_ptr);
	}
}

static const struct dcerpc_binding_handle_ops wbint_bh_ops = {
	.name			= "wbint",
	.is_connected		= wbint_bh_is_connected,
	.set_timeout		= wbint_bh_set_timeout,
	.raw_call_send		= wbint_bh_raw_call_send,
	.raw_call_recv		= wbint_bh_raw_call_recv,
	.disconnect_send	= wbint_bh_disconnect_send,
	.disconnect_recv	= wbint_bh_disconnect_recv,

	.ref_alloc		= wbint_bh_ref_alloc,
	.do_ndr_print		= wbint_bh_do_ndr_print,
};

/* initialise a wbint binding handle */
struct dcerpc_binding_handle *wbint_binding_handle(TALLOC_CTX *mem_ctx,
						struct winbindd_domain *domain,
						struct winbindd_child *child)
{
	struct dcerpc_binding_handle *h;
	struct wbint_bh_state *hs;

	h = dcerpc_binding_handle_create(mem_ctx,
					 &wbint_bh_ops,
					 NULL,
					 &ndr_table_winbind,
					 &hs,
					 struct wbint_bh_state,
					 __location__);
	if (h == NULL) {
		return NULL;
	}
	hs->domain = domain;
	hs->child = child;

	return h;
}

enum winbindd_result winbindd_dual_ndrcmd(struct winbindd_domain *domain,
					  struct winbindd_cli_state *state)
{
	struct pipes_struct p;
	const struct api_struct *fns;
	int num_fns;
	bool ret;

	fns = winbind_get_pipe_fns(&num_fns);

	if (state->request->data.ndrcmd >= num_fns) {
		return WINBINDD_ERROR;
	}

	DEBUG(10, ("winbindd_dual_ndrcmd: Running command %s (%s)\n",
		   fns[state->request->data.ndrcmd].name,
		   domain ? domain->name : "no domain"));

	ZERO_STRUCT(p);
	p.mem_ctx = talloc_stackframe();
	p.in_data.data = data_blob_const(state->request->extra_data.data,
					 state->request->extra_len);

	ret = fns[state->request->data.ndrcmd].fn(&p);
	if (!ret) {
		TALLOC_FREE(p.mem_ctx);
		return WINBINDD_ERROR;
	}

	state->response->extra_data.data =
		talloc_move(state->mem_ctx, &p.out_data.rdata.data);
	state->response->length += p.out_data.rdata.length;
	p.out_data.rdata.length = 0;

	TALLOC_FREE(p.mem_ctx);

	if (state->response->extra_data.data == NULL) {
		return WINBINDD_ERROR;
	}
	return WINBINDD_OK;
}
