/* Copyright (C) 1998 Trent Piepho  <xyzzy@u.washington.edu>
 *           (C) 1999-2001 Trent Piepho  <xyzzy@speakeasy.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>
#include "defaults.h"

#define VERSION "1.8"

/* X related globals */
Display *dpy;
Window win;
GC gc;
XFontStruct *font;
int Ascent=33,Descent=0;
int Width=55,Height=33;
XColor FgXColor,HiXColor;
Pixmap ShapeMask;			/* shape stuff */
GC ShapeGC;					/* shape stuff */
int MaskWidth,MaskHeight;	/* shape stuff */
Atom DeleteWindow;			/* Atom of delete window message */
Atom KWMDockWindow;			/* Atom for KWM hint */

/* app-default related globals */
char *FgColor="Black",*BgColor="White",*HiColor="Red";
char *FontName=NULL;
char *DisplayName=NULL;
char *AppName;
int Number;
int Interval=INTERVAL_SPOOL;
char SpoolFile[256];
char Command[256];
char Username[32];
char Password[32];
char NewMailCommand[256];
char *Geometry=NULL;
int Options=0;
#define BELL_MODE		0x0001
#define COMMAND_MODE	0x0002
#define SHAPED_WINDOW	0x0004
#define HILIGHT_MAIL	0x0008
#define OFFLINE			0x0010
#define USE_POP3		0x0020
#define USE_APOP3		0x0040
#define USE_IMAP		0x0080
#define USE_MAILDIR		0x0100
#define WM_MODE			0x0200
#define KDE1_MODE		0x0400
#define KDE2_MODE		0x0800
#define NOKDE			0x1000
#define	NOWM			0x2000

char *BackupFonts[3][4] = {{ FONTNAME, FONTNAME1, FONTNAME2, FONTNAME3 }, 
						   { KDEFONT, KDEFONT1, KDEFONT2, KDEFONT3 },
						   { WMFONT, WMFONT1, WMFONT2, WMFONT3 } };

void usage();
void update();
void handler(int);
void parse_options(int argc,char *argv[]);
void init(int argc,char *argv[]);
int count_mail();
int count_mail_mbox();

void update()
{
	static int old_number=-1;
	char str[32];
	static int oldw=-1,oldh=-1;
	int w,h;

	if(Options&OFFLINE && Number==-1)  {
		strcpy(str,"X");
	} else {
		sprintf(str,"%d",Number);
	}

	w = (Width-XTextWidth(font,str,strlen(str)))/2;
	h = (Height+Ascent-Descent)/2;

	if(Options&SHAPED_WINDOW)  {
		if(Number!=old_number || oldw!=w || oldh!=h)  {
			old_number=Number; oldw=w; oldh=h;
			/* these next 3 lines clear the pixmap, is there a cleaner way? */
			XSetFunction(dpy,ShapeGC,GXclear);
			XFillRectangle(dpy,ShapeMask,ShapeGC,0,0,MaskWidth,MaskHeight);
			XSetFunction(dpy,ShapeGC,GXset);
			XDrawString(dpy,ShapeMask,ShapeGC,0,Ascent,str,strlen(str));
			XShapeCombineMask(dpy, win, ShapeBounding, 
				w, h-Ascent, ShapeMask, ShapeSet);
		};
	} else {
		XClearWindow(dpy,win);
	}

	if(Options&HILIGHT_MAIL)  {
		XSetForeground(dpy,gc,Number?HiXColor.pixel:FgXColor.pixel);
	}

	XDrawString(dpy,win,gc,w,h,str,strlen(str));
}

void handler(int nothing)
{
	int old;

	old = Number;
	count_mail();
	if(old==Number) return;
	update();
	if(Number>old)  {
		if(Options&BELL_MODE)  XBell(dpy,100);
		if(Options&COMMAND_MODE) {
			char str[256];
			sprintf(str, NewMailCommand, Number);
			system(str);
		}
	}
	XFlush(dpy);
}

void font_height(void)
{
	int foo,bar,baz;
	XCharStruct extents;

	XTextExtents(font,"0123456789",10,&foo,&bar,&baz,&extents);
	Ascent=extents.ascent;
	Descent=extents.descent;
}

void usage(void)
{
	printf("XLassie V" VERSION "\tby Trent Piepho <xyzzy@speakeasy.org>\n");
	printf("\thttp://www.speakeasy.org/~xyzzy/xlassie/\n");
	printf("Usage: xlassie [-h] | [-options]\n");
	printf("where options include:\n");
	printf("    -h,  -help              Print usage\n");
	printf("    -v,  -version           Print version\n");
	printf("    -name <name>            Set app name\n");
	printf("    -bg <color>             Background color [%s]\n",BgColor);
	printf("    -fg <color>             Foreground color [%s]\n",FgColor);
	printf("    -hilight [color]        Use a different foreground color when there\n"
		   "                                is non-zero mail [%s]\n", HiColor);
	printf("    -update <number>        Check mail every <number> seconds\n"
	       "                                default %d sec local, %d sec remote\n", INTERVAL_SPOOL, INTERVAL_POP3);
	printf("    -bell                   Ring bell when count increases from zero\n");
	printf("    -mailcommand <command>  Command to execute on new mail arrival\n");
	printf("    -fn <fontname>          Font for the new mail count\n");
	printf("    -display <displayname>  X server to connect to\n");
	printf("    -shape                  Use a shaped window\n");
	printf("    -file <filename>        File to use for mail spool or maildir directory\n");
	printf("    -command <command>      Command to execute when clicked on\n");
#ifdef POP3
	printf("    -pop3 <server>          Connect to pop3 server rather than local mail spool\n");
	printf("    -apop3 <server>         Like -pop3, but uses a diferent method.\n");
	printf("                                Use when -pop3 doesn't find the correct number\n");
	printf("    -imap <server>          Use the IMAP protocol instead of pop3\n");
	printf("    -username <name>        Username for pop3/imap server\n"
		   "                                if different from local username\n");
	printf("    -password <word>        Password to use on pop3/imap server\n");
	printf("                                Use the password 'ask' to get prompted for\n"
		   "                                the password on stdin.\n");
	printf("    -offline                Don't exit when the server is unreachable\n");
#endif
	printf("    -wmaker                 Do stuff to get swallowed into the WindowMaker dock\n");
	printf("    -kde                    Do stuff to get swallowed into the KDE dock\n");
	printf("                                Normally KDE or WindowMaker is autodetected\n"
		   "                                and you don't need to use the above options\n");
	printf("    -nokde, -nowmaker       Don't do the stuff to get swallowed\n");
}

void ask_password()
{
	int i;

	printf("Mail account password: ");
	fgets(Password, 32, stdin);
	i = strlen(Password)-1;
	if(Password[i]=='\n') Password[i]='\0';
}

void parse_options(int argc,char *argv[])
{
	int i;
	int intervalused=0;

	for(i=1;i<argc;i++)  {
		if(!strcmp(argv[i],"-fg"))  {
			if(++i==argc)  { usage(); exit(2); };
			FgColor=argv[i];
		} else if(!strcmp(argv[i],"-bg"))  {
			if(++i==argc)  { usage(); exit(2); };
			BgColor=argv[i];
		} else if(!strcmp(argv[i],"-update"))  {
			if(++i==argc)  { usage(); exit(2); };
			Interval=atoi(argv[i]);
			intervalused=1;
		} else if(!strcmp(argv[i],"-bell"))  {
			Options|=BELL_MODE;
		} else if(!strcmp(argv[i],"-shape"))  {
			Options|=SHAPED_WINDOW;
		} else if(!strcmp(argv[i],"-fn") || !strcmp(argv[i],"-font"))  {
			if(++i==argc)  { usage(); exit(2); };
			FontName=argv[i];
		} else if(!strcmp(argv[i],"-display"))  {
			if(++i==argc)  { usage(); exit(2); };
			DisplayName=argv[i];
		} else if(!strcmp(argv[i],"-file"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
		} else if(!strcmp(argv[i],"-hilight"))  {
			Options|=HILIGHT_MAIL;
			if(i+1!=argc && argv[i+1][0]!='-') HiColor=argv[++i];
		} else if(!strcmp(argv[i],"-command"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(Command,argv[i]);
		} else if(!strcmp(argv[i],"-geometry"))  {
			if(++i==argc)  { usage(); exit(2); };
			Geometry=argv[i];
		} else if(!strcmp(argv[i],"-name")) {
			if(++i==argc)  { usage(); exit(2); };
			AppName=argv[i];
#ifdef POP3
		} else if(!strcmp(argv[i],"-pop3"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
			Options|=USE_POP3;
			if(!intervalused) Interval=INTERVAL_POP3;
		} else if(!strcmp(argv[i],"-apop3"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
			Options|=USE_APOP3;
			if(!intervalused) Interval=INTERVAL_POP3;
		} else if(!strcmp(argv[i],"-imap"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(SpoolFile,argv[i]);
			Options|=USE_IMAP;
			if(!intervalused) Interval=INTERVAL_POP3;
		} else if(!strcmp(argv[i],"-username"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(Username,argv[i]);
		} else if(!strcmp(argv[i],"-password"))  {
			if(++i==argc)  { usage(); exit(2); };
			strcpy(Password,argv[i]);
			memset(argv[i],0,strlen(argv[i]));
			if(!strcmp(Password,"ask"))  ask_password();
		} else if(!strcmp(argv[i],"-offline"))  {
			Options|=OFFLINE;
#endif
		} else if(!strcmp(argv[i],"-wmaker"))  {
			Options|=WM_MODE;
		} else if(!strcmp(argv[i],"-kde")) {
			Options|=KDE1_MODE|KDE2_MODE;
		} else if(!strcmp(argv[i],"-nokde")) {
			Options|=NOKDE;
		} else if(!strcmp(argv[i],"-nowmaker")) {
			Options|=NOWM;
		} else if(!strcmp(argv[i],"-mailcommand")) {
			Options|=COMMAND_MODE;
			if(++i==argc)  { usage(); exit(2); };
			strcpy(NewMailCommand, argv[i]);
		} else if(!strcmp(argv[i],"-h") || !strcmp(argv[i],"-help"))  {
			usage(); exit(0);
		} else if(!strcmp(argv[i],"-v") || !strcmp(argv[i],"-version"))  {
			printf("XLassie " VERSION "\n"); exit(0);
		} else {
			fprintf(stderr,"Unknown option %s\n",argv[i]);
			fprintf(stderr,"Use -h for help\n");
			exit(2);
		};
	};
}

void init(int argc,char *argv[])
{
	int screen;
	XWMHints *wmh;
	XSizeHints *xsh;
	XClassHint *classh;
	XColor color,tmp;
	Window icon=0;
	int x,y,g;

	parse_options(argc,argv);

	dpy=XOpenDisplay(DisplayName);
	if(dpy==NULL)  {
		fprintf(stderr,"Error: Can't open display: %s\n",DisplayName);
		exit(1);
	}
	screen=DefaultScreen(dpy);
	DeleteWindow = XInternAtom(dpy, "WM_DELETE_WINDOW", False);

	if(!(Options&NOKDE)) {
		/* Turn on KDE mode if we find the KWM docking hint atom */
		KWMDockWindow = XInternAtom(dpy, "KWM_DOCKWINDOW", True);
		if(KWMDockWindow!=None)  {
			Options |= KDE1_MODE;
		} else  { 
			/* Looks like KWM_DOCKWINDOW wasn't the magic word.  Try the KDE2,
			   "NET" method and see if that works.  */
			KWMDockWindow = XInternAtom(dpy, "_KDE_NET_WM_SYSTEM_TRAY_WINDOW_FOR",
			                            True);
			if(KWMDockWindow!=None)  {
				Options |= KDE2_MODE;
			}
		}
	}
	if(!(Options&NOWM)) {
		/* Find a window maker atom... */
		Atom a = XInternAtom(dpy, "_WINDOWMAKER_WM_PROTOCOLS", True);
		if(a!=None) {
			Options |= WM_MODE;
			Options |= SHAPED_WINDOW;
		}
	}

	xsh=XAllocSizeHints();
	wmh=XAllocWMHints();
	classh=XAllocClassHint();

	/* Try and find a font */
	if((Options&KDE2_MODE) || (Options&KDE1_MODE)) y=1;
	else if(Options&WM_MODE) y=2; else y=0;

	if(FontName == NULL) { FontName = BackupFonts[y][0]; x=1; }
	else { x = 0; }

	font=XLoadQueryFont(dpy, FontName);
	for(;x<4 && font==NULL;x++) {
		fprintf(stderr, "Font \"%s\" not found\n", FontName);
		FontName = BackupFonts[y][x];
		fprintf(stderr, " trying \"%s\"\n", FontName);
		font=XLoadQueryFont(dpy, FontName);
	}
	if(font==NULL) {
		fprintf(stderr, "Can't find any fonts, I give up!\n");
		exit(1);
	}

	font_height();
	Height=Ascent+Descent;
	Width=XTextWidth(font,"88",2);
	xsh->flags = PSize; xsh->width = Width; xsh->height = Height;
/*	printf("%d x %d\n", Width, Height);*/

	g=XWMGeometry(dpy,screen,Geometry,NULL,0,xsh,&x,&y,&Width,&Height,&g);
	if(g&XValue)  { xsh->x=x; xsh->flags|=USPosition; };
	if(g&YValue)  { xsh->y=y; xsh->flags|=USPosition; };
	if(g&WidthValue)  {xsh->width=Width; xsh->flags|=USSize; 
	} else            {Width = xsh->width; };
	if(g&HeightValue)  {xsh->height=Height; xsh->flags|=USSize;
	} else            {Height = xsh->height; };

/*	printf("%dx%d+%d+%d\n",Width,Height,x,y);
	printf("%s%s%s%s\n",
		(xsh->flags&USPosition)?"USPosition ":"",
		(xsh->flags&PPosition)?"PPosition ":"",
		(xsh->flags&USSize)?"USSize ":"",
		(xsh->flags&PSize)?"PSize ":""); */

	win=XCreateSimpleWindow(dpy,RootWindow(dpy,screen),x,y,Width,Height,0,
		BlackPixel(dpy,screen),WhitePixel(dpy,screen));

	wmh->initial_state=NormalState;
	wmh->input=False;
	wmh->window_group = win;
	wmh->flags = StateHint | InputHint | WindowGroupHint;
	classh->res_name = (AppName==NULL)?"xlassie":AppName;
	classh->res_class = "XBiff";

	if(Options&WM_MODE) {
		/* WindowMaker stuff */
		icon=XCreateSimpleWindow(dpy,RootWindow(dpy,screen),x,y,Width,Height,
				 0, BlackPixel(dpy,screen), WhitePixel(dpy,screen));
		wmh->initial_state=WithdrawnState;
		wmh->icon_window=icon;
		wmh->flags |= IconWindowHint;
		XSetClassHint(dpy, icon, classh);
		XSelectInput(dpy, icon, ExposureMask|ButtonPressMask|StructureNotifyMask);
	}
	XmbSetWMProperties(dpy, win, "xlassie", "xlassie", argv, argc,
					   xsh, wmh, classh);
	XSetWMProtocols(dpy, win, &DeleteWindow, 1);
	XSelectInput(dpy, win, ExposureMask|ButtonPressMask|StructureNotifyMask);

	if(Options&KDE1_MODE) {
		unsigned char data = 1;
		if(KWMDockWindow==None)  {
			fprintf(stderr, "KDE1 mode isn't going to work, skipping\n");
		} else {
			XChangeProperty(dpy, win, KWMDockWindow, KWMDockWindow, 8, 
							PropModeReplace, &data, 1);
		}
	}
	if(Options&KDE2_MODE) {
		if(KWMDockWindow==None)  {
			fprintf(stderr, "KDE2 mode isn't going to work, skipping\n");
		} else {
			XChangeProperty(dpy, win, KWMDockWindow, XA_WINDOW, 32, 
							PropModeReplace, (void*)&win, 1);
		}
	}

	XMapWindow(dpy, win);
	if(Options&WM_MODE) win = icon;		/* Use the icon window from now on */

	gc=XCreateGC(dpy,win,0,NULL);

	XAllocNamedColor(dpy,DefaultColormap(dpy,screen),FgColor,&color,&tmp);
	XSetForeground(dpy,gc,color.pixel); FgXColor=color;
	XAllocNamedColor(dpy,DefaultColormap(dpy,screen),BgColor,&color,&tmp);
	XSetBackground(dpy,gc,color.pixel);
	XSetWindowBackground(dpy,win,color.pixel);
	if(Options&HILIGHT_MAIL)  XAllocNamedColor(dpy,DefaultColormap(dpy,screen),
												HiColor,&HiXColor,&tmp);
	XSetFont(dpy,gc,font->fid);

	if(Options&SHAPED_WINDOW)  {
		MaskWidth = Width; MaskHeight = Height;
		ShapeMask = XCreatePixmap(dpy,win,MaskWidth,MaskHeight,1);
		ShapeGC = XCreateGC(dpy,ShapeMask,0,NULL);
		XSetFont(dpy,ShapeGC,font->fid);
	};
}

int check_spool()
{
	struct stat st;
	char str[PATH_MAX];

	if(Options&(USE_POP3|USE_APOP3|USE_IMAP)) return 0;
	if(stat(SpoolFile, &st)) {
		perror("stat");
		fprintf(stderr, "Can't seem to find your mail file/directory: '%s'\n",
		        SpoolFile);
		exit(errno);
	}

	if(S_ISDIR(st.st_mode)) {
		Options|=USE_MAILDIR;

		/* Check for the subdir "new" and use it if it exists */
		strcpy(str, SpoolFile);
		strcat(str, (str[strlen(str)-1]=='/')?"new":"/new");
		if(!stat(str, &st) && S_ISDIR(st.st_mode)) {
			strcpy(SpoolFile, str);
		}
	}
	return 1;
}

int main(int argc,char *argv[])
{
	XEvent xev;
	Atom WM_PROTOCOLS;
	struct itimerval itv;
	struct sigaction sig;
	struct passwd* pwd;
	char *mail;

	pwd = getpwuid(getuid());
	mail= getenv("MAIL");
	if(mail==NULL) {
		sprintf(SpoolFile,"/var/spool/mail/%s", pwd->pw_name);
	} else {
		strcpy(SpoolFile, mail);
	}
	strcpy(Username,pwd->pw_name);
	strcpy(Command, COMMAND);

	init(argc,argv);
	check_spool();

	WM_PROTOCOLS = XInternAtom(dpy, "WM_PROTOCOLS", False);

#ifdef OFFLINE_DEFAULT
	Options|=OFFLINE;
#endif

	count_mail();

	if(Options&BELL_MODE && Number) XBell(dpy,100);

	sig.sa_handler=handler;
	sigemptyset(&sig.sa_mask);
	sig.sa_flags=SA_RESTART;
	sigaction(SIGALRM,&sig,NULL);
	itv.it_interval.tv_sec=Interval; itv.it_interval.tv_usec=0;
	itv.it_value = itv.it_interval;
	setitimer(ITIMER_REAL,&itv,NULL);

	for(;;)  {
		XNextEvent(dpy,&xev);
		switch(xev.type)  {
		case Expose:
			while(XCheckTypedEvent(dpy,Expose,&xev));
			update();
			break;
		case ButtonPress:
			system(Command);
			break;
		case ConfigureNotify:
			Width = xev.xconfigure.width;
			Height = xev.xconfigure.height;
			update();
			break;
		case ClientMessage:
			if(xev.xclient.message_type == WM_PROTOCOLS)  {
				if(xev.xclient.data.l[0] == DeleteWindow) {
					XCloseDisplay(dpy);
					exit(0);
				};
			};
			break;
		case DestroyNotify:
			XCloseDisplay(dpy);
			exit(0);
		default:
		};
	};
}

#ifdef POP3
int sock_connect(char *,int);

FILE *pop_login()
{
	int fd;
	FILE *f;
	char buf[256];

	fd=sock_connect(SpoolFile,110);
	if(fd==-1)  {
		if(Options&OFFLINE)  {
			Number=-1; return NULL;
		} else {
			fprintf(stderr,"Error connecting to POP3 server %s\n",SpoolFile);
			exit(1);
		};
	};

	f=fdopen(fd,"r+");

	fgets(buf,256,f);

	fflush(f); fprintf(f,"USER %s\r\n",Username);
	fflush(f); fgets(buf,256,f);

	if(buf[0]!='+')  {
		fprintf(stderr,"Server doesn't like your name!\n");
		fprintf(f,"QUIT\r\n");fclose(f);
		return NULL;
	};
	fflush(f); fprintf(f,"PASS %s\r\n",Password);
	fflush(f); fgets(buf,256,f);
	if(buf[0]!='+')  {
		fprintf(stderr,"Server doesn't like your password!\n");
		fprintf(stderr,"It said %s",buf);
		fprintf(f,"QUIT\r\n");fclose(f);
		return NULL;
	};

	return(f);
}

FILE *imap_login()
{
	int fd;
	FILE *f;
	char buf[128];

	fd=sock_connect(SpoolFile, 143);
	if(fd==-1)  {
		if(Options&OFFLINE)  {
			Number=-1; return NULL;
		} else {
			fprintf(stderr, "Error connecting to IMAP server %s\n", SpoolFile);
			exit(1);
		};
	};

	f=fdopen(fd, "r+");

	fgets(buf, 127, f);
	fflush(f); fprintf(f, "a001 LOGIN %s \"%s\"\r\n", Username, Password);
	fflush(f); fgets(buf, 127, f);

	if(buf[5]!='O')  {       /* Looking for "a001 OK" */
		fprintf(stderr, "Server doesn't like your name and/or password!\n");
		fprintf(f, "a002 LOGOUT\r\n"); fclose(f);
		return NULL;
	};

	return f;
}

int count_mail_pop()
{
	FILE *f;
	char buf[256];
	int total,read;

	f = pop_login();
	if(f==NULL) return -1;

	fflush(f); fprintf(f,"STAT\r\n");
	fflush(f); fgets(buf,256,f);
	sscanf(buf,"+OK %d",&total);
	fflush(f); fprintf(f,"LAST\r\n");
	fflush(f); fgets(buf,256,f);
	sscanf(buf,"+OK %d",&read);
	Number=total-read;

	fprintf(f,"QUIT\r\n");
	fclose(f);

	return 1;
}

struct msg {
	int size;
	int read;
};

struct msg *get_list(FILE *f,int *nout)
{
	char buf[256];
	struct msg *l;
	int i,n;

	fflush(f); fprintf(f,"STAT\r\n");
	fflush(f); fgets(buf,256,f);
	sscanf(buf,"+OK %d",&n);

	l=malloc(sizeof(struct msg)*n);
	fflush(f); fprintf(f,"LIST\r\n");
	fflush(f); fgets(buf,256,f);
	if(strncmp(buf,"+OK",3))  {
		fprintf(stderr,"Can't get list?\n");
		return NULL;
	};

	for(i=0;i<n;i++)  {
		fgets(buf,256,f);
		sscanf(buf,"%*d %d",&l[i].size);
	};
	fgets(buf,256,f);
	if(buf[0]!='.')  {
		fprintf(stderr,"To many messages?\n");
		return NULL;
	};

	*nout=n;
	return l;
}

void get_header(FILE *f,struct msg *l,int i)
{
	char buf[256];
	
	fflush(f); fprintf(f,"TOP %d 1\r\n",i+1);
	fflush(f); fgets(buf,256,f);
	if(strncmp(buf,"+OK",3))  {
		fprintf(stderr,"TOP failed!\n");
		fprintf(stderr,"Returned %s\n",buf);
		return;
	};
	l->read=0;
	for(;;)  {
		fgets(buf,256,f);
		if(!strcmp(buf,".\r\n"))  break;
		if(!strncmp(buf,"Status: R",9))
			l->read=1;
	};
	/* fprintf(stderr,"msg %d %s\n",i+1,l->read?"read":"unread"); */
}

int count_mail_pop_2()
{
	FILE *f;
	int i,j,n2;
	int total=0;
	static struct msg *l1=NULL,*l2;
	static int n;

	f = pop_login();
	if(f==NULL) return -1;

	if(l1==NULL)  {  /* get initial list */
		l1=get_list(f,&n);
		if(l1==NULL) return -1;
		for(i=0;i<n;i++) {
			get_header(f,l1+i,i);
			if(!l1[i].read) total++;
		};
		Number=total; 
		return 1;
	};

	l2=get_list(f,&n2);

	/* compare lists, retreive new messages */
	for(i=0,j=0;i<n && j<n2;i++,j++)  {
/*		fprintf(stderr,"Old list %d size %d %c, new list %d size %d\n",
		               i+1,l1[i].size,l1[i].read?'R':'U', j+1,l2[j].size); */
		if(l1[i].size != l2[j].size) break;
		if(!l1[i].read)  {
			get_header(f,l2+j,j);
		} else {
			l2[j].read=1;
		};
	};
	for(;j<n2;j++) get_header(f,l2+j,j);

	fclose(f);

	free(l1); l1=l2; l2=NULL; n=n2;
	for(i=0;i<n;i++)  if(!l1[i].read) total++;
	Number=total;

	return 1;
}

int count_mail_imap()
{
	FILE *f;
	char buf[128];
	int total=0;

	f = imap_login();
	if(f==NULL) return -1;

	fflush(f); fprintf(f, "a003 STATUS INBOX (UNSEEN)\r\n");
	fflush(f); fgets(buf, 127, f);
	if(!sscanf(buf, "* STATUS INBOX (UNSEEN %d)", &total) &&
	   !sscanf(buf, "* STATUS \"INBOX\" (UNSEEN %d)", &total)) {
		fprintf(stderr, "Couldn't understand response from server\n");
		fprintf(stderr, "%s", buf);
		return 0;
	}

	Number = total;

	fclose(f);

	return 1;
}
#endif

int count_mail_mbox()
{
	static time_t mtime=0;
	static off_t size=0;
	struct stat st;
	int isread=0;
	FILE *spool;
	char buf[256];

	spool = fopen(SpoolFile,"r");
	if(spool==NULL)  {
		fprintf(stderr,"Trying to open spoolfile %s\n",SpoolFile);
		perror("xlassie");
		XCloseDisplay(dpy);
		exit(errno);
	};

	if(fstat(fileno(spool),&st)) {
		perror("fstat");
		XCloseDisplay(dpy);
		exit(errno);
	};
	/* check to see if file was changed */
	if(st.st_mtime != mtime || st.st_size != size)  {  
		mtime = st.st_mtime;
		size = st.st_size;
		Number = 0;
		while(fgets(buf, 256, spool)) {
			if(!strncmp(buf, "From ", 5))  {
				Number++; isread=0;
			} else if(!strncmp(buf, "Status: R", 9))  {
				if(!isread) { Number--; isread=1; };
			};
		};
	}
	fclose(spool);

	return 1;
}

int count_mail_maildir()
{
	static time_t mtime=0;
	struct stat st;
	DIR *dir;
	struct dirent *de;

	if(stat(SpoolFile, &st)) {
		perror("stat");
		fprintf(stderr, "Can't access incomming mail directory '%s'\n", 
		        SpoolFile);
		XCloseDisplay(dpy);
		exit(errno);
	}

	if(st.st_mtime != mtime) {
		mtime = st.st_mtime;
		if((dir = opendir(SpoolFile)) == NULL) {
			perror("opendir");
			XCloseDisplay(dpy);
			exit(errno);
		}
		Number = 0;
		while((de = readdir(dir)) != NULL) {
			if(de->d_name[0] != '.') Number++;
		}
		closedir(dir);
	}
	return 1;
}

int count_mail()
{
	if(Options&USE_MAILDIR) {
		return count_mail_maildir();
#ifdef POP3
	} else if(Options&USE_APOP3)  {
		return count_mail_pop_2();
	} else if(Options&USE_POP3)  {
		return count_mail_pop();
	} else if(Options&USE_IMAP) {
		return count_mail_imap();
#endif
	} else {
		return count_mail_mbox();
	}
}
