/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/cosine/cdf' );
import Cosine = require( './../../../../../base/dists/cosine/ctor' );
import kurtosis = require( './../../../../../base/dists/cosine/kurtosis' );
import logcdf = require( './../../../../../base/dists/cosine/logcdf' );
import logpdf = require( './../../../../../base/dists/cosine/logpdf' );
import mean = require( './../../../../../base/dists/cosine/mean' );
import median = require( './../../../../../base/dists/cosine/median' );
import mgf = require( './../../../../../base/dists/cosine/mgf' );
import mode = require( './../../../../../base/dists/cosine/mode' );
import pdf = require( './../../../../../base/dists/cosine/pdf' );
import quantile = require( './../../../../../base/dists/cosine/quantile' );
import skewness = require( './../../../../../base/dists/cosine/skewness' );
import stdev = require( './../../../../../base/dists/cosine/stdev' );
import variance = require( './../../../../../base/dists/cosine/variance' );

/**
* Interface describing the `cosine` namespace.
*/
interface Namespace {
	/**
	* Raised cosine distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 0.5, 0.0, 1.0 );
	* // returns ~0.909
	*
	* var mycdf = ns.cdf.factory( 3.0, 1.5 );
	*
	* y = mycdf( 4.0 );
	* // returns ~0.971
	*/
	cdf: typeof cdf;

	/**
	* Cosine distribution.
	*/
	Cosine: typeof Cosine;

	/**
	* Returns the excess kurtosis for a raised cosine distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns excess kurtosis
	*
	* @example
	* var y = ns.kurtosis( 0.0, 1.0 );
	* // returns ~-0.594
	*
	* @example
	* var y = ns.kurtosis( 5.0, 2.0 );
	* // returns ~-0.594
	*
	* @example
	* var y = ns.kurtosis( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, 0.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Raised cosine distribution logarithm of cumulative distribution function (logCDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 0.5, 0.0, 1.0 );
	* // returns ~-0.095
	*
	* var mylogcdf = ns.logcdf.factory( 3.0, 1.5 );
	*
	* y = mylogcdf( 4.0 );
	* // returns ~--0.029
	*/
	logcdf: typeof logcdf;

	/**
	* Raised cosine distribution logarithm of probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.0, 1.0 );
	* // returns ~-2.254
	*
	* var mylogpdf = ns.logpdf.factory( 10.0, 2.0 );
	* y = mylogpdf( 10.0 );
	* // returns ~-0.693
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value for a raised cosine distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns expected value
	*
	* @example
	* var y = ns.mean( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.mean( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.mean( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, 0.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median for a raised cosine distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns median
	*
	* @example
	* var y = ns.median( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.median( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.median( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, 0.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Raised cosine distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.5, 0.0, 1.0 );
	* // returns ~1.016
	*
	* var myMGF = ns.mgf.factory( 3.0, 1.5 );
	*
	* y = myMGF( 1.0 );
	* // returns ~23.219
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode for a raised cosine distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns mode
	*
	* @example
	* var y = ns.mode( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.mode( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.mode( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, 0.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Raised cosine distribution probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.0, 3.0 );
	* // returns ~0.083
	*
	* var myPDF = ns.pdf.factory( 0.0, 3.0 );
	* y = myPDF( 2.0 );
	* // returns ~0.083
	*/
	pdf: typeof pdf;

	/**
	* Raised cosine distribution quantile function.
	*
	* @param p - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.0, 1.0 );
	* // returns ~0.327
	*
	* var myQuantile = ns.quantile.factory( 10.0, 2.0 );
	* y = myQuantile( 0.5 );
	* // returns ~10.0
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness for a raised cosine distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns skewness
	*
	* @example
	* var y = ns.skewness( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.skewness( 5.0, 2.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.skewness( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, 0.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation for a raised cosine distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns standard deviation
	*
	* @example
	* var y = ns.stdev( 0.0, 1.0 );
	* // returns ~0.362
	*
	* @example
	* var y = ns.stdev( 5.0, 2.0 );
	* // returns ~0.723
	*
	* @example
	* var y = ns.stdev( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, 0.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance for a raised cosine distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns variance
	*
	* @example
	* var y = ns.variance( 0.0, 1.0 );
	* // returns ~0.131
	*
	* @example
	* var y = ns.variance( 5.0, 2.0 );
	* // returns ~0.523
	*
	* @example
	* var y = ns.variance( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, 0.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Raised cosine distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
