# Here are defined general and miscellaneous functions:
#
#     echox                      [WORD...]
#     sort_by_filename           [SUMROST0...] -> [SUMPATH0...]
#     sort_by_filename --no-sums [ROSTER0...]  -> [FILE0...]
#     decompress                 [FILE0...]
#     inform_of_scan
#
# Note that a few functions have already been defined
# in 03pre-initialization.bash. Numerous further functions are defined
# in 1{1..9}*-definitions.bash.
#
# The following notes apply to 1{0..9}*.bash or, in some cases, to the
# entire program.
#
# Function argments have these meanings:
#
#     FILE    a file's pathname
#     LINK    a symlink's pathname
#     DIR     a directory's pathname
#     PATH    a file's or directory's pathname
#     SUMPATH a pathname preceded by a checksum
#     ROSTER  name of a file that lists FILEs, DIRs or PATHs, one per line
#     SUMROST a ROSTER whose data are preceded by checksums
#     LIST    name of a file that lists patterns
#     CTRL    the pathname of a Debian control file, as of a Packages file
#     BASEn   the n leading elements omitted from FILEn or the like
#     AREA    the leading element of DIR2: main; contrib; non-free
#
# Several of these have n-forms such as FILEn, the n being a nonnegative
# integer count of leading path elements. This is best explained by example:
#
#     FILE    /home/thb/debian-10.8/dists/buster/main/binary-amd64/Packages
#     FILE0                         dists/buster/main/binary-amd64/Packages
#     FILE2                                      main/binary-amd64/Packages
#     BASE2                         dists/buster
#
# A ROSTERn, SUMROSTn or LISTn is specified by its full pathname, but its
# contents are lines, each of which identifies a FILEn, DIRn or such.
#
# The scripts 1{0..9}*.bash only define functions. They neither set
# variables nor write data files for external use (though they do regard
# variables set and data files written by other scripts).
#
# The script 1?.bash reserves all names F1?_* for internal use. For
# example, the script you are reading reserves F10_*. Other scripts must
# not touch these. However, any further variables the script internally
# uses can safely be overwritten by other scripts.

# This function affords the echo builtin an alternate interface that
# ignores command-line options:
#
#      echo  -e foo -> foo
#      echox -e foo -> -e foo
#
function echox {
    local ECHOX_WORD ECHOX_IS_FIRST
    declare -i ECHOX_IS_FIRST=1
    for ECHOX_WORD; do
        if (($ECHOX_IS_FIRST)); then
            ECHOX_IS_FIRST=0
        else
            printf ' '
        fi
        printf '%s' "$ECHOX_WORD"
    done
    printf '\n'
    return 0
}
readonly -f echox

function sort_by_filename {
    local SBF_DO_SWAP SBF_SED_SCRIPT='s/^(\S+)(\s+)(\S+)\s*$/\3\2\1/p;t;q1'
    declare -i SBF_DO_SWAP=1; [ "$1" = '--no-sums' ] && shift && SBF_DO_SWAP=0
    readonly SBF_DO_SWAP SBF_SED_SCRIPT
    if (($SBF_DO_SWAP)); then
        sed -rn -- "$SBF_SED_SCRIPT" "$@" | sort | uniq\
        | sed -rn -- "$SBF_SED_SCRIPT"
    else
        sort -- "$@" | uniq
    fi
    return 0
}
readonly -f sort_by_filename

readonly F10_DEC1='the filename '"'"'%s'"'"' indicates no decompression'
readonly F10_DEC2='method of which this program is capable'
readonly F10_DEC_MSG=$(gettext "$F10_DEC1 $F10_DEC2")
function decompress {
    local DECOMPRESS_DIR
    local DECOMPRESS_BASE
    local DECOMPRESS_STEM
    local DECOMPRESS_EXT
    local DECOMPRESS_CMD
    local DECOMPRESS_FILEPATH
    for DECOMPRESS_FILEPATH; do
        DECOMPRESS_DIR=$(dirname -- "$DECOMPRESS_FILEPATH")
        DECOMPRESS_BASE=$(basename -- "$DECOMPRESS_FILEPATH")
        DECOMPRESS_STEM=${DECOMPRESS_BASE%.+([^/.])}
        [ "$DECOMPRESS_STEM" = "$DECOMPRESS_BASE" ] && continue;
        DECOMPRESS_EXT="${DECOMPRESS_BASE#$(printf '%q' "$DECOMPRESS_STEM").}"
        DECOMPRESS_CMD='false'
        case "$DECOMPRESS_EXT" in
            xz)  DECOMPRESS_CMD='xzcat';;
            bz2) DECOMPRESS_CMD='bzcat';;
            gz)  DECOMPRESS_CMD='zcat';;
            *)   die "$(printf "$F10_DEC_MSG" "$DECOMPRESS_FILEPATH")"
        esac
        pushd >/dev/null "$TARGET/$DECOMPRESS_DIR"
        $DECOMPRESS_CMD >"$DECOMPRESS_STEM" -- "$DECOMPRESS_BASE"
        touch -r "$(realpath -e -- "$DECOMPRESS_BASE")" --\
         "$DECOMPRESS_STEM"
        popd >/dev/null
    done
    return 0
}
readonly -f decompress

readonly F10_IOS_MSG=$(gettext 'scanning %s')
function inform_of_scan {
    inform "$(printf "$F10_IOS_MSG..." "$1")"
    return 0
}
readonly -f inform_of_scan
true

