/*
 * Copyright (C) 2025 The Phosh.mobi e.V.
 *
 * SPDX-License-Identifier: GPL-3.0+
 *
 * Author: Evangelos Ribeiro Tzaras <devrtz@fortysixandtwo.eu>
 */

#include "lcb-message-priv.h"

static void
test_message_simple (void)
{
  LcbMessage *msg;

  msg = lcb_message_new ("This is a test",
                         4391, 12, 0, 42, LCB_SEVERITY_LEVEL_TEST,
                         "Test alert", NULL);

  g_assert_cmpstr (lcb_message_get_text (msg), ==, "This is a test");
  g_assert_cmpuint (lcb_message_get_channel (msg), ==, 4391);
  g_assert_cmpuint (lcb_message_get_msg_code (msg), ==, 12);
  g_assert_cmpuint (lcb_message_get_update (msg), ==, 0);
  g_assert_cmpuint (lcb_message_get_timestamp (msg), ==, 42);
  g_assert_cmpuint (lcb_message_get_severity (msg), ==, LCB_SEVERITY_LEVEL_TEST);
  g_assert_cmpstr (lcb_message_get_severity_subject (msg), ==, "Test alert");
  g_assert_null (lcb_message_get_operator_code (msg));
  g_assert_finalize_object (msg);

  msg = g_object_new (LCB_TYPE_MESSAGE,
                      "text", "This is another test",
                      "channel", 4223,
                      "message-code", 125,
                      "update", 1,
                      "timestamp", G_GINT64_CONSTANT (31337),
                      "severity", LCB_SEVERITY_LEVEL_PRESIDENTIAL,
                      "severity-subject", "Presidential alert",
                      "operator-code", "00101",
                      NULL);

  g_assert_cmpstr (lcb_message_get_text (msg), ==, "This is another test");
  g_assert_cmpuint (lcb_message_get_channel (msg), ==, 4223);
  g_assert_cmpuint (lcb_message_get_msg_code (msg), ==, 125);
  g_assert_cmpuint (lcb_message_get_update (msg), ==, 1);
  g_assert_cmpuint (lcb_message_get_timestamp (msg), ==, 31337);
  g_assert_cmpuint (lcb_message_get_severity (msg), ==, LCB_SEVERITY_LEVEL_PRESIDENTIAL);
  g_assert_cmpstr (lcb_message_get_severity_subject (msg), ==, "Presidential alert");
  g_assert_cmpstr (lcb_message_get_operator_code (msg), ==, "00101");
  g_assert_finalize_object (msg);
}

static void
test_message_timestamp (void)
{
  g_autoptr (GDateTime) before = NULL;
  g_autoptr (GDateTime) after = NULL;
  LcbMessage *msg;
  gint64 timestamp_before, timestamp_after;

  before = g_date_time_new_now_utc ();
  timestamp_before = g_date_time_to_unix (before);
  msg = lcb_message_new ("This is a test",
                         4391, 12, 1, 0, LCB_SEVERITY_LEVEL_TEST,
                         "It really is a test", NULL);

  g_assert_cmpstr (lcb_message_get_text (msg), ==, "This is a test");
  g_assert_cmpuint (lcb_message_get_channel (msg), ==, 4391);
  g_assert_cmpuint (lcb_message_get_msg_code (msg), ==, 12);
  g_assert_cmpuint (lcb_message_get_update (msg), ==, 1);
  g_assert_cmpuint (lcb_message_get_severity (msg), ==, LCB_SEVERITY_LEVEL_TEST);
  g_assert_cmpstr (lcb_message_get_severity_subject (msg), ==, "It really is a test");

  after = g_date_time_new_now_utc ();
  timestamp_after = g_date_time_to_unix (after);

  g_assert_cmpuint (lcb_message_get_timestamp (msg), >=, timestamp_before);
  g_assert_cmpuint (lcb_message_get_timestamp (msg), <=, timestamp_after);

  g_assert_finalize_object (msg);
}

int
main (int argc, char *argv[])
{
  g_test_init (&argc, &argv, NULL);

  g_test_add_func ("/cbd/message/simple", test_message_simple);
  g_test_add_func ("/cbd/message/timestamp", test_message_timestamp);

  return g_test_run ();
}
