"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const helper_1 = require("../helper");
const definitionProvider_1 = require("../../src/providers/definitionProvider");
const misc_1 = require("../../src/utils/misc");
const vscode_uri_1 = require("vscode-uri");
function testModuleNamesForDefinition(context, textDoc) {
    const tests = [
        {
            word: "definition for builtin modules (ansible.builtin.debug)",
            position: { line: 5, character: 8 },
            selectionRange: {
                start: { line: 5, character: 6 },
                end: { line: 5, character: 27 },
            },
            provideDefinition: true,
        },
        {
            word: "no definition for invalid module names",
            position: { line: 13, character: 8 },
            selectionRange: {
                start: { line: 13, character: 6 },
                end: { line: 13, character: 15 },
            },
            provideDefinition: false,
        },
        {
            word: "definition for collection modules (org_1.coll_3.module_3)",
            position: { line: 18, character: 8 },
            selectionRange: {
                start: { line: 18, character: 6 },
                end: { line: 18, character: 27 },
            },
            provideDefinition: true,
        },
    ];
    tests.forEach(({ word, position, selectionRange, provideDefinition }) => {
        it(`should provide '${word}'`, async function () {
            const actualDefinition = await (0, definitionProvider_1.getDefinition)(textDoc, position, await context.docsLibrary);
            if (!provideDefinition) {
                (0, chai_1.expect)(actualDefinition).to.be.null;
                return;
            }
            (0, chai_1.expect)(actualDefinition).to.have.length(1);
            if (actualDefinition) {
                const definition = actualDefinition[0];
                // file uri check
                (0, chai_1.expect)(definition.targetUri.startsWith("file:///")).to.be.true;
                (0, chai_1.expect)(definition.targetUri).satisfy((fileUri) => (0, misc_1.fileExists)(vscode_uri_1.URI.parse(fileUri).path));
                // nodule name range check in the playbook
                (0, chai_1.expect)(definition.originSelectionRange).to.deep.equal(selectionRange);
                // original document range checks
                (0, chai_1.expect)(definition).to.haveOwnProperty("targetRange");
                (0, chai_1.expect)(definition).to.haveOwnProperty("targetSelectionRange");
            }
        });
    });
}
describe("getDefinition()", function () {
    const workspaceManager = (0, helper_1.createTestWorkspaceManager)();
    const fixtureFilePath = "definition/playbook_for_module_definition.yml";
    const fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    const context = workspaceManager.getContext(fixtureFileUri);
    const textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    const docSettings = context?.documentSettings.get(textDoc.uri);
    describe("Module name definitions", function () {
        describe("With EE enabled @ee", function () {
            before(async function () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                if (docSettings) {
                    await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testModuleNamesForDefinition(context, textDoc);
            }
            after(async function () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        describe("With EE disabled", function () {
            before(async function () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
            if (context) {
                testModuleNamesForDefinition(context, textDoc);
            }
        });
    });
});
