use core:geometry;
use ui;

/**
 * An animation for an element inside a slide.
 *
 * Similar to, but not identical to the animations used for slide intros. These animations are
 * configurable to a higher degree, and multiple of these animations may play simultaneously.
 *
 * Note: Many aspects of the animation are configured from a custom syntax, so that new animations
 * do not have to overload a large number of constructors for commonly used scenarios.
 */
class Animation on Render {
	// Play at step number.
	Nat step;

	// Duration of this animation.
	Duration duration;

	// Offset from the start of this animation step.
	Duration offset;

	// Create, default duration.
	init(Nat step) {
		init() {
			step = step;
			duration = Duration();
			offset = Duration();
		}
	}

	// Setup the animation. Called when added to a slide so that we may set defaults.
	void setup(Presentation p, Element element) {
		if (duration == 0 ms)
			duration = p.animationTime;
	}

	// Called before drawing the element. May return 'false' to prevent painting the element.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : abstract;

	// Called after drawing the element to restore any graphics state set up by 'before'.
	void after(Element element, Graphics g, Nat cStep, Duration time) : abstract;

	// Convert a time into a float in the range 0..1.
	Float val(Duration d) {
		clamp((d - offset) / duration, 0.0, 1.0);
	}

	// Convert to a smooth value in the range 0..1.
	Float smoothVal(Duration d) {
		Angle v = rad(val(d) * pi);
		0.5 - v.cos()*0.5;
	}
}

/**
 * An animation that acts like a trigger, i.e. it triggers once at the desired time and does
 * something at that moment only. This class is intended to provide functionality to other embedded
 * things, and has no real use in isolation as-is.
 */
class Trigger extends Animation {
	// Last time triggered.
	private Duration lastTime;

	// Init.
	init(Nat step) {
		init(step) {}
	}

	// Called when the event is supposed to trigger.
	void trigger(Element element) : abstract;

	// Setup. Initialize 'lastTime'.
	void setup(Presentation p, Element element) : override {
		super:setup(p, element);
		// Won't work properly with a zero animation time.
		if (duration == 0 ms)
			duration = 10 ms;
		lastTime = duration;
	}

	// Look at timestamps to determine when to trigger.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (step == cStep) {
			Bool fire = false;
			if ((lastTime < offset) & (time > offset)) {
				fire = true;
			} else if ((lastTime > time) & (time >= offset)) {
				fire = true;
			}
			lastTime = time;

			if (fire)
				trigger(element);
		}
		true;
	}

	// Need to implement since base is abstract.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}


/**
 * Simple "show" animation. Simply displays the item at the specified step.
 */
class ShowAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
		// To indicate that we're not being animated.
		duration = 1 ms;
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (step == cStep) {
			time >= offset;
		} else {
			cStep >= step;
		}
	}

	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}

/**
 * Simple "hide" animation. Hides the item at the specified step.
 */
class HideAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
		duration = 1 ms;
	}

	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (step == cStep) {
			time < offset;
		} else {
			cStep < step;
		}
	}

	void after(Element element, Graphics g, Nat cStep, Duration time) : override {}
}


/**
 * Fade in animation.
 */
class FadeInAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset)
			g.push(val(time));

		cStep >= step;
	}

	// Tear down.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset)
			g.pop();
	}

}

/**
 * Fade out animation.
 */
class FadeOutAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset) {
			g.push(1 - val(time));
			return true;
		}

		cStep < step;
	}

	// Tear down.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset)
			g.pop();
	}

}


/**
 * Grow vertically.
 */
class GrowAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset) {
			Float scale = smoothVal(time);
			g.push(scale);

			// The font drawing does not seem to like zero scaling, at least in the Y direction
			// neither on Windows nor on Linux.
			scale = max(scale, 0.05);

			Point center = element.pos.center;
			g.transform(scale(Point(1, scale), center));
		}

		cStep >= step;
	}

	// Tear down.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset)
			g.pop();
	}
}

/**
 * Shrink vertically.
 */
class ShrinkAnimation extends Animation {
	// Create.
	init(Nat step) {
		init(step) {}
	}

	// Set up the animation.
	Bool before(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset) {
			Float scale = 1 - smoothVal(time);
			g.push(scale);

			// The font drawing does not seem to like zero scaling, at least in the Y direction
			// neither on Windows nor on Linux.
			scale = max(scale, 0.05);

			Point center = element.pos.center;
			g.transform(scale(Point(1, scale), center));
			return true;
		}

		cStep < step;
	}

	// Tear down.
	void after(Element element, Graphics g, Nat cStep, Duration time) : override {
		if (cStep == step & time < duration + offset)
			g.pop();
	}
}
