/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef NANCY8DATA_H
#define NANCY8DATA_H

#include "types.h"

const GameConstants _nancy8Constants ={
	45,												// numItems
	576,											// numEventFlags
	{	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10,			// genericEventFlags
		11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
		21, 22, 23, 24, 25, 26, 27, 28, 29, 30 },
	20,												// numCursorTypes
	4000,											// logoEndAfter
	32												// wonGameFlagID
};

const Common::Array<GameLanguage> _nancy8LanguagesOrder = {
	GameLanguage::kEnglish,
	GameLanguage::kRussian
};

const Common::Array<Common::Array<ConditionalDialogue>> _nancy8ConditionalDialogue = {
{	// Elliott, 13 responses
	{	0, 1150, "NCS50",
		{ { kEv, 228, true }, { kEv, 107, true }, { kEv, 205, true }, { kEv, 116, false }, { kEv, 111, false } } },
	{	0, 1151, "NCS51",
		{ { kEv, 228, true }, { kEv, 110, false } } },
	{	0, 1152, "NCS52",
		{ { kEv, 228, true }, { kEv, 187, true }, { kEv, 290, false } } },
	{	0, 1153, "NCS53",
		{ { kEv, 228, true }, { kEv, 304, true }, { kEv, 108, false } } },
	{	0, 1154, "NCS54",
		{ { kEv, 228, true }, { kEv, 113, false } } },
	{	0, 1156, "NCS56",
		{ { kEv, 314, true }, { kEv, 345, false }, { kEv, 112, false } } },
	{	0, 1157, "NCS57",
		{ { kIn, 34, true }, { kEv, 114, false } } },
	{	0, 1158, "NCS58",
		{ { kEv, 243, true }, { kEv, 357, false }, { kEv, 106, false } } },
	{	0, 1162, "NCS62",
		{ { kIn, 3, true }, { kEv, 360, false }, { kEv, 109, false } } },
	{	0, 1165, "NCS65",
		{ { kEv, 315, true }, { kEv, 229, false }, { kEv, 116, true }, { kEv, 337, true }, { kEv, 115, false } } },
	{	0, 1166, "NCS66",
		{ { kEv, 315, true }, { kEv, 229, true }, { kEv, 116, true }, { kEv, 242, true }, { kEv, 337, true }, { kEv, 105, false } } },
	{	0, 1167, "NCS67",
		{ { kEv, 340, true }, { kEv, 162, false }, { kEv, 117, false } } },
	{	0, 1168, "NHS53",
		{ { kEv, 389, true }, { kEv, 355, false }, { kEv, 171, false }, { kEv, 188, false }, { kEv, 103, false } } },
},
{	// Harlan, 14 responses + 1 repeat
	{	0, 1450, "NHS50",
		{ { kEv, 179, false } } },
	{	0, 1451, "NHS08a",
		{ { kEv, 156, false } } },
	{	0, 1452, "NHS52",
		{ { kEv, 302, true }, { kEv, 286, false } } },
	{	0, 1453, "NHS53",
		{ { kEv, 233, true }, { kEv, 188, false }, { kEv, 389, true }, { kEv, 355, false }, { kEv, 171, false } } },
	{	0, 1454, "NHS54",
		{ { kEv, 233, true }, { kEv, 369, true }, { kEv, 153, false } } },
	{	0, 1455, "NHS55",
		{ { kEv, 233, true }, { kEv, 356, true }, { kIn, 23, true }, { kEv, 362, true }, { kEv, 170, false } } },
	{	0, 1456, "NHS56",
		{ { kEv, 325, true }, { kEv, 213, true }, { kEv, 274, false }, { kEv, 164, false } } },
	{	0, 1456, "NHS56",
		{ { kEv, 325, true }, { kEv, 392, true }, { kEv, 274, false }, { kEv, 164, false } } },
	{	0, 1457, "NHS57",
		{ { kEv, 243, true }, { kEv, 357, false }, { kEv, 158, false } } },
	{	0, 1458, "NHS58",
		{ { kEv, 244, true }, { kEv, 388, true }, { kEv, 385, false }, { kEv, 198, false }, { kEv, 167, false } } },
	{	0, 1459, "NHS59",
		{ { kIn, 24, true }, { kEv, 177, false } } },
	{	0, 1460, "NHS60",
		{ { kEv, 365, true }, { kIn, 23, true }, { kEv, 174, false } } },
	{	0, 1461, "NHS61",
		{ { kEv, 89, true }, { kEv, 172, false } } },
	{	0, 1463, "NHS56a",
		{ { kEv, 314, true }, { kEv, 325, true }, { kEv, 176, false } } },
	{	0, 1464, "NHS64",
		{ { kEv, 286, true }, { kEv, 526, false } } },
},
{	// Ingrid, 19 responses + 3 repeats
	{	0, 1050, "NIE08b",
		{ { kEv, 204, false } } },
	{	0, 1051, "NIE51",
		{ { kEv, 205, false } } },
	{	0, 1052, "NIE08a",
		{ { kEv, 195, false } } },
	{	0, 1053, "NIE53",
		{ { kEv, 231, true }, { kEv, 194, false }, { kEv, 199, false } } },
	{	0, 1054, "NIE54",
		{ { kEv, 233, true }, { kEv, 389, true }, { kEv, 355, false }, { kEv, 171, false }, { kEv, 188, false } } },
	{	0, 1055, "NIE55",
		{ { kEv, 233, true }, { kEv, 389, true }, { kEv, 355, false }, { kEv, 171, true }, { kEv, 188, false } } },
	{	0, 1056, "NIE56",
		{ { kEv, 383, true }, { kEv, 197, false } } },
	{	0, 1056, "NIE56",
		{ { kEv, 384, true }, { kEv, 197, false } } },
	{	0, 1056, "NIE56",
		{ { kEv, 388, true }, { kEv, 197, false } } },
	{	0, 1057, "NIE57",
		{ { kEv, 228, true }, { kEv, 186, false } } },
	{	0, 1058, "NIE58",
		{ { kIn, 24, true }, { kEv, 201, false } } },
	{	0, 1059, "NIE59",
		{ { kEv, 325, true }, { kEv, 213, true }, { kEv, 192, false } } },
	{	0, 1059, "NIE59",
		{ { kEv, 325, true }, { kEv, 392, true }, { kEv, 192, false } } },
	{	0, 1060, "NIE60",
		{ { kIn, 24, true }, { kEv, 201, true }, { kEv, 314, true }, { kEv, 191, false }, { kEv, 345, false } } },
	{	0, 1061, "NIE61",
		{ { kEv, 243, true }, { kEv, 357, false }, { kEv, 189, false } } },
	{	0, 1062, "NIE62",
		{ { kIn, 27, true }, { kEv, 355, true }, { kEv, 196, false } } },
	{	0, 1063, "NIE63",
		{ { kEv, 193, false } } },
	{	0, 1064, "NIE64",
		{ { kIn, 22, true }, { kEv, 297, true }, { kEv, 191, false }, { kEv, 190, false } } },
	{	0, 1065, "NIE65",
		{ { kEv, 334, true }, { kEv, 206, false } } },
	{	0, 1067, "NIE67",
		{ { kIn, 22, true }, { kEv, 297, true }, { kEv, 191, true }, { kEv, 190, false } } },
	{	0, 1068, "NIE68",
		{ { kEv, 391, true }, { kEv, 365, false }, { kEv, 202, false } } },
	{	0, 1070, "NIE70",
		{ { kEv, 415, true }, { kEv, 350, false }, { kEv, 200, false } } },
},
{	// Joy, 8 responses
	{	0, 1250, "NJO50",
		{ { kIn, 29, true }, { kEv, 234, true }, { kEv, 367, false }, { kEv, 217, false } } },
	{	0, 1251, "NJO51",
		{ { kEv, 232, true }, { kEv, 228, true }, { kEv, 234, true }, { kEv, 215, false } } },
	{	0, 1252, "NJO52",
		{ { kEv, 353, true }, { kIn, 28, true }, { kEv, 216, false } } },
	{	0, 1253, "NJO53",
		{ { kEv, 348, true }, { kEv, 354, false }, { kEv, 248, false } } },
	{	0, 1254, "NJO54",
		{ { kEv, 354, true }, { kIn, 12, false }, { kEv, 344, false } } },
	{	0, 1231, "NJO58",
		{ { kEv, 344, true }, { kEv, 367, true }, { kEv, 211, false } } },
	{	0, 1289, "NJO55",
		{ { kEv, 354, true }, { kIn, 12, true }, { kEv, 344, false } } },
	{	0, 1222, "NJO00e",
		{ { kEv, 234, true }, { kEv, 340, true }, { kIn, 21, true }, { kEv, 338, false } } },
},
{	// Paula, 12 responses + 1 repeat
	{	0, 1562, "NPS62",
		{ { kEv, 233, true }, { kEv, 267, false } } },
	{	0, 1551, "NPS51",
		{ { kEv, 325, true }, { kEv, 274, false }, { kEv, 314, false } } },
	{	0, 1552, "NPS52",
		{ { kEv, 314, true }, { kEv, 345, false }, { kEv, 275, false } } },
	{	0, 1553, "NPS53",
		{ { kEv, 336, true }, { kEv, 206, false }, { kEv, 279, false } } },
	{	0, 1553, "NPS53",
		{ { kEv, 336, true }, { kEv, 87, false }, { kEv, 279, false } } },
	{	0, 1554, "NPS54",
		{ { kEv, 389, true }, { kEv, 355, false }, { kEv, 171, false }, { kEv, 188, false }, { kEv, 270, false } } },
	{	0, 1555, "NPS55",
		{ { kEv, 237, true }, { kEv, 233, true }, { kEv, 280, false } } },
	{	0, 1556, "NPS56",
		{ { kEv, 237, true }, { kEv, 272, false } } },
	{	0, 1557, "NPS57",
		{ { kEv, 313, true }, { kEv, 305, true }, { kEv, 85, false }, { kEv, 281, false }, { kEv, 271, false } } },
	{	0, 1558, "NPS58",
		{ { kEv, 112, true }, { kEv, 277, false } } },
	{	0, 1559, "NPS59",
		{ { kEv, 85, true }, { kEv, 92, true }, { kEv, 316, true }, { kEv, 221, false }, { kEv, 273, false } } },
	{	0, 1560, "NPS60",
		{ { kEv, 352, true }, { kEv, 331, true }, { kEv, 86, false }, { kEv, 282, false } } },
	{	0, 1561, "NPS61",
		{ { kIn, 29, true }, { kEv, 367, false }, { kEv, 276, false } } },
},
{	// Bess & George, 34 responses + 1 repeat
	{	0, 1650, "NBG50",
		{ { kEv, 231, false }, { kEv, 69, false } } },
	{	0, 1651, "NBG51",
		{ { kEv, 231, false }, { kEv, 69, true }, { kEv, 317, true }, { kEv, 65, false } } },
	{	0, 1652, "NBG52",
		{ { kEv, 231, true }, { kEv, 156, true }, { kEv, 53, false } } },
	{	0, 1653, "NBG53",
		{ { kEv, 233, false }, { kEv, 237, false }, { kEv, 49, false }, { kEv, 281, false }, { kEv, 271, false } } },
	{	0, 1654, "NBG54",
		{ { kEv, 233, true }, { kEv, 234, false }, { kEv, 77, false } } },
	{	0, 1655, "NBG55",
		{ { kEv, 281, true }, { kEv, 227, false }, { kEv, 78, false } } },
	{	0, 1655, "NBG55",
		{ { kEv, 271, true }, { kEv, 227, false }, { kEv, 78, false } } },
	{	0, 1656, "NBG55", // Is this an error?
		{ { kEv, 227, true }, { kEv, 291, false }, { kEv, 78, false } } },
	{	0, 1657, "NBG57",
		{ { kEv, 232, true }, { kEv, 56, false } } },
	{	0, 1658, "NBG58",
		{ { kEv, 232, true }, { kEv, 56, true }, { kEv, 298, true }, { kEv, 63, false } } },
	{	0, 1659, "NBG59",
		{ { kIn, 29, true }, { kEv, 393, true }, { kEv, 64, false } } },
	{	0, 1660, "NBG60",
		{ { kEv, 234, true }, { kEv, 243, false }, { kEv, 74, false } } },
	{	0, 1661, "NBG61",
		{ { kEv, 228, true }, { kEv, 70, false } } },
	{	0, 1662, "NBG62",
		{ { kEv, 340, true }, { kEv, 52, false } } },
	{	0, 1664, "NBG64",
		{ { kEv, 64, true }, { kEv, 75, false }, { kEv, 76, false } } },
	{	0, 1665, "NBG65",
		{ { kEv, 111, true }, { kEv, 78, true }, { kEv, 68, false } } },
	{	0, 1666, "NBG66",
		{ { kEv, 243, true }, { kEv, 106, false }, { kEv, 74, true }, { kEv, 72, false } } },
	{	0, 1667, "NBG67",
		{ { kEv, 367, true }, { kEv, 78, true }, { kEv, 246, false }, { kEv, 62, false } } },
	{	0, 1668, "NBG68",
		{ { kIn, 24, true }, { kEv, 238, false }, { kEv, 66, false } } },
	{	0, 1669, "NBG69",
		{ { kEv, 325, true }, { kEv, 274, false }, { kEv, 314, false }, { kEv, 307, false }, { kEv, 73, false } } },
	{	0, 1670, "NBG70",
		{ { kEv, 201, true }, { kEv, 66, true }, { kEv, 60, false } } },
	{	0, 1671, "NBG71",
		{ { kEv, 342, true }, { kEv, 72, true }, { kEv, 244, false }, { kEv, 47, false } } },
	{	0, 1672, "NBG72",
		{ { kIn, 1, true }, { kEv, 244, true }, { kEv, 348, false }, { kEv, 59, false } } },
	{	0, 1673, "NBG73",
		{ { kEv, 216, true }, { kEv, 354, false }, { kEv, 80, false } } },
	{	0, 1674, "NBG74",
		{ { kEv, 248, true }, { kEv, 238, false }, { kEv, 80, true }, { kEv, 55, false } } },
	{	0, 1675, "NBG75",
		{ { kEv, 209, true }, { kEv, 289, false } } },
	{	0, 1676, "NBG76",
		{ { kEv, 92, true }, { kEv, 78, true }, { kEv, 221, false }, { kEv, 54, false } } },
	{	0, 1677, "NBG77",
		{ { kEv, 316, true }, { kEv, 54, true }, { kEv, 347, false }, { kEv, 79, false } } },
	{	0, 1678, "NBG78",
		{ { kEv, 334, true }, { kEv, 396, true }, { kEv, 87, false }, { kEv, 50, false } } },
	{	0, 1679, "NBG79",
		{ { kEv, 206, true }, { kEv, 57, false } } },
	{	0, 1680, "NBG80",
		{ { kEv, 178, true }, { kEv, 168, false }, { kEv, 287, true }, { kEv, 81, false } } },
	{	0, 1681, "NBG81",
		{ { kEv, 367, true }, { kEv, 211, true }, { kEv, 62, false }, { kEv, 58, false } } },
	{	0, 1682, "NBG82",
		{ { kEv, 168, true }, { kIn, 17, true }, { kEv, 297, true }, { kEv, 365, false }, { kEv, 61, false } } },
	{	0, 1687, "NBG87",
		{ { kDi, 2, true }, { kEv, 224, true }, { kEv, 48, false } } },
	{	0, 1689, "NBG87",
		{ { kDi, 0, true }, { kEv, 224, true } } },
},
{	// Frank & Joe, 21 responses + 2 repeats
	{	0, 1750, "NFJ50",
		{ { kEv, 231, true }, { kEv, 148, false } } },
	{	0, 1751, "NFJ51",
		{ { kEv, 231, true }, { kEv, 389, false }, { kEv, 232, false }, { kEv, 228, false }, { kEv, 148, false } } },
	{	0, 1751, "NFJ51",
		{ { kEv, 233, true }, { kEv, 389, false }, { kEv, 232, false }, { kEv, 228, false }, { kEv, 148, false } } },
	{	0, 1752, "NFJ52",
		{ { kEv, 311, true }, { kEv, 343, false }, { kEv, 147, false } } },
	{	0, 1753, "NFJ53",
		{ { kEv, 336, true }, { kEv, 132, false } } },
	{	0, 1754, "NFJ54",
		{ { kEv, 227, true }, { kEv, 111, true }, { kEv, 146, false } } },
	{	0, 1755, "NFJ55",
		{ { kEv, 340, true }, { kEv, 353, false }, { kEv, 145, false } } },
	{	0, 1756, "NFJ56",
		{ { kEv, 221, false }, { kEv, 92, true }, { kEv, 85, false }, { kEv, 111, true }, { kEv, 133, false } } },
	{	0, 1757, "NFJ57",
		{ { kEv, 221, false }, { kEv, 92, true }, { kEv, 111, true }, { kEv, 133, true }, { kEv, 85, false }, { kEv, 135, false } } },
	{	0, 1758, "NFJ58",
		{ { kEv, 221, false }, { kEv, 92, true }, { kEv, 111, true }, { kEv, 133, true }, { kEv, 85, false }, { kEv, 135, true }, { kEv, 134, false } } },
	{	0, 1759, "NFJ59",
		{ { kEv, 221, false }, { kEv, 92, true }, { kEv, 111, true }, { kEv, 133, true }, { kEv, 85, false }, { kEv, 135, true }, { kEv, 134, true }, { kEv, 130, false } } },
	{	0, 1760, "NFJ60",
		{ { kEv, 234, true }, { kEv, 243, false }, { kEv, 129, false } } },
	{	0, 1761, "NFJ61",
		{ { kEv, 234, true }, { kEv, 243, true }, { kEv, 129, true }, { kEv, 142, false } } },
	{	0, 1762, "NFJ62",
		{ { kEv, 361, true }, { kEv, 238, false }, { kEv, 147, true }, { kEv, 143, false } } },
	{	0, 1763, "NFJ63",
		{ { kEv, 85, true }, { kEv, 146, true }, { kEv, 92, true }, { kEv, 87, true }, { kEv, 206, false }, { kEv, 221, false }, { kEv, 141, false } } },
	{	0, 1764, "NFJ64",
		{ { kEv, 213, true }, { kEv, 314, true }, { kEv, 345, false }, { kEv, 325, true }, { kEv, 136, false } } },
	{	0, 1764, "NFJ64",
		{ { kEv, 392, true }, { kEv, 314, true }, { kEv, 345, false }, { kEv, 325, true }, { kEv, 136, false } } },
	{	0, 1765, "NFJ65",
		{ { kEv, 238, true }, { kEv, 341, false }, { kEv, 139, false } } },
	{	0, 1766, "NFJ66",
		{ { kEv, 221, true }, { kEv, 297, false }, { kEv, 144, false }, { kEv, 133, true } } },
	{	0, 1767, "NFJ67",
		{ { kEv, 297, true }, { kEv, 149, false } } },
	{	0, 1768, "NFJ68",
		{ { kEv, 394, true }, { kEv, 128, false } } },
	{	0, 1769, "NFJ69",
		{ { kEv, 209, true }, { kEv, 206, true } , { kEv, 178, true } , { kEv, 101, true } , { kEv, 138, false } } },
	{	0, 1775, "NFJ75",
		{ { kEv, 230, true } } },
},
{	// Perris, 3 responses
	{	0, 1850, "NJP50",
		{ { kEv, 111, true }, { kEv, 291, false } } },
	{	0, 1851, "NJP51",
		{ { kEv, 93, false } } },
	{	0, 1852, "NJP52",
		{ { kEv, 178, false }, { kEv, 85, true }, { kEv, 95, false } } },
},
{	// Anton, 7 responses + 6 repeats
	{	0, 1950, "NAS50",
		{ { kEv, 245, false }, { kEv, 122, false } } },
	{	0, 1950, "NAS50",
		{ { kEv, 125, true }, { kEv, 122, false } } },
	{	0, 1951, "NAS51",
		{ { kEv, 245, false }, { kEv, 127, false } } },
	{	0, 1951, "NAS51",
		{ { kEv, 125, true }, { kEv, 127, false } } },
	{	0, 1952, "NAS52",
		{ { kEv, 245, false }, { kEv, 127, true }, { kEv, 124, true } } },
	{	0, 1952, "NAS52",
		{ { kEv, 125, true }, { kEv, 127, true }, { kEv, 124, true } } },
	{	0, 1953, "NAS53",
		{ { kEv, 245, false }, { kEv, 120, true }, { kEv, 123, false } } },
	{	0, 1953, "NAS53",
		{ { kEv, 125, true }, { kEv, 120, true }, { kEv, 123, false } } },
	{	0, 1954, "NAS54",
		{ { kEv, 245, false }, { kEv, 122, true }, { kEv, 126, false } } },
	{	0, 1954, "NAS54",
		{ { kEv, 125, true }, { kEv, 122, true }, { kEv, 126, false } } },
	{	0, 1956, "NAS56",
		{ { kEv, 245, true }, { kEv, 125, false } } },
	{	0, 1957, "NAS57",
		{ { kEv, 245, false }, { kEv, 315, true }, { kEv, 131, true }, { kEv, 121, false } } },
	{	0, 1957, "NAS57",
		{ { kEv, 125, true }, { kEv, 315, true }, { kEv, 131, true }, { kEv, 121, false } } },
},
{	// Operator, 8 responses
	{	0, 1960, "NOP60",
		{ { kEv, 310, true }, { kEv, 258, false }, { kEv, 263, false }, { kEv, 261, false } } },
	{	0, 1961, "NOP61",
		{ { kEv, 310, true }, { kEv, 261, false }, { kEv, 262, false }, { kEv, 260, false } } },
	{	0, 1962, "NOP62",
		{ { kEv, 308, true }, { kEv, 106, false }, { kEv, 259, false }, { kEv, 262, false }, { kEv, 263, false } } },
	{	0, 1963, "NOP63",
		{ { kEv, 106, true }, { kEv, 308, false }, { kEv, 257, false }, { kEv, 262, false }, { kEv, 263, false } } },
	{	0, 1964, "NOP64",
		{ { kEv, 106, true }, { kEv, 308, true }, { kEv, 257, false }, { kEv, 262, false }, { kEv, 263, false } } },
	{	0, 1965, "NOP64", // Is this an error?
		{ { kEv, 257, true }, { kEv, 261, false }, { kEv, 262, false }, { kEv, 263, false } } },
	{	0, 1966, "NOP66",
		{ { kEv, 258, true }, { kEv, 261, false }, { kEv, 262, false }, { kEv, 263, false } } },
	{	0, 1967, "NOP67",
		{ { kEv, 260, true }, { kEv, 261, false }, { kEv, 262, false }, { kEv, 263, false } } },
},
{	// Tink, 4 responses
	{	0, 1970, "NTI70",
		{ { kEv, 182, true }, { kEv, 266, false }, { kEv, 376, false } } },
	{	0, 1971, "NTI71",
		{ { kEv, 341, true }, { kEv, 265, false }, { kEv, 374, false } } },
	{	0, 1972, "NTI72",
		{ { kEv, 245, true }, { kEv, 125, false }, { kEv, 373, false } } },
	{	0, 1973, "NTI73",
		{ { kEv, 375, false } } },
}
};

const Common::Array<Goodbye> _nancy8Goodbyes = {
	{ "NCS90", { { { 1190, 1191, 1192, 1193 }, {}, NOFLAG } } },	// Elliott
	{ "NHS90", { { { 1490, 1491, 1492, 1493 }, {}, NOFLAG } } },	// Harlan
	{ "NIE90", { { { 1090, 1091, 1092, 1093 }, {}, NOFLAG } } },	// Ingrid
	{ "NJO90", { { { 1290, 1291, 1292, 1293 }, {}, NOFLAG } } },	// Joy
	{ "NPS90", { { { 1590, 1591, 1592, 1593 }, {}, NOFLAG } } },	// Paula
	{ "NBG90", { { { 1690, 1691, 1692, 1693 }, {}, NOFLAG } } },	// Bess & George
	{ "NFJ90", { { { 1790, 1791, 1792, 1793 }, {}, NOFLAG } } },	// Frank & Joe
	{ "NKJ90", { { { 1890, 1891, 1892, 1893 }, {}, NOFLAG } } },	// Perris
	{ "NAS90", { { { 1990, 1991, 1992, 1993 }, {}, NOFLAG } } },	// Anton
	{ "NOP95", { { { 1995, 1996, 1997, 1998 }, {}, NOFLAG } } },	// Operator
	{ "NTI85", { { { 1985, 1986, 1987 }, {}, NOFLAG } } }			// Tink
};

// The Telephone AR was replaced with the NewPhone AR in nancy8. Since the Telephone
// string is no longer used, we store the NewPhone string only.
const Common::Array<const char *> _nancy8TelephoneRinging = {
	"dialing...<n><e>", // English
	"dialing...<n><e>" // Russian. Not translated, probably because the new AR draws in the viewport
};

const Common::Array<const char *> _nancy8EventFlagNames = {
	"EV_Generic0",
	"EV_Generic1",
	"EV_Generic2",
	"EV_Generic3",
	"EV_Generic4",
	"EV_Generic5",
	"EV_Generic6",
	"EV_Generic7",
	"EV_Generic8",
	"EV_Generic9",
	"EV_Generic10",
	"EV_Generic11",
	"EV_Generic12",
	"EV_Generic13",
	"EV_Generic14",
	"EV_Generic15",
	"EV_Generic16",
	"EV_Generic17",
	"EV_Generic18",
	"EV_Generic19",
	"EV_Generic20",
	"EV_Generic21",
	"EV_Generic22",
	"EV_Generic23",
	"EV_Generic24",
	"EV_Generic25",
	"EV_Generic26",
	"EV_Generic27",
	"EV_Generic28",
	"EV_Generic29",
	"EV_Generic30",
	"EV_TimeForEndgame",
	"EV_PlayerWonGame",
	"EV_StopPlayerScrolling",
	"EV_Easter_Eggs",
	"EV_Award_Avid",
	"EV_Award_Face",
	"EV_Award_Game",
	"EV_Award_Gluttony",
	"EV_Award_Iron",
	"EV_Award_Magellan",
	"EV_Award_Nine",
	"EV_Award_Open",
	"EV_Award_Puzzle",
	"EV_Award_Tenacity",
	"EV_Band_Organ_On",
	"EV_BG_Called",
	"EV_BG_Said_Ate_Fundae",
	"EV_BG_Said_Call_Hardys",
	"EV_BG_Said_Call_PS",
	"EV_BG_Said_Call_Victim",
	"EV_BG_Said_Called",
	"EV_BG_Said_Coaster",
	"EV_BG_Said_Cute",
	"EV_BG_Said_HB_Fits",
	"EV_BG_Said_Headcase",
	"EV_BG_Said_IC_Cute",
	"EV_BG_Said_IC_Still_Guilty",
	"EV_BG_Said_Know_Name",
	"EV_BG_Said_Lemon",
	"EV_BG_Said_List",
	"EV_BG_Said_Lured",
	"EV_BG_Said_Memo",
	"EV_BG_Said_Money",
	"EV_BG_Said_Need_Help",
	"EV_BG_Said_Neither",
	"EV_BG_Said_Next_Stop",
	"EV_BG_Said_No_Call",
	"EV_BG_Said_Not_Original",
	"EV_BG_Said_Not_Yet",
	"EV_BG_Said_Procrastinator",
	"EV_BG_Said_Radius",
	"EV_BG_Said_Riddle",
	"EV_BG_Said_Romantic",
	"EV_BG_Said_Snoop",
	"EV_BG_Said_Steno_Done",
	"EV_BG_Said_Still_Working",
	"EV_BG_Said_Strange_Drink",
	"EV_BG_Said_Treasure",
	"EV_BG_Said_Trespass",
	"EV_BG_Said_Twisted",
	"EV_BG_Said_Wrong",
	"EV_BG_Steno_Ready1",
	"EV_BG_Steno_Ready2",
	"EV_BG_Steno_Ready3",
	"EV_Called_Luis",
	"EV_Called_Tink",
	"EV_Called_Victim",
	"EV_Can_Meet_EC",
	"EV_Depleted_Fun_Pass",
	"EV_Detective_Ready",
	"EV_Detective_Said_Case",
	"EV_Detective_Said_Cellmate",
	"EV_Detective_Said_Darryl",
	"EV_Detective_Said_ID",
	"EV_Detective_Said_No_HB",
	"EV_Detective_Said_No_Info",
	"EV_Detective_Said_PS_Called",
	"EV_EC_Available",
	"EV_EC_Escape",
	"EV_EC_Said_Break",
	"EV_EC_Said_Bye",
	"EV_EC_Said_Carousel_Book",
	"EV_EC_Said_Check",
	"EV_EC_Said_Dream",
	"EV_EC_Said_Forgery",
	"EV_EC_Said_Fundae",
	"EV_EC_Said_Junk",
	"EV_EC_Said_Know_Poppy",
	"EV_EC_Said_Lathe",
	"EV_EC_Said_No_Motive",
	"EV_EC_Said_Not_Original",
	"EV_EC_Said_Nutty",
	"EV_EC_Said_Others",
	"EV_EC_Said_Receipt",
	"EV_EC_Said_Remember",
	"EV_EC_Said_Replacement",
	"EV_EC_Said_Robocop",
	"EV_EV_BG_Said_Call_PS",
	"EV_Expert_Said_Apologize",
	"EV_Expert_Said_Black_Market",
	"EV_Expert_Said_Forgeries",
	"EV_Expert_Said_Heard",
	"EV_Expert_Said_How_Sell",
	"EV_Expert_Said_Lead",
	"EV_Expert_Said_Nickname",
	"EV_Expert_Said_Publicity",
	"EV_Expert_Said_Worth",
	"EV_FJ_Said_Cheery",
	"EV_FJ_Said_Computer",
	"EV_FJ_Said_EC_Suspect",
	"EV_FJ_Said_Forgeries",
	"EV_FJ_Said_Fraud",
	"EV_FJ_Said_HB_Suspect",
	"EV_FJ_Said_IC_Suspect",
	"EV_FJ_Said_JT_Suspect",
	"EV_FJ_Said_Kessler",
	"EV_FJ_Said_Key",
	"EV_FJ_Said_Narrow",
	"EV_FJ_Said_No_Repair",
	"EV_FJ_Said_No_Solve",
	"EV_FJ_Said_Nosy",
	"EV_FJ_Said_Ratted",
	"EV_FJ_Said_Remote",
	"EV_FJ_Said_Scratch",
	"EV_FJ_Said_Spirits",
	"EV_FJ_Said_Stashed",
	"EV_FJ_Said_Static",
	"EV_FJ_Said_Strategy",
	"EV_FJ_Said_Trap",
	"EV_Got_Fundae",
	"EV_HB_Available",
	"EV_HB_Said_Begged",
	"EV_HB_Said_EC_Behind",
	"EV_HB_Said_Elliott_Joy",
	"EV_HB_Said_Envelope",
	"EV_HB_Said_Flemington",
	"EV_HB_Said_Glitch",
	"EV_HB_Said_Huh",
	"EV_HB_Said_Impatient",
	"EV_HB_Said_Impatientl",
	"EV_HB_Said_Ingrid",
	"EV_HB_Said_JT_Pencil",
	"EV_HB_Said_Lady",
	"EV_HB_Said_Letter",
	"EV_HB_Said_Meditation",
	"EV_HB_Said_No_Biggie",
	"EV_HB_Said_Offline",
	"EV_HB_Said_Please",
	"EV_HB_Said_Prove",
	"EV_HB_Said_Reader_On",
	"EV_HB_Said_Red_Tag",
	"EV_HB_Said_Refill",
	"EV_HB_Said_Report",
	"EV_HB_Said_Spook10_On",
	"EV_HB_Said_Still",
	"EV_HB_Said_There",
	"EV_HB_Said_Toy",
	"EV_HB_Said_Upset",
	"EV_HB_Said_Weeks",
	"EV_HB_Said_Whale",
	"EV_HB_Said_You_Better",
	"EV_Heard_Tune",
	"EV_IC_Available",
	"EV_IC_Said_Curse",
	"EV_IC_Said_Dabbled",
	"EV_IC_Said_EC_Strange",
	"EV_IC_Said_Elliott_Gun",
	"EV_IC_Said_Fix_Reader",
	"EV_IC_Said_Fool",
	"EV_IC_Said_Forgetful",
	"EV_IC_Said_Hidden_Door",
	"EV_IC_Said_House_Workshop",
	"EV_IC_Said_Loves_Job",
	"EV_IC_Said_Niacin",
	"EV_IC_Said_No_Blueprints",
	"EV_IC_Said_Not_Genius",
	"EV_IC_Said_Online",
	"EV_IC_Said_Pliers",
	"EV_IC_Said_Psycho",
	"EV_IC_Said_Reprogram",
	"EV_IC_Said_See_Device",
	"EV_IC_Said_Spook10_Part",
	"EV_IC_Said_Still",
	"EV_IC_Said_Trenton",
	"EV_IC_Said_Valuable",
	"EV_IC_Said_Victim",
	"EV_Iron_On",
	"EV_JT_Available",
	"EV_JT_Said_Apologize",
	"EV_JT_Said_Awful",
	"EV_JT_Said_Glory",
	"EV_JT_Said_Got_Tune",
	"EV_JT_Said_Irony",
	"EV_JT_Said_Remember",
	"EV_JT_Said_Socializing",
	"EV_JT_Said_Spoon",
	"EV_JT_Said_Steno",
	"EV_Left_Note",
	"EV_Loaded_Roll",
	"EV_Luis_Available",
	"EV_Luis_Said_HB_Innocent",
	"EV_Made_Cut",
	"EV_Measured_Dowel",
	"EV_Met_BG",
	"EV_Met_Carousel",
	"EV_Met_Coaster",
	"EV_Met_Detective",
	"EV_Met_EC",
	"EV_Met_Expert",
	"EV_Met_FJ",
	"EV_Met_HB",
	"EV_Met_IC",
	"EV_Met_JT",
	"EV_Met_Miles",
	"EV_Met_Op",
	"EV_Met_Park",
	"EV_Met_PS",
	"EV_Met_Tink",
	"EV_Miles_Gives_Paper",
	"EV_Miles_Misc1",
	"EV_Miles_Misc2",
	"EV_Miles_Said_Not_Lead",
	"EV_Miles_Said_Quest1",
	"EV_Miles_Said_Quest2",
	"EV_Miles_Said_Quest3",
	"EV_Miles_Said_Remember_Name",
	"EV_Miles_Said_Snooping",
	"EV_Miles_Said_Tune_Hint",
	"EV_Miles_Said_Welcome",
	"EV_Miles_Takes_Spoon",
	"EV_Miles_Wants_Spoon",
	"EV_Modern2",
	"EV_Note_Stall",
	"EV_Op_Said_Another_Burger",
	"EV_Op_Said_Another_Fundae",
	"EV_Op_Said_Another_Pasta",
	"EV_Op_Said_Fundae",
	"EV_Op_Said_Hamburger",
	"EV_Op_Said_No_Fundae",
	"EV_Op_Said_Pasta",
	"EV_Ordered_Fundae",
	"EV_Ordered_Hamburger",
	"EV_Ordered_Pasta",
	"EV_Placed_Blank_Dowel",
	"EV_Placed_New_Dowel",
	"EV_Played_Tune",
	"EV_PS_Said_Charisma",
	"EV_PS_Said_Cobwebs",
	"EV_PS_Said_Dreary",
	"EV_PS_Said_IC_Prob",
	"EV_PS_Said_Interesting",
	"EV_PS_Said_Investigators",
	"EV_PS_Said_Knows",
	"EV_PS_Said_Letter",
	"EV_PS_Said_No_Key",
	"EV_PS_Said_No_Steno",
	"EV_PS_Said_Putter",
	"EV_PS_Said_Shoot",
	"EV_PS_Said_Squeeze",
	"EV_PS_Said_Theft_Motive",
	"EV_PS_Said_Thought",
	"EV_PS_Said_Tink",
	"EV_Said_Break",
	"EV_Said_Cabinet_Snoop",
	"EV_Said_EC_Suspect",
	"EV_Said_Haunting_Tape",
	"EV_Said_HB_Fits",
	"EV_Said_IC_Suspect",
	"EV_Said_JT_Responsible",
	"EV_Said_Need_Gun",
	"EV_Said_Not_Original",
	"EV_Said_Nothing",
	"EV_Said_PS_Message",
	"EV_Said_Tink_Call",
	"EV_Said_Under_Carousel",
	"EV_Saw_Appo_Book",
	"EV_Saw_Booby_Trap",
	"EV_Saw_Brochure",
	"EV_Saw_Carousel_Band_Organs",
	"EV_Saw_Carousel_Carvers",
	"EV_Saw_Carousel_History",
	"EV_Saw_Carousel_Monitor",
	"EV_Saw_CC_PD",
	"EV_Saw_Dada",
	"EV_Saw_Detective",
	"EV_Saw_Empty_Office",
	"EV_Saw_Expert_Number",
	"EV_Saw_Fundae",
	"EV_Saw_Gear_Trap_Door",
	"EV_Saw_Hamburger",
	"EV_Saw_Haunting",
	"EV_Saw_HB_Gone",
	"EV_Saw_Heist_Article",
	"EV_Saw_Hidden_Door",
	"EV_Saw_Horse_Tails",
	"EV_Saw_Ingrid_Tape",
	"EV_Saw_Kessler_Name",
	"EV_Saw_Locked_Door",
	"EV_Saw_Luis",
	"EV_Saw_Modern_Game",
	"EV_Saw_Pasta",
	"EV_Saw_Police_Number",
	"EV_Saw_Radiator",
	"EV_Saw_Remote_Logo",
	"EV_Saw_Rolfe_Letter",
	"EV_Saw_Rolfe_Letter_Stuck",
	"EV_Saw_Roll_Cabinet",
	"EV_Saw_Squid",
	"EV_Saw_Stairs_Collapse",
	"EV_Saw_Tile_Game",
	"EV_Saw_Tink_Note",
	"EV_Saw_Tink_Number",
	"EV_Saw_Vent",
	"EV_Saw_Victim_Number",
	"EV_Saw_Watch_Receipt",
	"EV_Saw_Whiplash_Article",
	"EV_Saw_Wood_Receipt",
	"EV_Showed_JT_Pencil",
	"EV_Solved_Brass_Ring",
	"EV_Solved_Coaster",
	"EV_Solved_Dowel",
	"EV_Solved_Fundae",
	"EV_Solved_Gear_Room",
	"EV_Solved_Harmonica_Tune",
	"EV_Solved_Hidden_Door",
	"EV_Solved_House",
	"EV_Solved_IC_Keypad",
	"EV_Solved_Lemon",
	"EV_Solved_Locker",
	"EV_Solved_Midway_Repair",
	"EV_Solved_Nickname_Spin",
	"EV_Solved_Organ",
	"EV_Solved_Quest1",
	"EV_Solved_Quest2",
	"EV_Solved_Reader",
	"EV_Solved_Resistors",
	"EV_Solved_Riddle",
	"EV_Solved_Rolfe_Letter",
	"EV_Solved_Roll_Cabinet",
	"EV_Solved_Roll_Frame",
	"EV_Solved_Skewer",
	"EV_Solved_Solder",
	"EV_Solved_Spook_Arm",
	"EV_Solved_Spook_Red_Tag",
	"EV_Solved_Spook10",
	"EV_Solved_Spook10_Ring",
	"EV_Solved_Steno_Horse",
	"EV_Solved_Tissues",
	"EV_Spilled_Paint",
	"EV_Sticky_Pos01",
	"EV_Tink_Said_Combination",
	"EV_Tink_Said_No_Fish",
	"EV_TO_Said_Book",
	"EV_TO_Said_How_Load",
	"EV_TO_Said_Name",
	"EV_TO_Said_Notes",
	"EV_Took_Broken_Dowel",
	"EV_Took_Ingrid_Tape",
	"EV_Took_Key",
	"EV_Trap_Door_Open",
	"EV_Tried_Brass_Ring",
	"EV_Tried_Carousel",
	"EV_Tried_Gear_Room",
	"EV_Tried_House",
	"EV_Tried_IC",
	"EV_Tried_IC_Post_Coaster",
	"EV_Tried_Lathe",
	"EV_Tried_Organ",
	"EV_Tried_Reader",
	"EV_Tried_Santos",
	"EV_Tried_Spook10",
	"EV_Said_Comment01",
	"EV_Said_Comment02",
	"EV_Said_Comment03",
	"EV_Said_Comment04",
	"EV_Said_Comment05",
	"EV_Said_Comment06",
	"EV_Said_Comment07",
	"EV_Said_Comment08",
	"EV_Said_Comment09",
	"EV_Said_Comment10",
	"EV_Said_Comment11",
	"EV_Said_Comment12",
	"EV_Said_Comment13",
	"EV_Said_Comment14",
	"EV_Said_Comment15",
	"EV_Said_Comment16",
	"EV_Said_Comment17",
	"EV_Said_Comment18",
	"EV_Said_Comment19",
	"EV_Said_Comment20",
	"EV_Said_Comment21",
	"EV_Said_Comment22",
	"EV_Said_Comment23",
	"EV_Said_Comment24",
	"EV_Said_Comment25",
	"EV_Said_Comment26",
	"EV_Said_Comment27",
	"EV_Said_Comment28",
	"EV_Said_Comment29",
	"EV_Said_Comment30",
	"EV_Said_Comment31",
	"EV_Said_Comment32",
	"EV_Said_Comment33",
	"EV_Said_Comment34",
	"EV_Said_Comment35",
	"EV_Said_Comment36",
	"EV_Said_Comment37",
	"EV_Said_Comment38",
	"EV_Said_Comment39",
	"EV_Said_Comment40",
	"EV_Said_Comment41",
	"EV_Said_Comment42",
	"EV_Said_Comment43",
	"EV_Said_Comment44",
	"EV_Said_Comment45",
	"EV_Said_Comment46",
	"EV_Said_Comment47",
	"EV_Said_Comment48",
	"EV_Said_Comment49",
	"EV_Said_Comment50",
	"EV_NotUsed1",
	"EV_NotUsed2",
	"EV_NotUsed3",
	"EV_NotUsed4",
	"EV_NotUsed5",
	"EV_NotUsed6",
	"EV_NotUsed7",
	"EV_NotUsed8",
	"EV_NotUsed9",
	"EV_NotUsed10",
	"EV_NotUsed11",
	"EV_NotUsed12",
	"EV_NotUsed13",
	"EV_NotUsed14",
	"EV_NotUsed15",
	"EV_NotUsed16",
	"EV_NotUsed17",
	"EV_NotUsed18",
	"EV_NotUsed19",
	"EV_NotUsed20",
	"EV_NotUsed21",
	"EV_NotUsed22",
	"EV_NotUsed23",
	"EV_NotUsed24",
	"EV_NotUsed25",
	"EV_NotUsed26",
	"EV_NotUsed27",
	"EV_NotUsed28",
	"EV_NotUsed29",
	"EV_NotUsed30",
	"EV_NotUsed31",
	"EV_NotUsed32",
	"EV_NotUsed33",
	"EV_NotUsed34",
	"EV_NotUsed35",
	"EV_NotUsed36",
	"EV_NotUsed37",
	"EV_NotUsed38",
	"EV_NotUsed39",
	"EV_NotUsed40",
	"EV_NotUsed41",
	"EV_NotUsed42",
	"EV_NotUsed43",
	"EV_NotUsed44",
	"EV_NotUsed45",
	"EV_NotUsed46",
	"EV_NotUsed47",
	"EV_NotUsed48",
	"EV_NotUsed49",
	"EV_NotUsed50",
	"EV_NotUsed51",
	"EV_NotUsed52",
	"EV_NotUsed53",
	"EV_NotUsed54",
	"EV_NotUsed55",
	"EV_NotUsed56",
	"EV_NotUsed57",
	"EV_NotUsed58",
	"EV_NotUsed59",
	"EV_Empty1",
	"EV_Empty2",
	"EV_Empty3",
	"EV_Empty4",
	"EV_Empty5",
	"EV_Empty6",
	"EV_Empty7",
	"EV_Empty8",
	"EV_Empty9",
	"EV_Empty10",
	"EV_Empty11",
	"EV_Empty12",
	"EV_Empty13",
	"EV_Empty14",
	"EV_Empty15",
	"EV_Empty16",
	"EV_Empty17",
	"EV_Empty18",
	"EV_Empty19",
	"EV_Empty20",
	"EV_Empty21",
	"EV_Empty22",
	"EV_Empty23",
	"EV_Empty24",
	"EV_Empty25",
	"EV_Empty26",
	"EV_Empty27",
	"EV_Empty28",
	"EV_Empty29",
	"EV_Empty30",
	"EV_Empty31",
	"EV_Empty32",
	"EV_Empty33",
	"EV_Empty34",
	"EV_Empty35",
	"EV_Empty36",
	"EV_Empty37",
	"EV_Empty38",
	"EV_Empty39",
	"EV_Empty40",
	"EV_Empty41",
	"EV_Empty42",
	"EV_Empty43",
	"EV_Empty44",
	"EV_Empty45",
	"EV_Empty46",
	"EV_Empty47",
	"EV_Empty48",
	"EV_Empty49",
	"EV_Empty50",
	"EV_Empty51",
	"EV_Empty52",
	"EV_Empty53",
	"EV_Empty54",
	"EV_Empty55",
	"EV_Empty56",
	"EV_Empty57",
	"EV_Empty58",
	"EV_Empty59",
	"EV_Empty60",
	"EV_Empty61",
	"EV_Empty62",
	"EV_Empty63",
	"EV_Empty64",
	"EV_Empty65",
	"EV_Empty66",
	"EV_Empty67",
	"EV_Empty68",
	"EV_Empty69",
	"EV_Empty70",
	"EV_Empty71",
	"EV_Empty72",
	"EV_Empty73",
	"EV_Empty74",
	"EV_Empty75",
};

#endif // NANCY7DATA_H
