/**
 * Copyright (c) Tiny Technologies, Inc. All rights reserved.
 * Licensed under the LGPL or a commercial license.
 * For LGPL see License.txt in the project root for license information.
 * For commercial licenses see https://www.tiny.cloud/
 */

import { Cell, Optional, Throttler } from '@ephox/katamari';
import Editor from '../api/Editor';

// This is based heavily on Alloy's TapEvent.ts, just modified to use TinyMCE's event system.

const SIGNIFICANT_MOVE = 5;
const LONGPRESS_DELAY = 400;

export interface TouchHistoryData {
  readonly x: number;
  readonly y: number;
  readonly target: Node;
}

const getTouch = (event: TouchEvent): Optional<Touch> => {
  if (event.touches === undefined || event.touches.length !== 1) {
    return Optional.none();
  }
  return Optional.some(event.touches[0]);
};

const isFarEnough = (touch: Touch, data: TouchHistoryData): boolean => {
  const distX = Math.abs(touch.clientX - data.x);
  const distY = Math.abs(touch.clientY - data.y);
  return distX > SIGNIFICANT_MOVE || distY > SIGNIFICANT_MOVE;
};

const setup = (editor: Editor) => {
  const startData = Cell<Optional<TouchHistoryData>>(Optional.none());
  const longpressFired = Cell<boolean>(false);

  const debounceLongpress = Throttler.last((e) => {
    editor.fire('longpress', { ...e, type: 'longpress' });
    longpressFired.set(true);
  }, LONGPRESS_DELAY);

  editor.on('touchstart', (e) => {
    getTouch(e).each((touch) => {
      debounceLongpress.cancel();

      const data: TouchHistoryData = {
        x: touch.clientX,
        y: touch.clientY,
        target: e.target
      };

      debounceLongpress.throttle(e);
      longpressFired.set(false);
      startData.set(Optional.some(data));
    });
  }, true);

  editor.on('touchmove', (e) => {
    debounceLongpress.cancel();
    getTouch(e).each((touch) => {
      startData.get().each((data) => {
        if (isFarEnough(touch, data)) {
          startData.set(Optional.none());
          longpressFired.set(false);
          editor.fire('longpresscancel');
        }
      });
    });
  }, true);

  editor.on('touchend touchcancel', (e) => {
    debounceLongpress.cancel();

    if (e.type === 'touchcancel') {
      return;
    }

    // Cancel the touchend event if a longpress was fired, otherwise fire the tap event
    startData.get()
      .filter((data) => data.target.isEqualNode(e.target))
      .each(() => {
        if (longpressFired.get()) {
          e.preventDefault();
        } else {
          editor.fire('tap', { ...e, type: 'tap' });
        }
      });
  }, true);
};

export { setup };
