/**********************************************************************
 * $read_test_vector example -- C source code using TF/ACC PLI routines
 *
 * C source to read a test vector value from a file and apply the 
 * value to a system task argument.
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Usage:
 * ------
 * The $read_test_vector routine is intended to be used in a loop, 
 * where each time it is called, it reads the next vector from a file.
 * When the last vector has been read, the PLI routine will stop 
 * simulation.
 *
 *   Syntax:   $read_test_vector("<file_name>",<register_variable>);
 *
 *   Example:
 *    reg [11:0] vector;
 *    always @(negedge clk) $read_test_vector("vector.pat",vector);
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_ReadVector_checktf(),
 *              PLIbook_ReadVector_calltf(),
 *              PLIbook_ReadVector_misctf();
 *  /* table entries -/
 *   {usertask,                     /* type of PLI routine -/
 *     0,                           /* user_data value -/
 *     PLIbook_ReadVector_checktf,  /* checktf routine -/
 *     0,                           /* sizetf routine -/
 *     PLIbook_ReadVector_calltf,   /* calltf routine -/
 *     PLIbook_ReadVector_misctf,   /* misctf routine -/
 *     "$read_test_vector",         /* system task/function name -/
 *     1                            /* forward reference = true -/
 *   },
 *********************************************************************/

#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */
#include <stdlib.h>           /* ANSI C standard library */
#include <stdio.h>            /* ANSI C standard input/output library */
/**********************************************************************
 * checktf routine
 *********************************************************************/
int PLIbook_ReadVector_checktf()
{
  if (tf_nump() != 2) /* check for two system task/function */
    tf_error("Usage: $read_test_vector(\"<file\",<reg_variable>);");
  if (tf_typep(1) != TF_STRING) /* check that first arg is a string */
    tf_error("$read_test_vector arg 1 must be a quoted file name");
  if (tf_typep(2) != TF_READWRITE) /* check that 2nd arg is reg type */
    tf_error("$read_test_vector arg 2 must be a register data type");
  return(0);
}

/**********************************************************************
 * misctf routine
 *   Use the misctf routine to open test vector file at the beginning
 *   of simulation, and save the file pointer in the work area for the
 *   instance of $read_test_vector that called the misctf routine.
 *********************************************************************/
int PLIbook_ReadVector_misctf(int user_data, int reason)
{
  FILE *in_file;
  char *file_name;

  if (reason == REASON_ENDOFCOMPILE) { /* time to open vector file */
    if ( (in_file = fopen(tf_getcstringp(1),"r")) == NULL)
      tf_error("$read_test_vector cannot open file %s",
               tf_getcstringp(1));
    tf_setworkarea((char*)in_file); /* save file pointer in workarea */
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_ReadVector_calltf() {
  FILE *in_file;
  int   vec_size = tf_sizep(2);          /* bit size of tfarg 2   */
  char *vector = malloc(vec_size+1);     /* memory for vector string */
  bool  VERBOSE = FALSE;                 /* flag for debug output */

  if (mc_scan_plusargs("debug")) VERBOSE = TRUE; /* set verbose flag */

  in_file = (FILE*)tf_getworkarea();     /* retrieve file pointer */

  if ( (fscanf(in_file,"%s\n", vector)) == EOF) {   /* read a vector */
    tf_warning("$read_test_vector reached End-Of-File %s",
               tf_getcstringp(1));  /* get file name from task arg 1 */
    fclose(in_file);
    tf_dofinish();  /* exit simulation at end-of-file */
    return(0);
  }
  if (VERBOSE)
    io_printf("$read_test_vector: Value read from file=%s\n", vector);

  /* write test vector value onto system task arg 2 */
  if (!(tf_strdelputp(2,vec_size,'b',vector,0,0)) )
    if (VERBOSE)
      tf_error("$read_test_vector could not write to arg 2 at time %s",
               tf_strgettime() );
  return(0);
}
/*********************************************************************/
