/**********************************************************************
 * $list_pathout_ports example -- PLI application using VPI routines
 *
 * C source to find module ports which are connected to the outputs
 * of pin-to-pin path delays.
 *
 * Usage: $list_pathout_ports(<module_instance_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of the PLI application routines */
int PLIbook_ListPorts_compiletf(), PLIbook_ListPorts_calltf();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_ListPorts_register()
{
  s_vpi_systf_data tf_data;
  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$list_pathout_ports";
  tf_data.calltf    = PLIbook_ListPorts_calltf;
  tf_data.compiletf = PLIbook_ListPorts_compiletf;
  tf_data.sizetf    = NULL;

  vpi_register_systf(&tf_data);
}
/*********************************************************************/

/**********************************************************************
 * Compiletf application
 *********************************************************************/
int PLIbook_ListPorts_compiletf(char *user_data)
{
  vpiHandle systf_h, tfarg_itr, tfarg_h;

  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain handle to systf call\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  tfarg_itr = vpi_iterate(vpiArgument, systf_h);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain iterator to systf args\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  tfarg_h = vpi_scan(tfarg_itr);
  if (vpi_get(vpiType, tfarg_h) != vpiModule) {
    vpi_printf("ERROR: $list_pathout_ports arg must be module instance\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  if (vpi_scan(tfarg_itr) != NULL) {
    tf_error("ERROR: $list_pathout_ports requires 1 argument\n");
    vpi_free_object(tfarg_itr); /* because not scanning until null */
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_ListPorts_calltf(char *user_data) 
{
  vpiHandle module_handle, systf_h, arg_itr,
            path_itr, path_handle,
            term_itr, term_handle,
            port_itr, port_handle,
            net_handle;

  /* get module handle from first system task argument.  Assume the  */
  /* compiletf routine has already verified correct argument type.   */
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain handle to systf call\n");
    return(0);
  }
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain iterator to systf args\n");
    return(0);
  }
  module_handle = vpi_scan(arg_itr);
  vpi_free_object(arg_itr); /* free itr since did not scan until nul */

  vpi_printf("\nModule %s\n", vpi_get_str(vpiDefName, module_handle));

  path_itr = vpi_iterate(vpiModPath, module_handle);
  if (path_itr == NULL) {
    vpi_printf("   No module paths found\n");
    return(0);
  }
  while (path_handle = vpi_scan(path_itr)) {
    term_itr = vpi_iterate(vpiModPathOut, path_handle);
    if (term_itr == NULL) {
      vpi_printf("   No path output terminal found\n");
      break; /* go to next path */
    }
    while (term_handle = vpi_scan(term_itr)) {
      net_handle = vpi_handle(vpiExpr, term_handle);
      port_itr = vpi_iterate(vpiPort, net_handle);
      if (port_itr == NULL) {
        vpi_printf("   Path output does not connect to a port\n");
        break; /* go to next path output terminal */
      }
      while (port_handle = vpi_scan(port_itr)) {
        vpi_printf("  Port %s is connected to a path delay output\n",
                   vpi_get_str(vpiName, port_handle));
      }
    }
  }
  return(0);
}
/*********************************************************************/
