/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_suite_traversal.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref
 * diagnostics::unittest::Test_Suite_Traversal
 *
 * $Id: test_suite_traversal.t.cpp,v 1.13 2005/06/23 09:54:27 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/test_system/test_suite_traversal.hpp>

// used components
#include <diagnostics/unittest/test_system_exception.hpp>
#include <diagnostics/unittest/name_separators.hpp>

// test support
#include <diagnostics/util/dummy_test_suite_traversal.ts.hpp>
#include <diagnostics/util/test_suites.ts.hpp>

#include <sstream>

#define TEST_COMPONENT_NAME Test_Suite_Traversal
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

void check_visit(Test_Data & test_data,
		 Test_Suite & suite,
		 ::std::string mask,
		 Level_t const level)
{
    ::std::ostringstream stream;
    Dummy_Test_Suite_Traversal traversal(stream);    
    traversal.traverse(suite,mask,level);
    ::std::string::size_type const npos(::std::string::npos);
    ::std::string::size_type f;
    for(f=mask.find('/');f!=npos;f=mask.find('/')) mask[f]='-';
    for(f=mask.find('*');f!=npos;f=mask.find('*')) mask[f]='A';
    TEST_ASSERT(test_data.compare(::std::string(level_to_string(level))
						 + "::"
						 + mask,stream.str()));
}



void dump_simple_test_suite(Test_Data & test_data)
{
    
    Test_Suite *s1(simple_test_suite());
    check_visit(test_data,*s1,"/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/S2/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/S3/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*/C1",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*/C2",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/C1",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*/*/*",LEVEL_TEST);
    delete s1;
}

void dump_simple_unnamed_test_suite(Test_Data & test_data)
{
    
    Test_Suite *s1(simple_unnamed_test_suite());
    check_visit(test_data,*s1,"/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/S2/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/S3/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/C1",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/C2",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/C1",LEVEL_TEST);
    delete s1;
}


void dump_test_suite_all_test_case_variations(Test_Data & test_data)
{
    Test_Suite *s1(test_suite_all_test_case_variations());
    check_visit(test_data,*s1,"/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*",LEVEL_PROD);

    check_visit(test_data,*s1,"/*/S2/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/S2/*",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*/S2/*",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*/S2/*",LEVEL_PROD);

    check_visit(test_data,*s1,"/*/S3/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/S3/*",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*/S3/*",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*/S3/*",LEVEL_PROD);

    check_visit(test_data,*s1,"/*/*/name1",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*/name1",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*/*/name1",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*/*/name1",LEVEL_PROD);

    check_visit(test_data,*s1,"/*/*/name2",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*/name2",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*/*/name2",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*/*/name2",LEVEL_PROD);

    check_visit(test_data,*s1,"/*/*/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*/*",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*/*/*",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*/*/*",LEVEL_PROD);

    check_visit(test_data,*s1,"/*/name1",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/name1",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*/name1",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*/name1",LEVEL_PROD);

    check_visit(test_data,*s1,"/*/*",LEVEL_TEST);
    check_visit(test_data,*s1,"/*/*",LEVEL_AUDIT);
    check_visit(test_data,*s1,"/*/*",LEVEL_DEBUG);
    check_visit(test_data,*s1,"/*/*",LEVEL_PROD);
    delete s1;
}

void empty_suite(Test_Data & test_data)
{
    ::std::ostringstream stream;
    Dummy_Test_Suite_Traversal traversal(stream);    
    Test_Suite s1("S1");

    check_visit(test_data,s1,"/*",LEVEL_TEST);
    check_visit(test_data,s1,"/*",LEVEL_AUDIT);
    check_visit(test_data,s1,"/*",LEVEL_DEBUG);
    check_visit(test_data,s1,"/*",LEVEL_PROD);

    check_visit(test_data,s1,"/S1/*",LEVEL_TEST);
    check_visit(test_data,s1,"/S1/S2/*",LEVEL_TEST);
    check_visit(test_data,s1,"/*/*/*",LEVEL_TEST);
}


class Throwing_Test_Suite_Traversal :
    public Test_Suite_Traversal
{
public:
    Throwing_Test_Suite_Traversal() 
	: m_exception_going_through(false)
    {
    }

    bool exception_going_through() const 
    {
	return m_exception_going_through;
    }
    
protected:
    virtual void visit_hook(Test_Case const & test_case,
			    Path_t const & path,
			    Level_t const level)
    {
	throw 5;
    }

    virtual void exit_hook(Test_Suite const & test_suite,
			   ::std::string const & mask,
			   Level_t const level,
			   bool const exception_going_through) 
    {
	m_exception_going_through=exception_going_through;
    }
private:
    bool m_exception_going_through;
};


void throwing_test_suite_traversal(Test_Data & test_data)
{
    Throwing_Test_Suite_Traversal traversal;    
    Test_Suite *s1(simple_unnamed_test_suite());

    TEST_ASSERT(traversal.exception_going_through()==false);
    
    traversal.traverse(*s1,"/NOTHING",LEVEL_TEST);
    
    TEST_ASSERT(traversal.exception_going_through()==false);

    TEST_THROWING_BLOCK_ENTER;
    traversal.traverse(*s1,"/*",LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(int);
    
    TEST_ASSERT(traversal.exception_going_through()==true);

}


void incorrect_masks(Test_Data & test_data)
{
    ::std::ostringstream stream;
    Dummy_Test_Suite_Traversal traversal(stream);    
    Test_Suite suite("S1");

    TEST_THROWING_BLOCK_ENTER;
    traversal.traverse(suite,"a",LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    traversal.traverse(suite,DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR,LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    traversal.traverse(suite,"a" DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR,LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    traversal.traverse(suite,
		       DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR 
		       DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR,
		       LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    TEST_THROWING_BLOCK_ENTER;
    traversal.traverse(suite,
		       DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR 
		       "a"
		       DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR
		       DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR,
		       LEVEL_TEST);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}



TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&dump_simple_unnamed_test_suite,LEVEL_PROD);
TEST_NORMAL_CASE(&dump_simple_test_suite,LEVEL_PROD);
TEST_NORMAL_CASE(&dump_test_suite_all_test_case_variations,LEVEL_PROD);
TEST_NORMAL_CASE(&empty_suite,LEVEL_PROD);
TEST_NORMAL_CASE(&throwing_test_suite_traversal,LEVEL_PROD);
TEST_ABNORMAL_CASE(&incorrect_masks,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
