
/*
  xmms - DBMix output plugin

   Description: 
   ============
   This program is an output plugin for xmms v0.9 or greater.
   The plugin allows a data stream to be generated using xmms,
   an to be sent to the DBMix sound daemon for 
   Fourier/Additive Synthesis.  This allows a user may output multiple
   audio streams concurrently to the same audio device.
  

   Original Output Plugin code: (C) 1998-1999 Mikael Alm, Olle Hallnas,
                                Thomas Nillson and 4Front Technologies

   Modifications by Robert Michael S Dean, (c) 1999-2002

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*

DESIGN NOTES

This plugin is based off of the OSS output plugin. 

There are two modes of operation:
1) "Realtime" where data is written directly to dbmix as soon as the plugin
  recieves it
2) "Buffered" mode where data is copied into a prebuffer, and a separate
   thread reads the data from the buffer in order to output it to dbmix

If your system skips or crackles when the pitch is scaled up to 115%, you
should use Buffered mode (selectable from the configuration dialog). 
The skipping/crackling occurs because dbmix is starved for data. Buffered
mode writes much larger data chunks to dbmix at a single time.  Becuase
it is a separate thread of execution, by the time dbmix consumes the larger
data chuck, xmms has already written new data to the buffer. 

Use of the buffer is enabled in the configuration dialog, and its state
is stored in the dbmix_cfg.buffer_enable variable.  This variable is
only used in the dbmix_open function to start the buffer thread. Once the
thread is started, the dbmix_thread_started flag is set, and cleared only
when the dbmix_close function is called.  Thus, once a song is started
in Buffered mode or Realtime mode, it will continue to use that mode for
the duration of its playback.  Otherwise, it would be possible to change
playback modes during playback causing memory leaks, seg faults etc.

*/

#include <dbsoundcard.h>
#include <dbchannel.h>
#include <dbdebug.h>
#include <dbaudiolib.h>

#include <xmms/plugin.h>
#include <xmms/util.h>
#include <xmms/xmmsctrl.h>

#include "dbmixout.h"

#include <errno.h>

#if 0
#define PRINT_FUNCTION_NAMES
#endif

#define DBMIX_REALTIME

#define MIN_WRITE_AMOUNT (PIPE_BUF * 2)

static gpointer buffer;
static gboolean going = FALSE, msg_going = TRUE, prebuffer, paused = FALSE, unpause = FALSE,
	do_pause = FALSE, remove_prebuffer = FALSE;
static gint device_buffer_used, buffer_size, prebuffer_size, blk_size;
static gint rd_index = 0, wr_index = 0;
static gint output_time_offset = 0;
static guint64 written = 0, output_bytes = 0;
static gint bps, ebps;
static gint flush;
static gint fragsize, dbmix_format, format, channels, frequency, efrequency, device_buffer_size;
static gint input_bps, input_format, input_frequency, input_channels;
static gchar *device_name;
static pthread_t buffer_thread;
static gint dbmix_thread_started = 0;
pthread_t msg_thread;
static gboolean realtime = FALSE;

gint xmms_session_id = 0;

/* dbaudiolib variables */
extern int errno;
int  loop_flag;
int  debug_level;
float dbprebuffer_percentage;
int dbbuffer_size;

#ifdef CLOSE_FLAG
int close_flag = FALSE;
#endif

#ifdef DBMIX_DYNAMIC_LIBRARY
extern DBAudioLibFunctions * dbaudiofxns;
extern void* dbaudiolib_handle;
#endif

extern char * backup_name;

void dbmix_set_audio_params(void);

int dbmix_message_handler_callback(gpointer data)
{
/* 	Debug("called %d\n",msg_going); */

#ifdef DBMIX_DYNAMIC_LIBRARY

		dbaudiofxns->DBAudio_Handle_Message_Queue();
#else
		DBAudio_Handle_Message_Queue();
#endif
	
	return msg_going;
}

void dbmix_message_handler(dbfsd_msg msg)
{
	gint value;

#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_message_handler ");
#endif

	Debug("session id is %d\n",xmms_session_id);
	
	switch (msg.msg_type)
	{
		case DBMSG_PAUSE:
			Debug("Got pause message");
			xmms_remote_pause(xmms_session_id);
			break;
		case DBMSG_UNPAUSE:
			Debug("Got unpause message");
			if (!xmms_remote_is_playing(xmms_session_id) 
				|| xmms_remote_is_paused(xmms_session_id))
			{			
				xmms_remote_play(xmms_session_id);
			}
			break;
		case DBMSG_PLAY:
			if (!xmms_remote_is_playing(xmms_session_id) 
				|| xmms_remote_is_paused(xmms_session_id))
			{
				xmms_remote_play(xmms_session_id);
			}
			break;
			
		case DBMSG_MUTE:
			Debug("Setting MUTE.");
			DBAudio_Set_Mute(1);
			break;
		case DBMSG_UNMUTE:
			Debug("Clearing MUTE");
			DBAudio_Set_Mute(0);
			break;
		case DBMSG_STOP:
			xmms_remote_stop(xmms_session_id);
			break;
		case DBMSG_EJECT:
			xmms_remote_eject(xmms_session_id);
			break;
		case DBMSG_REWIND:
			value = xmms_remote_get_output_time(xmms_session_id);
			value -= msg.data * 1000;
			if (value < 0) value = 0;
			xmms_remote_jump_to_time(xmms_session_id,value);
			break;
		case DBMSG_FFORWARD:
			value = xmms_remote_get_output_time(xmms_session_id);
			value += msg.data * 1000;
			if (value < 0) value = 0;
			xmms_remote_jump_to_time(xmms_session_id,value);
			break;
		case DBMSG_NEXT:
			xmms_remote_playlist_next(xmms_session_id);
			break;
		case DBMSG_PREV:
			xmms_remote_playlist_prev(xmms_session_id);
			break;
		default:
			Error("dbmix_message_handler: unknown message %d\n ",msg);
	}

	return;
}



static void dbmix_calc_device_buffer_used(void)
{
	device_buffer_used = 0;

	/* the problem with setting buffer_used to 1 is that it's value is used
       to change rd_index */
/* 	if(paused) */
/* 		device_buffer_used = 0; */
/* 	else */
/* 		device_buffer_used = 1; */  
}


static void dbmix_setup_format(AFormat fmt,gint rate, gint nch)
{
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_setup_format ");
#endif

	format = fmt;
	frequency = rate;
	channels = nch;

	switch (fmt)
	{
	case FMT_U8:
		dbmix_format = AFMT_U8;
		break;
	case FMT_S8:
		dbmix_format = AFMT_S8;
		break;
	case FMT_U16_LE:
		dbmix_format = AFMT_U16_LE;
		break;
	case FMT_U16_BE:
		dbmix_format = AFMT_U16_BE;
		break;
	case FMT_U16_NE:
#ifdef AFMT_U16_NE
		dbmix_format = AFMT_U16_NE;
#else
#ifdef WORDS_BIGENDIAN
		dbmix_format = AFMT_U16_BE;
#else
		dbmix_format = AFMT_U16_LE;
#endif
#endif
		break;
	case FMT_S16_LE:
		dbmix_format = AFMT_S16_LE;
		break;
	case FMT_S16_BE:
		dbmix_format = AFMT_S16_BE;
		break;
	case FMT_S16_NE:
#ifdef AFMT_S16_NE
		dbmix_format = AFMT_S16_NE;
#else
#ifdef WORDS_BIGENDIAN
		dbmix_format = AFMT_S16_BE;
#else
		dbmix_format = AFMT_S16_LE;
#endif
#endif
		break;
	default: Debug("UNKNOWN AUDIO FORMAT"); break;
	}

	bps = rate * nch;
	if (dbmix_format == AFMT_U16_BE || dbmix_format == AFMT_U16_LE || dbmix_format == AFMT_S16_BE || dbmix_format == AFMT_S16_LE)
		bps *= 2;
	fragsize = 0;
	while ((1L << fragsize) < bps / 25)
		fragsize++;
	fragsize--;

	device_buffer_size = DB_BUFSIZE_CHAR;
}
        

gint dbmix_get_written_time(void)
{
	gint writtentime;

	if (!going)
	{
		return 0;
	}

	writtentime = (gint) ((written * 1000) / input_bps);
	
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_get_written_time %d",writtentime);
#endif

	return writtentime;
}

gint dbmix_get_output_time(void)
{
	guint64 bytes;
	guint64 tempoutput;

	if (!going)
	{
		return 0;
	}

/* 	bytes = output_bytes < device_buffer_used ? 0 : output_bytes - device_buffer_used; */

	bytes = output_bytes;

	tempoutput = output_time_offset + (gint) ((bytes  * 1000) / ebps);

#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_get_output_time time: %d",tempoutput);
#endif

	return tempoutput;
}

gint dbmix_used(void)
{
	gint value;

	if (dbmix_thread_started)
	{
		if (wr_index >= rd_index)
			value = wr_index - rd_index;
		else
			value =  buffer_size - (rd_index - wr_index);
	}
	else
	{
		value = 0;
	}


#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_used used: %d",value);
#endif

	return value;
}

gint dbmix_playing(void)
{
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_playing ");
#endif

	if(!going)
	{
		return 0;
	}

	if(!dbmix_used()) return FALSE;

	return TRUE;
}


gint dbmix_free(void)
{
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_free ");
#endif

	/* if we are using a local buffer, return amount of free space in that buffer */
	if (dbmix_thread_started)
	{
		if (remove_prebuffer && prebuffer)
		{
			prebuffer = FALSE;
			remove_prebuffer = FALSE;
		}
		if (prebuffer)
			remove_prebuffer = TRUE;
		
		if (rd_index > wr_index)
			return (rd_index - wr_index) - device_buffer_size - 1;
		return (buffer_size - (wr_index - rd_index)) - device_buffer_size - 1;
	}
	
	/* not using a buffer, so return 0 on pause, and big number otherwise */
	if (paused)
	{
		return 0;
	}
	
	return 1000000;
}


static int dbmix_write_audio(gpointer data,gint length)
{
	AFormat new_format;
	gint new_frequency,new_channels;
	EffectPlugin *ep;
    gint count = 0;
    gint original_length;

#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_write_audio ");
#endif

	original_length = length;

	new_format = input_format;
	new_frequency = input_frequency;
	new_channels = input_channels;
        
	ep = get_current_effect_plugin();
	if(effects_enabled() && ep && ep->query_format)
	{
		ep->query_format(&new_format,&new_frequency,&new_channels);
			
		if(new_format != format || new_frequency != frequency 
		   || new_channels != channels)
		{
			output_time_offset += (gint) ((output_bytes * 1000) / ebps);
			output_bytes = 0;
			dbmix_setup_format(new_format, new_frequency, new_channels);
			frequency = new_frequency;
			channels = new_channels;

			dbmix_set_audio_params();
		}

		if(effects_enabled() && ep && ep->mod_samples)
		{
			length = ep->mod_samples(&data,length, input_format, 
									 input_frequency, input_channels);
		}
	}

#ifdef DBMIX_DYNAMIC_LIBRARY
	if((count = dbaudiofxns->DBAudio_Write(data,length)) == FAILURE)
#else
	if((count = DBAudio_Write(data,length)) == FAILURE)		
#endif
	{
		switch(errno)
		{
		case ERROR_TOO_MUCH_DATA:
			blk_size -= 1024;
			if(blk_size < PIPE_BUF) {blk_size = MIN_WRITE_AMOUNT;}
			break;
		case ERROR_TOO_LITTLE_DATA:
			if(length < blk_size) break;

			blk_size+= 1024;
			if(blk_size > MIN_WRITE_AMOUNT) {blk_size = MIN_WRITE_AMOUNT;}
			break;
		default:
			perror("DBMix output plugin, failure on write: ");
			break;
		}
		/* error, so no data was written */
		count = 0;
	}
	else
	{
		output_bytes += count;
	}

	return count;
}


#if 0
static void swap_words(guint16 *buffer, gint length)
{
	guint16 *ptr = buffer;
	gint i;
	for(i = 0; i < length; i++, ptr++)
		*ptr = ((*ptr & 0x00FF) << 8) | (*ptr >> 8);
}
#endif


/* function called by xmms to output data */
void dbmix_write(gpointer ptr, gint length)
{
	gint cnt, off = 0;
	sampler_state sampstate;

#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_write  length: %d",length);
#endif

	/* if we are using a prebuffer, add the data to that buffer */
	if (dbmix_thread_started)
	{
		remove_prebuffer = FALSE;
		
		written += length;

		while (length > 0)
		{	
			/* only add data to the buffer if we are not playing a sample */

#ifdef DBMIX_DYNAMIC_LIBRARY
		dbaudiofxns->sampstate = DBAudio_Sampler_Get_State();
#else
		sampstate = DBAudio_Sampler_Get_State();
#endif

			if ((sampstate != SAMPLER_PLAY_SINGLE) && (sampstate != SAMPLER_PLAY_LOOP))
			{
				cnt = MIN(length, buffer_size - wr_index);
				memcpy(buffer + wr_index, ptr + off, cnt);
				wr_index = (wr_index + cnt) % buffer_size;
				length -= cnt;
				off += cnt;
			}
		}
	}
	else
	{
		/* paused - output no data, this function should never be called if
           xmms is paused */
		if (paused)
		{
			return;
		}		
		
		cnt = output_bytes;
		
		/* write data until it is actually written - i.e. dbmix may be paused internally,
           do not return until dbmix accepts the data */
		while (cnt == output_bytes)
		{
			dbmix_write_audio(ptr,length);
		}
		
		written += length;
	}
}


void dbmix_close(void)
{
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_close ");
#endif

	if (!going)
		return;

	going = 0;

	if (dbmix_thread_started)
	{
		pthread_join(buffer_thread, NULL);
		dbmix_thread_started = 0;
	}

	/* enable timeout for message handler to handle dbmix messages during downtime */
	msg_going = TRUE;
	gtk_timeout_add(500,dbmix_message_handler_callback,NULL);

	if(dbmix_cfg.close_flag)
	{
#ifdef DBMIX_DYNAMIC_LIBRARY
		dbaudiofxns->DBAudio_Close();
#else
		DBAudio_Close();
#endif
	}

	g_free(device_name);
	wr_index = 0;
	rd_index = 0;
}

void dbmix_flush(gint time)
{
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_flush time is %d",time);
#endif
	/* this is the code that resets the buffer when you move
	   the position bar in xmms */

	/* if we are using a prebuffer, wait for the buffer to empty */
	if (dbmix_thread_started)
	{
		flush = time;
		while (flush != -1)
			xmms_usleep(10000);
	}
	else
	{
		/* not using prebuffer, therefore all data is considered written, update timers */
		output_time_offset = time;
		written = ((guint64) time * input_bps) / 1000;
		output_bytes = 0;
	}
}


void dbmix_pause(short p)
{
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_pause ");
#endif

#ifdef DBMIX_DYNAMIC_LIBRARY
	dbaudiofxns->DBAudio_Pause(p);
#else
	DBAudio_Pause(p);
#endif
	paused = p;
}


/*
  dbmix_loop is the function executed by the buffer read thread to output data
  to dbmix.  
*/
void *dbmix_loop(void *arg)
{
	gint length, cnt;

#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_loop ");
#endif

	while (going)
	{	
		
#ifdef DBMIX_DYNAMIC_LIBRARY
		dbaudiofxns->DBAudio_Handle_Message_Queue();
#else
		DBAudio_Handle_Message_Queue();
#endif

		if (dbmix_used() > prebuffer_size)
		{
			prebuffer = FALSE;
		}

		if (dbmix_used() > 0 && !paused && !prebuffer)
		{
			length = MIN(blk_size, dbmix_used());

			while (length > 0)
			{
				cnt = MIN(length,buffer_size-rd_index);

				/* write data and update rd_index by amount of data written */
				cnt = dbmix_write_audio(buffer + rd_index, cnt);
				rd_index = (rd_index+cnt) % buffer_size;
				length -= cnt;                            
			}
		}
		else
		{
			xmms_usleep(10000);
		}
		
		dbmix_calc_device_buffer_used();

		if (flush != -1)
		{
			dbmix_set_audio_params();
			output_time_offset = flush;
			written = (guint64)(flush / 10) * (guint64)(input_bps / 100);
			rd_index = wr_index = output_bytes = 0;
			flush = -1;
			prebuffer = TRUE;
		}
	}

#ifdef CLOSE_FLAG
	if(dbmix_cfg.close_flag)
	{
#ifdef DBMIX_DYNAMIC_LIBRARY
		dbaudiofxns->DBAudio_Close();
#else
		DBAudio_Close();
#endif
	}
#endif

	g_free(buffer);
	buffer = NULL;

	pthread_exit(NULL);
}

void dbmix_set_audio_params(void)
{
	gint stereo;
        
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_set_audio_params ");
#endif

	stereo = channels - 1;
	efrequency = frequency;

	blk_size = PIPE_BUF + 2048;

#ifdef DBMIX_DYNAMIC_LIBRARY

	if(dbaudiofxns->DBAudio_Set_Format(dbmix_format) == FAILURE)
	{
		perror("dbmix_set_audio_params: ");
	}

	if(dbaudiofxns->DBAudio_Set_Channels(channels) == FAILURE)
	{
		perror("dbmix_set_audio_params: ");
	}	

	if(dbaudiofxns->DBAudio_Set_Rate(efrequency) == FAILURE)
	{
		perror("dbmix_set_audio_params: ");
	}
#else
	if(DBAudio_Set_Format(dbmix_format) == FAILURE)
	{
		perror("dbmix_set_audio_params: ");
	}

	if(DBAudio_Set_Channels(channels) == FAILURE)
	{
		perror("dbmix_set_audio_params: ");
	}	

	if(DBAudio_Set_Rate(efrequency) == FAILURE)
	{
		perror("dbmix_set_audio_params: ");
	}	
#endif


	ebps = efrequency * channels;
	
	if (dbmix_format == AFMT_U16_BE || dbmix_format == AFMT_U16_LE ||
		dbmix_format == AFMT_S16_BE || dbmix_format == AFMT_S16_LE)
		ebps *= 2;
}

gint dbmix_open(AFormat fmt, gint rate, gint nch)
{
#ifdef PRINT_FUNCTION_NAMES
	Debug("dbmix_open ");
#endif
	
#ifdef DBMIX_DEBUG
	debug_level = 1;
#else 
	debug_level = 0;
#endif

	dbmix_setup_format(fmt,rate,nch);
        
	input_format = format;
	input_channels = channels;
	input_frequency = frequency;
	input_bps = bps;

	realtime = xmms_check_realtime_priority();
      
#if 0

	buffer_size = (dbmix_cfg.buffer_size * input_bps) / 1000;
	if (buffer_size < 8192)
		buffer_size = 8192;
	prebuffer_size = (buffer_size * dbmix_cfg.prebuffer) / 100;
	if (buffer_size - prebuffer_size < 4096)
		prebuffer_size = buffer_size - 4096;

	buffer_size += device_buffer_size;
#endif

	buffer = NULL;

	/* if we are using a buffer, determine the buffer size and allocate the buffer */
	if (dbmix_cfg.buffer_enable)
	{
		int tempi;
		
		buffer_size = (dbmix_cfg.buffer_size * input_bps) / 1000;
		tempi = buffer_size % MIN_WRITE_AMOUNT;
		buffer_size -= tempi;

		if (buffer_size < MIN_WRITE_AMOUNT)
		{
			buffer_size = MIN_WRITE_AMOUNT * 20;
		}

		buffer = g_malloc0(buffer_size);

	}

	flush = -1;
	prebuffer = 1;
	wr_index = rd_index = output_time_offset = written = output_bytes = 0;
	paused = FALSE;
	do_pause = FALSE;
	unpause = FALSE;
	remove_prebuffer = FALSE;

	paused = 0;

#ifdef DBMIX_DYNAMIC_LIBRARY

	if(dbaudiofxns->DBAudio_Ready() == FAILURE) 
	{
		strcpy(dbmix_cfg.channel_name,"Xmms");

		if(dbaudiofxns->DBAudio_Init(dbmix_cfg.channel_name,
									 dbmix_format,frequency,nch,
									 PIPE_CHANNEL,0) == FAILURE)
		{
			return 0;
		}

		dbaudiofxns->DBAudio_Set_Message_Handler(dbmix_message_handler,DBMIX_MESSAGES);
	}
	else
	{
		dbmix_set_audio_params();
	}

	/* this function is called at the start of a song, so clear pause */
	dbaudiofxns->DBAudio_Pause(0);
#else

	if(DBAudio_Ready() == FAILURE) 
	{
		strcpy(dbmix_cfg.channel_name,"Xmms");

		if(DBAudio_Init(dbmix_cfg.channel_name,
									 dbmix_format,frequency,nch,
									 PIPE_CHANNEL,0) == FAILURE)
		{
			return 0;
		}

		DBAudio_Set_Message_Handler(dbmix_message_handler,DBMIX_MESSAGES);
	}
	else
	{
		dbmix_set_audio_params();
	}

	/* this function is called at the start of a song, so clear pause */
	DBAudio_Pause(0);
#endif

	going = 1;
	msg_going = FALSE;
	
	if (dbmix_cfg.buffer_enable)
	{
		Debug("dbmix_open: buffer enabled, launching buffer thread.");
		pthread_create(&buffer_thread, NULL, dbmix_loop, NULL);
		dbmix_thread_started = 1;
	}
	else
	{
		Debug("dbmix_open: buffer not enabled.");
		dbmix_thread_started = 0;
	}

	return 1;
}
