/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#include "mod_ftp.h"
#include "ftp_internal.h"

/* Placeholder of "unknown" username (not valid for ftp_cmd_pass)
 */
const char ftp_unknown_username[] = "unknown";

/*
 * initialize_ftp_connection: Used to initialize the ftp_connection structure
 *                            when accepting an incoming FTP connection.
 *
 * Arguments: c - The connection associated with the session
 *            fc - The ftp connection to initialize
 *
 * Returns: nothing
 */
static void initialize_ftp_connection(conn_rec *c, ftp_connection *fc)
{
    /*
     * The ftp_connection structure is calloc'ed so only initalize the
     * members that we need to.
     */
    fc->connection = c;
    fc->user = ftp_unknown_username;
    fc->auth = FTP_AUTH_NONE;
    fc->prot = FTP_PROT_CLEAR;
    fc->type = TYPE_A;
    fc->passive_created = -1;
    fc->orig_server = c->base_server;

    fc->cwd = apr_palloc(c->pool, APR_PATH_MAX + 1);
    fc->cwd[0] = '/';
    fc->cwd[1] = '\0';

    fc->rename_from = apr_palloc(c->pool, APR_PATH_MAX + 1);
    fc->rename_from[0] = '\0';

    apr_pool_create(&fc->login_pool, c->pool);
    apr_pool_tag(fc->login_pool, "login");
    apr_pool_create(&fc->data_pool, c->pool);
    apr_pool_tag(fc->data_pool, "data");

    fc->cntlsock = ap_get_module_config(c->conn_config, &core_module);

    ftp_set_module_config(c->conn_config, fc);
}

/* ftp_ssl_init: Fakes a read on the SSL filters to force initialization.
 *
 * Arguments: cdata - The data connection
 *
 * Return: apr_status_t
 */
apr_status_t ftp_ssl_init(conn_rec *cdata)
{
    ftp_connection *fc = ftp_get_module_config(cdata->conn_config);
    apr_bucket_brigade *bb;
    apr_status_t rv;
    apr_socket_t *client_socket;

    /*
     * This is handled in the NET_TIME filter, which unfortunately ignores
     * the timeout for the purpose of AP_MODE_INIT. Fix a timeout so the core
     * read filter will behave.
     */
    client_socket = ap_get_module_config(cdata->conn_config, &core_module);
    rv = apr_socket_timeout_set(client_socket, fc->orig_server->timeout);

    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_DEBUG, rv, fc->orig_server,
                     "Failed to set socket timeout");
    }

    bb = apr_brigade_create(cdata->pool, cdata->bucket_alloc);
    rv = ap_get_brigade(cdata->input_filters, bb, AP_MODE_INIT,
                        APR_BLOCK_READ, 0);
    apr_brigade_destroy(bb);

    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_DEBUG, rv, fc->orig_server,
                     "Failed to initialize the ftp ssl data stream");
    }

    return rv;
}

/*
 * ftp_send_welcome: Send the server welcome string to the client
 *
 * Arguments: fc - The ftp connection to send the welcome message to
 *
 * Returns: apr_status_t
 */
static apr_status_t ftp_send_welcome(ftp_connection *fc)
{
    conn_rec *c = fc->connection;
    ftp_server_config *fsc;
    apr_status_t rv;

    fsc = ftp_get_module_config(fc->orig_server->module_config);

    /* Check if a login message has been configured */
    if (fsc->banner_message) {
        if (fsc->banner_message_isfile) {
            rv = ftp_show_file(c->output_filters, fc->data_pool,
                               FTP_REPLY_SERVICE_READY, fc,
                               fsc->banner_message);
            if (rv != APR_SUCCESS) {
                return rv;
            }
        }
        else {
            char outbuf[BUFSIZ];

            ftp_message_generate(fc, fsc->banner_message,
                                 outbuf, sizeof(outbuf));

            rv = ftp_reply(fc, c->output_filters, fc->data_pool,
                           FTP_REPLY_SERVICE_READY, 1, outbuf);

            if (rv != APR_SUCCESS) {
                return rv;
            }
        }
    }

    rv = ftp_reply(fc, c->output_filters, fc->data_pool,
                   FTP_REPLY_SERVICE_READY, 0,
                   "%s FTP Server (%s) ready.",
                   fc->orig_server->server_hostname,
#if AP_MODULE_MAGIC_AT_LEAST(20060905,0)
                   ap_get_server_banner());
#else
                   ap_get_server_version());
#endif
    return rv;
}

/*
 * ftp_process_connection: Connection handler for the FTP module.
 *
 * Arguments: c - The incoming connection
 *
 * Returns: Returns OK when the connection is to be closed.
 */
int ftp_process_connection(conn_rec *c)
{
    ap_filter_t *f;
    request_rec *r;
    ftp_connection *fc;
    apr_status_t rv;
    int idle_timeout_set = 0;
    extern ap_filter_rec_t *ftp_ssl_input_filter_handle;
    extern ap_filter_rec_t *ftp_ssl_output_filter_handle;
    ftp_server_config *fsc;

    fsc = ftp_get_module_config(c->base_server->module_config);

    if (!fsc->enabled) {
        return DECLINED;
    }

    /*
     * Allocate for FTP connection structure, and initialize
     */
    fc = apr_pcalloc(c->pool, sizeof(*fc));
    initialize_ftp_connection(c, fc);

    /* Check for implicit security on the control connection */
    if (!fsc->implicit_ssl) {
        for (f = c->output_filters; f; f = f->next) {
            if (strcasecmp(f->frec->name, FTP_SSL_FILTER) == 0) {
                fc->ssl_output_ctx = f->ctx;
                ap_remove_output_filter(f);
            }
        }
        for (f = c->input_filters; f; f = f->next) {
            if (strcasecmp(f->frec->name, FTP_SSL_FILTER) == 0) {
                fc->ssl_input_ctx = f->ctx;
                ap_remove_input_filter(f);
            }
        }
    }
    else {
        /* Initialize control connection */
        if ((rv = ftp_ssl_init(c)) != APR_SUCCESS) {
            /*
             * Error initializing the connection, most likely from a client
             * attempting to connect w/o SSL.  Just drop the connection
             */
            ap_log_error(APLOG_MARK, APLOG_INFO, rv, fc->orig_server,
                         "Error initializing SSL connection.  Client "
                         "connecting without SSL?");
            return OK;
        }

        /*
         * From draft-murray-auth-ftp-ssl-06.txt: For implicit SSL the data
         * connection should be implicitly protected (i.e. the PBSZ 0, PROT P
         * command sequence is not required but the client and server will
         * protect the data channel as if it had).
         * 
         * Support for Implicit SSL was declared deprecated as of
         * draft-murray-auth-ftp-ssl-07.txt, and is not documented whatsoever
         * within RFC4217.
         */
        fc->auth = FTP_AUTH_SSL;
        fc->prot = FTP_PROT_PRIVATE;
        fc->is_secure = 1;
    }

    rv = ftp_send_welcome(fc);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_INFO, rv, fc->orig_server,
                     "Error sending server welcome string");
        return OK;
    }

    /*
     * Begin processing requests
     */

    /* Set initial login timeout value */
    rv = apr_socket_timeout_set(fc->cntlsock,
                                fsc->timeout_login * APR_USEC_PER_SEC);
    if (rv != APR_SUCCESS) {
        ap_log_error(APLOG_MARK, APLOG_DEBUG, rv, fc->orig_server,
                     "Couldn't set SO_TIMEOUT socket option");
    }

    ap_update_child_status(c->sbh,
                           SERVER_BUSY_READ, NULL);

    while ((r = ftp_read_request(fc)) != NULL) {

        /*
         * Set socket timeout values.  A user can use the SITE command to
         * change the idle timeout during an FTP session, but since we do not
         * support the SITE command, it doesn't make sense to be setting the
         * socket timeout on every single request.
         */
        if (!idle_timeout_set) {
            rv = apr_socket_timeout_set(fc->cntlsock,
                                        fsc->timeout_idle * APR_USEC_PER_SEC);
            if (rv != APR_SUCCESS) {
                ap_log_error(APLOG_MARK, APLOG_DEBUG, rv, r->server,
                             "Couldn't set socket timeout");
            }
            idle_timeout_set = 1;
        }

        ap_update_child_status(c->sbh,
                               SERVER_BUSY_WRITE, NULL);

        ftp_process_request(r);

        if (ap_extended_status) {
            ap_increment_counts(c->sbh, r);
        }

        apr_pool_destroy(r->pool);

        /* Check if the connection should be closed  */
        if (fc->close_connection) {
            break;
        }

        /* Check if the client has requested a secure connection */
        if ((fc->auth == FTP_AUTH_TLS || fc->auth == FTP_AUTH_SSL) &&
            fc->is_secure == 0) {
            ap_add_output_filter_handle(ftp_ssl_output_filter_handle,
                                        fc->ssl_output_ctx, NULL, c);
            ap_add_input_filter_handle(ftp_ssl_input_filter_handle,
                                       fc->ssl_input_ctx, NULL, c);
            fc->is_secure = 1;
        }
    }
    if (fc->logged_in) {
        ftp_limitlogin_loggedout(c);
    }
    return OK;
}
